'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _lodash = require('lodash');

var _check_es_version = require('../../elasticsearch/check_es_version');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Kibi Core - Migration 25
 *
 * Looks for zoom, enableAllDashboardsCounts, shieldAuthorizationWarning in the config object
 * if found remove these options.
 */

class Migration25 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;
    this._server = configuration.server;
    this._type = 'config';
    this._query = {
      query: {
        bool: {
          filter: [{
            term: {
              _id: 'siren'
            }
          }]
        }
      }
    };

    this.keyMap = ['siren:zoom', 'siren:enableAllDashboardsCounts', 'siren:shieldAuthorizationWarning'];
  }

  static get description() {
    return 'Remove deprecated advanced settings: siren:zoom, siren:enableAllDashboardsCounts, ' + 'siren:shieldAuthorizationWarning from config object';
  }

  _getUpgradeableKeys(object) {
    if (!object) {
      return [];
    } else {
      const keys = Object.keys(object);
      const upgradeableKeys = [];

      (0, _lodash.each)(keys, key => {
        if ((0, _lodash.contains)(this.keyMap, key)) {
          upgradeableKeys.push(key);
        }
      });

      return upgradeableKeys;
    }
  }

  _removeDeprecatedKeys(object) {
    if (!object) {
      return;
    } else {
      const keys = Object.keys(object);
      (0, _lodash.each)(keys, key => {
        if ((0, _lodash.contains)(this.keyMap, key)) {
          delete object[key];
        }
      });

      return object;
    }
  }

  _isUpgradeable(object) {
    return this._getUpgradeableKeys(object).length;
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    return objects.reduce((count, obj) => {
      return this._isUpgradeable(obj._source);
    }, 0);
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const objects = await this.scrollSearch(this._index, this._type, this._query);
    if (objects.length === 0) {
      return 0;
    }
    let body = '';
    let count = 0;

    for (const obj of objects) {
      if (this._isUpgradeable(obj._source) > 0) {
        let newSource = Object.assign({}, obj._source);

        const upgradeableKeys = this._getUpgradeableKeys(obj._source);
        newSource = this._removeDeprecatedKeys(newSource);

        count += upgradeableKeys.length;

        body += JSON.stringify({
          delete: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({
          index: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify(newSource) + '\n';
      }
    }
    if (count > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }
    return count;
  }
}
exports.default = Migration25;
module.exports = exports['default'];
