'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _lodash = require('lodash');

var _lodash2 = _interopRequireDefault(_lodash);

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _gremlin_server = require('../../../../../server/gremlin_server/gremlin_server');

var _gremlin_server2 = _interopRequireDefault(_gremlin_server);

var _gremlin_server_errors = require('../../../../../server/gremlin_server/gremlin_server_errors');

var _n3_model_to_json = require('../../ontology/n3_model_to_json');

var _n3_model_to_json2 = _interopRequireDefault(_n3_model_to_json);

var _server_ontology_client = require('../../ontology/server_ontology_client');

var _server_ontology_client2 = _interopRequireDefault(_server_ontology_client);

var _uuid = require('uuid');

var _uuid2 = _interopRequireDefault(_uuid);

var _check_es_version = require('../../elasticsearch/check_es_version');

var _entity_type = require('../../ontology/entity_type');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 27.
 *
 * Looks for:
 * index-pattern and searches objects
 *
 * Then:
 *
 * - for each index-pattern object define a savedSeaerch if such one does not exist
 * - replace all INDEX_PATTERN entities in ontology with coresponding SAVED_SEARCH entities
     keep the map of matching ids
 * - creates SAVED_SEARCH entities for all existing saved search objects
 * - update all relations in ontology - swap INDEX-PATTERN id with SAVED_SEARCH one
 */
class Migration27 extends _migration2.default {

  constructor(configuration) {
    super(configuration);

    this._logger = configuration.logger;
    this._client = configuration.client;
    this._server = configuration.server;
    this._index = configuration.config.get('kibana.index');

    this._indexPatternType = 'index-pattern';
    this._searchType = 'search';

    this._ontologyType = 'ontology-model';
    this._ontologyId = 'default-ontology';
    this._ontologyQuery = {
      query: {
        bool: {
          filter: [{
            term: {
              _id: 'default-ontology'
            }
          }]
        }
      }
    };

    this._gremlinStartTimeout = configuration.gremlinStartTimeout;
  }

  static get description() {
    return 'Convert the ontology model to use SAVED_SEARCH entities instead of INDEX_PATTERN ones.';
  }

  _findSearchesBasedOnIndexPattern(indexPatternId, savedSearches, excludeSavedSearchId) {
    const found = [];
    for (let i = 0; i < savedSearches.length; i++) {
      const hit = savedSearches[i];
      if (excludeSavedSearchId && hit._id === excludeSavedSearchId) {
        continue;
      }
      if (hit._source.kibanaSavedObjectMeta && hit._source.kibanaSavedObjectMeta.searchSourceJSON) {
        const searchSourceJSON = hit._source.kibanaSavedObjectMeta.searchSourceJSON;
        try {
          const searchSource = JSON.parse(searchSourceJSON);
          if (searchSource.index === indexPatternId) {
            found.push(hit);
          }
        } catch (e) {
          // should not happen if it happens the object is broken anyway
          this._logger.error('Detected broken kibanaSavedObjectMeta.searchSourceJSON object in saved search: ' + hit._id);
        }
      }
    };
    return found;
  }

  _findSearchWithNoFiltersNoQuery(savedSearches) {
    const found = [];
    for (let i = 0; i < savedSearches.length; i++) {
      const hit = savedSearches[i];
      // here no need for extra checks as they were done already
      const searchSource = JSON.parse(hit._source.kibanaSavedObjectMeta.searchSourceJSON);
      const filters = searchSource.filter;
      const query = searchSource.query;

      const isMatchAllQuery = query && query.match_all && _lodash2.default.isEqual(query.match_all, {});
      const isAsteriskQuery = query && query.query_string && _lodash2.default.isEqual(query.query_string.query, '*');

      if ((!filters || filters.length === 0) && (isMatchAllQuery || isAsteriskQuery)) {
        found.push(hit);
      }
    };
    return found;
  }

  _findRootSavedSearch(indexPatternId, savedSearches) {
    const basedOnSameIndexPattern = this._findSearchesBasedOnIndexPattern(indexPatternId, savedSearches);
    if (basedOnSameIndexPattern.length === 1) {
      return basedOnSameIndexPattern[0];
    }

    // there is more than 1
    // lets see if we can find one at least one with no filters nor queries (or default query or match_all query)
    const basedOnSameIndexPatternNoQueriesOrFilters = this._findSearchWithNoFiltersNoQuery(basedOnSameIndexPattern);
    if (basedOnSameIndexPatternNoQueriesOrFilters.length >= 1) {
      return basedOnSameIndexPatternNoQueriesOrFilters[0];
    }
    // there is none - we will create a new one
    return null;
  }

  _createSavedSearch(rootSavedSearchId, indexPatternId) {
    const source = {
      title: indexPatternId,
      columns: [],
      description: '',
      sort: ['_score', 'desc'],
      kibanaSavedObjectMeta: {
        searchSourceJSON: JSON.stringify({
          index: indexPatternId,
          filter: [],
          query: { query_string: { analyze_wildcard: true, query: '*' } },
          highlight: {
            pre_tags: ['@kibana-highlighted-field@'],
            post_tags: ['@/kibana-highlighted-field@'],
            fields: { '*': {} },
            require_field_match: false,
            fragment_size: 2147483647
          }
        })
      }
    };

    return this._client.index({
      index: this._index,
      type: 'search',
      id: rootSavedSearchId,
      body: source,
      refresh: true
    });
  }

  _updateRelations(relations, indexPatternToSavedSearchMap) {
    let updatedRelationsCount = 0;
    let updated;
    _lodash2.default.each(relations, rel => {
      updated = false;
      if (indexPatternToSavedSearchMap[rel.domain.id] && rel.domain.type === 'INDEX_PATTERN') {
        rel.domain.type = 'SAVED_SEARCH';
        rel.domain.label = rel.domain.id;
        rel.domain.id = indexPatternToSavedSearchMap[rel.domain.id];
        updated = true;
      }
      if (indexPatternToSavedSearchMap[rel.range.id] && rel.range.type === 'INDEX_PATTERN') {
        rel.range.type = 'SAVED_SEARCH';
        rel.range.label = rel.range.id;
        rel.range.id = indexPatternToSavedSearchMap[rel.range.id];
        updated = true;
      }
      if (updated) {
        updatedRelationsCount++;
      }
    });
    return updatedRelationsCount;
  }

  // model is an array of rdf triples
  // where each triple is an object with following properties
  // s - subject
  // p - predicate
  // o - object
  async _modelContainsAnyIndexPatternEntity(model) {
    const triples = await (0, _n3_model_to_json2.default)(model);
    for (let i = 0; i < triples.length; i++) {
      const triple = triples[i];
      if (triple.p === 'http://siren.io/model#type' && triple.o === 'INDEX_PATTERN') {
        return true;
      }
    }
    return false;
  }

  async _graphContainsAnyIndexPatternEntity(graph) {
    if (!graph) {
      return false;
    }
    for (let i = 0; i < graph.items.length; i++) {
      if (graph.items[i].type === 'node' && graph.items[i].d.entityType === 'INDEX_PATTERN') {
        return true;
      }
    }
    return false;
  }

  async count() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const ontologies = await this.scrollSearch(this._index, this._ontologyType, this._ontologyQuery);
    if (ontologies.length === 1 && ontologies[0]._source.model) {
      const containsIndexPatternEntities = await this._modelContainsAnyIndexPatternEntity(ontologies[0]._source.model);
      const relationalGraphContainsIndexPatternEntities = await this._graphContainsAnyIndexPatternEntity(ontologies[0]._source['relational-graph']);
      if (containsIndexPatternEntities === true || relationalGraphContainsIndexPatternEntities === true) {
        return 1;
      }
    }
    return 0;
  }

  // extra method easy to stub during the tests
  _getUUID() {
    return _uuid2.default.v1();
  }

  _createSavedSearchEntityFromSavedSearchHitAndIndexPatternEntity(hit, parentId, baseEntity) {
    const entity = _lodash2.default.cloneDeep(baseEntity);

    entity.id = hit._id;
    entity.type = 'SAVED_SEARCH';
    entity.label = hit._source.title;
    entity.shortDescription = baseEntity.shortDescription;
    entity.parentId = parentId;
    entity.indexPattern = baseEntity.id;

    return entity;
  }

  _createRootSavedSearchEntityFromIndexPatternEntity(savedSearchId, baseEntity) {
    const entity = _lodash2.default.cloneDeep(baseEntity);
    entity.id = savedSearchId;
    entity.type = 'SAVED_SEARCH';
    entity.label = baseEntity.id;
    entity.shortDescription = baseEntity.shortDescription;
    entity.indexPattern = baseEntity.id;

    return entity;
  }

  async upgrade() {
    const esVersion = await (0, _check_es_version.checkESVersion)(this._server);
    if (esVersion.major >= 6) {
      return 0;
    }
    const gremlin = new _gremlin_server2.default(this._server, this._gremlinStartTimeout);
    const serverOntologyClient = new _server_ontology_client2.default(this._server, this._client, this._index, this._ontologyType, this._ontologyId);

    let count = await this.count();
    if (count === 0) {
      return 0;
    }

    count = 0;

    try {
      await gremlin.start('Migration 27');

      // NOTE:
      // we have to grab relations first
      // as after deleting INDEX_PATTERN triples gremlin will not return them
      // as they will be invalid and sparql queries
      const relations = await serverOntologyClient.getRelations();
      const entities = await serverOntologyClient.getEntities();
      let res;

      // get all index-patterns
      const indexPatterns = await this.scrollSearch(this._index, this._indexPatternType);
      if (indexPatterns.length === 0) {
        this._logger.error('No index pattern objects while some in ontology');
      }

      const savedSearches = await this.scrollSearch(this._index, this._searchType);

      const indexPatternToSavedSearchMap = {};
      // NOTE:
      // here you have to use modern "for of" so the await inside the loop are respected
      for (const indexPattern of indexPatterns) {
        const indexPatternId = indexPattern._id;
        // NOTE:
        // for case where there is no INDEX-PATTER entity present in the ontology
        // we create an fake entity on the fly
        const indexPatternEntity = _lodash2.default.find(entities, 'id', indexPatternId) || { id: indexPatternId };

        const foundSavedSearch = this._findRootSavedSearch(indexPatternId, savedSearches);
        if (foundSavedSearch) {
          this._logger.info('Found root saved search for index pattern: ' + indexPatternId);
          const entity = this._createSavedSearchEntityFromSavedSearchHitAndIndexPatternEntity(foundSavedSearch, null, indexPatternEntity);
          res = await serverOntologyClient.createEntity(entity);
          if (res.result !== 'updated') {
            const msg = 'Not able to create SAVED_SEARCH entity: ' + entity.id;
            this._logger.error(msg + '. See response:');
            this._logger.error(res);
            throw new Error(msg);
          }
          indexPatternToSavedSearchMap[indexPatternId] = foundSavedSearch._id;
          count++;
        } else {
          // create savedSearch object + SAVED_SEARCH entity + add id to the map
          const rootSavedSearchId = this._getUUID();
          res = await this._createSavedSearch(rootSavedSearchId, indexPatternId);
          if (res.result !== 'created') {
            const msg = 'Not able to create saved search object: ' + rootSavedSearchId;
            this._logger.error(msg + '. See response:');
            this._logger.error(res);
            throw new Error(msg);
          }
          count++;

          const entity = this._createRootSavedSearchEntityFromIndexPatternEntity(rootSavedSearchId, indexPatternEntity);

          res = await serverOntologyClient.createEntity(entity);
          if (res.result !== 'updated') {
            const msg = 'Not able to create SAVED_SEARCH entity: ' + entity.id;
            this._logger.error(msg + '. See response:');
            this._logger.error(res);
            throw new Error(msg);
          }
          indexPatternToSavedSearchMap[indexPatternId] = rootSavedSearchId;
          count++;
        }

        await serverOntologyClient.deleteEntityRelations(indexPatternId);
        await serverOntologyClient.deleteEntity(indexPatternId);
      }

      // at this point all index-patterns have corresponding root saved search
      // now iterate over all root saved searches and find all based on the same index-pattern and mark them as children
      for (const indexPattern of indexPatterns) {
        const indexPatternId = indexPattern._id;
        const indexPatternEntity = _lodash2.default.find(entities, 'id', indexPatternId);
        const rootSavedSearchId = indexPatternToSavedSearchMap[indexPatternId];
        // find other saved searches based on the same index-pattern
        const basedOnSameIndexPatternExceptTheRootOne = this._findSearchesBasedOnIndexPattern(indexPatternId, savedSearches, rootSavedSearchId);

        this._logger.info('Found other saved searches for root ' + rootSavedSearchId);

        if (basedOnSameIndexPatternExceptTheRootOne.length > 0) {
          // create entities
          for (const hit of basedOnSameIndexPatternExceptTheRootOne) {
            const entity = this._createSavedSearchEntityFromSavedSearchHitAndIndexPatternEntity(hit, rootSavedSearchId, indexPatternEntity);
            res = await serverOntologyClient.createEntity(entity);
            if (res.result !== 'updated') {
              const msg = 'Not able to create SAVED_SEARCH entity: ' + entity.id;
              this._logger.error(msg + '. See response:');
              this._logger.error(res);
              throw new Error(msg);
            }
            count++;
          }
        }
      }

      this._logger.info('indexPatternToSavedSearchMap:');
      this._logger.info(indexPatternToSavedSearchMap);

      if (relations.length) {
        count += this._updateRelations(relations, indexPatternToSavedSearchMap);
        await serverOntologyClient.saveRelations(relations);
      }

      const object = await this._client.get({
        index: this._index,
        type: this._ontologyType,
        ignore: [404],
        id: this._ontologyId
      });
      if (object.found && object._source['relational-graph']) {
        const relationalGraphItems = object._source['relational-graph'].items;
        _lodash2.default.each(relationalGraphItems, graphItem => {
          if (graphItem.type === 'node' && graphItem.d.entityType === 'INDEX_PATTERN') {
            const foundSavedSearch = this._findRootSavedSearch(graphItem.id, savedSearches);

            if (foundSavedSearch) {
              graphItem.id = foundSavedSearch._id;
              graphItem.d.entityType = _entity_type.EntityType.SAVED_SEARCH;
            }
          } else if (graphItem.type === 'link') {
            let foundSavedSearch = this._findRootSavedSearch(graphItem.id1, savedSearches);

            if (foundSavedSearch) {
              graphItem.id1 = foundSavedSearch._id;
            }

            foundSavedSearch = this._findRootSavedSearch(graphItem.id2, savedSearches);

            if (foundSavedSearch) {
              graphItem.id2 = foundSavedSearch._id;
            }
          }
        });

        object._source['relational-graph'].items = relationalGraphItems;
        await this._client.index({
          index: this._index,
          type: this._ontologyType,
          id: this._ontologyId,
          body: object._source,
          refresh: true
        });
      };

      this._logger.info('All done stopping Siren Gremlin Server');
      await gremlin.stop('Migration 27 clean');
    } catch (e) {
      if (e instanceof _gremlin_server_errors.GremlinError) {
        this._logger.error('Could not start the Siren Gremlin Server for Migration 27');
        this._logger.error(e);
      } else {
        this._logger.error('Could not perform Migration 27');
        this._logger.error(e);
        await gremlin.stop('Migration 27 error');
      }
    }

    return count;
  }
}
exports.default = Migration27;
module.exports = exports['default'];
