'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _lodash = require('lodash');

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 37.
 *
 * To fix: 'Pattern element of include/exclude for terms aggregations has been removed,
 *  causes an exception when migrating from 5 to 6
 *
 * If significant terms aggregation visualization and an include or exclude field exists
 *
 * This migration was written to cover the upgrade from 10.1.x to 10.2.x
 */

class Migration37 extends _migration2.default {

  constructor(configuration) {
    super(configuration);
    this._server = configuration.server;
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;

    this._type = 'doc';

    this._visualizationQuery = {
      query: {
        match: {
          type: 'visualization'
        }
      }
    };
  }

  static get description() {
    return 'Looking for include or exclude patterns key in significant terms aggregations for visualizations';
  }

  _isUpgradeable(obj) {
    const visState = JSON.parse(obj._source.visualization.visState);
    let isUpgradeable = false;
    (0, _lodash.each)(visState.aggs, bucket => {
      if (bucket.type = 'significant_terms') {
        if (bucket.params.exclude && bucket.params.exclude.pattern || bucket.params.include && bucket.params.include.pattern) {
          isUpgradeable = true;
        }
      }
    });
    return isUpgradeable;
  }

  _upgradeSingle(obj) {
    function replacer(key, value) {
      if (key === 'include' && value.pattern) {
        return value.pattern;
      }
      if (key === 'exclude' && value.pattern) {
        return value.pattern;
      }
      return value;
    }

    const visState = JSON.parse(obj._source.visualization.visState);
    const newAggs = [];

    visState.aggs.forEach(function (obj) {
      newAggs.push(JSON.parse(JSON.stringify(obj, replacer)));
    });

    visState.aggs = newAggs;

    obj._source.visualization.visState = JSON.stringify(visState);
    return JSON.stringify({
      update: {
        _index: obj._index,
        _type: obj._type,
        _id: obj._id
      }
    }) + '\n' + JSON.stringify({ doc: _extends({}, obj._source) }) + '\n';
  }

  async count() {
    const visualizations = await this.scrollSearch(this._index, this._type, this._visualizationQuery);
    let count = 0;
    for (const obj of visualizations) {
      if (this._isUpgradeable(obj)) {
        count++;
      }
    }
    return count;
  }

  async upgrade() {
    const visualizations = await this.scrollSearch(this._index, this._type, this._visualizationQuery);
    let count = 0;
    let body = '';
    for (const obj of visualizations) {
      if (this._isUpgradeable(obj)) {
        body += this._upgradeSingle(obj);
        count++;
      }
    }
    if (body.length > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }
    return count;
  }
}
exports.default = Migration37;
module.exports = exports['default'];
