'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _lodash = require('lodash');

var _migration = require('kibiutils/lib/migrations/migration');

var _migration2 = _interopRequireDefault(_migration);

var _uuid = require('uuid');

var _uuid2 = _interopRequireDefault(_uuid);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Investigate Core - Migration 41.
 *
 * Fixes #10250. Dashboard saved objects have panel objects defined in panelsJSON, which should always point
 * to saved objects of type 'visualization'. In 10.2.0 and before, it was possible to specify searches, which
 * represent search tables (i.e. the record table visualisation).
 *
 * Since 10.2.1, these should be changed by creating a visualisation of type 'kibi-table-vis', which points to
 * the search, and update the panelsJSON entry to point to the visualisation instead of the search.
 */

class Migration41 extends _migration2.default {

  constructor(configuration) {
    super(configuration);
    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;

    this._type = 'doc';

    this._dashboardQuery = {
      query: {
        match: {
          type: 'dashboard'
        }
      }
    };
  }

  static get description() {
    return 'Replacing saved search panels with records table visualization in dashboards';
  }

  async count() {
    const dashboards = await this.scrollSearch(this._index, this._type, this._dashboardQuery);
    return dashboards.filter(Migration41._isUpgradeable).length;
  }

  async upgrade() {
    this.searches = await this.scrollSearch(this._index, this._type, this._searchesQuery);
    const dashboards = await this.scrollSearch(this._index, this._type, this._dashboardQuery);
    const modifiedDashboards = await Promise.all(dashboards.filter(Migration41._isUpgradeable).map(this._upgradeSingle.bind(this)));

    if (modifiedDashboards.length) {
      const body = modifiedDashboards.flatMap(({ _index, _type, _id, _source }) => [{ index: { _index, _type, _id } }, _source]);
      await this._client.bulk({ refresh: true, body });
    }
    return modifiedDashboards.length;
  }

  static _isUpgradeable(dashboard) {
    return JSON.parse(dashboard._source.dashboard.panelsJSON).some(panel => panel.type === 'search');
  }

  async _upgradeSingle(dashboard) {
    const panels = await Promise.all(JSON.parse(dashboard._source.dashboard.panelsJSON).map(panel => panel.type === 'search' ? this._upgradePanel(panel) : panel));
    dashboard._source.dashboard.panelsJSON = JSON.stringify(panels);
    return dashboard;
  }

  async _upgradePanel(panel) {
    panel.id = await this._createRecordsVis(panel.id, panel.columns);
    panel.type = 'visualization';
    delete panel.columns;
    delete panel.sort;
    return panel;
  }

  async _createRecordsVis(searchId, columns) {
    const search = (0, _lodash.find)(this.searches, '_id', searchId);
    const visId = 'visualization:' + _uuid2.default.v1();
    await this._client.index({
      index: this._index,
      type: this._type,
      id: visId,
      refresh: true,
      body: {
        type: 'visualization',
        visualization: {
          description: '',
          kibanaSavedObjectMeta: {
            searchSourceJSON: '{"filter":[]}'
          },
          savedSearchId: searchId,
          title: search && search._source.search.title,
          uiStateJSON: '{}',
          version: 1,
          visState: JSON.stringify({
            aggs: [],
            listeners: {},
            params: {
              clickOptions: [],
              columns: columns || [],
              queryDefinitions: []
            },
            title: search && search._source.search.title,
            type: 'kibi-data-table',
            version: 2
          })
        }
      }
    });
    return visId;
  }
}
exports.default = Migration41;
module.exports = exports['default'];
