import { filter } from 'lodash';
import { uiModules } from 'ui/modules';
import { SavedObjectNotFound } from 'ui/errors';
import uuid from 'uuid';
import Promise from 'bluebird';
import _ from 'lodash';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { CrudType } from './crud_type';
import { SavedObjectsClientProvider } from 'ui/saved_objects';
import { DataModelPermissionsProvider } from './data_model_permissions';
import { OntologyWrapperProvider } from 'ui/kibi/quick_relations/ontology_wrapper';
import { VisColorMappedColorsProvider } from 'ui/vis/components/color/mapped_colors';
import { DataModelErrorType } from '../../data_model/controllers/error_type';

import {
  sendCreateIndexPatternRequest
} from 'plugins/kibana/management/sections/indices/create_index_pattern/send_create_index_pattern_request';

class DataModel {
  constructor(notify, config, ontologyModel, indexPatterns, savedSearches, savedEids, savedRelations, savedObjectsClient,
    dataModelPermissions, mappedColors) {
    this.beforeSave = [];
    this.config = config;
    this.notify = notify;
    this.ontologyModel = ontologyModel;
    this.indexPatterns = indexPatterns;
    this.savedSearches = savedSearches;
    this.savedEids = savedEids;
    this.savedRelations = savedRelations;
    this.savedObjectsClient = savedObjectsClient;
    this.dataModelPermissions = dataModelPermissions;
    this.mappedColors = mappedColors;
  }

  // Register function to be executed before entity from data model is saved
  //
  // f - function to return either promise or value
  //
  registerBeforeSave(f) {
    this.beforeSave.push(f);
  }

  deregisterBeforeSave(f) {
    for (let i = this.beforeSave.length; i >= 0; i--) {
      if (this.beforeSave[i] === f) {
        this.beforeSave.splice(i, 1);
      }
    }
  }

  // function which triggers all registered beforeSave functions
  triggerBeforeSave() {
    if (this.beforeSave.length === 0) {
      return Promise.resolve(0);
    }
    const promises = this.beforeSave.map(function (f) {
      return f();
    });

    return Promise.all(promises);
  }

  getRootSearchId(entityId, entities) {
    for (const e of entities) {
      if (e.type === EntityType.SAVED_SEARCH && e.id === entityId) {
        if (e.parentId === null) {
          return e.id;
        } else {
          return this.getRootSearchId(e.parentId, entities);
        }
      } else {
        continue;
      }
    }
  }

  _initMappedColors() {
    return this.ontologyModel.getEntityList()
      .then(entities => {
        const entityKeys = _.map(entities, e => e.label ? e.label : e.id);
        this.mappedColors.mapKeys(entityKeys);
        return this.mappedColors;
      });
  }

  _copyOptionalEntityProperties(entity, sourceEntity, mappedColors) {
    if (sourceEntity.indexPattern !== undefined && sourceEntity.indexPattern !== null) {
      entity.indexPattern = sourceEntity.indexPattern;
    }
    if (sourceEntity.label !== undefined && sourceEntity.label !== null) {
      entity.label = sourceEntity.label;
    }
    if (sourceEntity.icon !== undefined && sourceEntity.icon !== null) {
      entity.icon = sourceEntity.icon;
    }
    if (sourceEntity.color !== undefined && sourceEntity.color !== null) {
      entity.color = sourceEntity.color;
    } else if (mappedColors) {
      // set a default color
      const entityKey = sourceEntity.label ? sourceEntity.label : entity.id;
      mappedColors.mapKeys([entityKey]);
      entity.color = mappedColors.get(entityKey);
    }
    if (sourceEntity.shortDescription !== undefined && sourceEntity.shortDescription !== null) {
      entity.shortDescription = sourceEntity.shortDescription;
    }
    if (sourceEntity.longDescription !== undefined && sourceEntity.longDescription !== null) {
      entity.longDescription = sourceEntity.longDescription;
    }
    if (sourceEntity.primaryKeys !== undefined && sourceEntity.primaryKeys.length > 0) {
      entity.primaryKeys = sourceEntity.primaryKeys;
    }
    if (sourceEntity.singleValues !== undefined && sourceEntity.singleValues.length > 0) {
      entity.singleValues = sourceEntity.singleValues;
    }
    if (sourceEntity.searchFilter !== undefined && sourceEntity.searchFilter !== null) {
      entity.searchFilter = sourceEntity.searchFilter;
    }
    if (sourceEntity.searchQuery !== undefined && sourceEntity.searchQuery !== null) {
      entity.searchQuery = sourceEntity.searchQuery;
    }
    if (sourceEntity.instanceLabel) {
      if (sourceEntity.instanceLabel.type !== undefined && sourceEntity.instanceLabel.type !== null) {
        entity.instanceLabelType = sourceEntity.instanceLabel.type;
      }
      if (sourceEntity.instanceLabel.value !== undefined && sourceEntity.instanceLabel.value !== null) {
        if (sourceEntity.instanceLabel.value.indexOf('\n') !== -1) {
          entity.instanceLabelValue = sourceEntity.instanceLabel.value.split('\n').join('\\n');
        } else {
          entity.instanceLabelValue = sourceEntity.instanceLabel.value;
        }
      }
    }
  }

  _generateId() {
    return uuid.v1();
  }

  getIndexPropertiesToInsert(indexProperties) {
    if (!indexProperties) {
      indexProperties = {};
    }
    indexProperties.id = this._generateId();
    return indexProperties;
  }

  getIndexPatternSearchEntityToInsert(sourceEntity, mappedColors) {
    const entity = {
      // no id here as it will come from corresponding saved search
      // no need for parentId as this is the IndexPatternSearch
      type: EntityType.SAVED_SEARCH,
    };
    this._copyOptionalEntityProperties(entity, sourceEntity, mappedColors);
    return entity;
  }

  getSearchEntityToInsert(sourceEntity, entities, mappedColors, savedSearch) {
    const entity = {
      type: EntityType.VIRTUAL_ENTITY
    };
    if (savedSearch) {
      const parsedSearchSource = JSON.parse(savedSearch.kibanaSavedObjectMeta.searchSourceJSON);
      if (parsedSearchSource) {
        const filters = _.map(parsedSearchSource.filter, filter => _.omit(filter, ['$state']));
        entity.searchFilter = JSON.stringify(filters);
        entity.searchQuery = JSON.stringify(parsedSearchSource.query);
      }
    }
    this._copyOptionalEntityProperties(entity, sourceEntity, mappedColors);
    return entity;
  }

  getEntityIdentifierToInsert(sourceEntity, mappedColors) {
    const entity = {
      id: this._generateId(),
      type: EntityType.VIRTUAL_ENTITY,
    };
    this._copyOptionalEntityProperties(entity, sourceEntity, mappedColors);
    return entity;
  }

  getEntityForUpdate(sourceEntity, mappedColors, savedSearch) {
    const entity = {
      id: sourceEntity.id,
      type: sourceEntity.type
    };

    if (savedSearch) {
      const parsedSearchSource = JSON.parse(savedSearch.kibanaSavedObjectMeta.searchSourceJSON);
      if (parsedSearchSource) {
        const filters = _.map(parsedSearchSource.filter, filter => _.omit(filter, ['$state']));
        entity.searchFilter = JSON.stringify(filters);
        entity.searchQuery = JSON.stringify(parsedSearchSource.query);
      }
    }
    if (sourceEntity.parentId !== undefined) {
      entity.parentId = sourceEntity.parentId;
    }
    this._copyOptionalEntityProperties(entity, sourceEntity, mappedColors);
    return entity;
  }

  _getBeforeSavePromise(triggerBeforeSave) {
    return triggerBeforeSave ? this.triggerBeforeSave() : Promise.resolve();
  };

  _isIndexPatternSearch(entity) {
    return entity.type === EntityType.SAVED_SEARCH && entity.parentId === null;
  }

  _createSavedSearchObject(indexPatternId, entity) {
    return Promise.all([
      this.indexPatterns.get(indexPatternId),
      this.savedSearches.get()
    ])
      .then(([indexPattern, savedSearch]) => {
        savedSearch.title = entity.label;
        savedSearch.columns = ['_source'];
        savedSearch.searchSource
          .set('index', indexPattern)
          .highlightAll(true)
          .version(true)
          .size(this.config.get('discover:sampleSize'))
          .sort([indexPattern.timeFieldName, 'desc'])
          .query(null);
        savedSearch.siren = {
          ui: {
            icon: entity.icon,
            color: entity.color,
            shortDescription: entity.shortDescription,
          }
        };

        return savedSearch.save({ checkIfDuplicateExists: false });
      });
  }

  _createIndexPatternObject(indexProperties) {
    return this.indexPatterns.getIds()
      .then(indexPatternIds => {
        return sendCreateIndexPatternRequest(this.indexPatterns, indexProperties)
          .then(createdId => {
            if (!createdId) {
              return;
            }
            this.indexPatterns.cache.clear(createdId);

            // If there are no other indexpatterns, set this one as the preferred one
            if (!indexPatternIds.length) {
              this.config.set('defaultIndex', createdId);
            }
            return createdId;
          });
      });
  }

  _createEntityIdentifierObject(entity) {
    return this.savedEids.get()
      .then(savedEid => {
        savedEid.title = entity.label;
        savedEid.siren = {
          ui: {
            icon: entity.icon,
            color: entity.color,
            shortDescription: entity.shortDescription,
          }
        };

        return savedEid.save({ checkIfDuplicateExists: false });
      });
  }

  _createRelationObject(relation) {
    return this.savedRelations.get()
      .then(savedRelation => {
        savedRelation.domainId = relation.domain.id;
        savedRelation.domainField = relation.domain.field;
        savedRelation.rangeId = relation.range.id;
        savedRelation.rangeField = relation.range.field;
        savedRelation.directLabel = relation.directLabel;
        savedRelation.inverseLabel = relation.inverseLabel;
        savedRelation.inverseOf = `relation: ${uuid.v1()}`;
        savedRelation.title = relation.title || `${relation.domain.label} -> ${relation.directLabel} -> ${relation.range.label}`;

        return savedRelation;
      });
  }

  _addPropertiesToSearch(sourceEntity, savedSearch, entities, mappedColors) {
    if (!savedSearch.siren) {
      savedSearch.siren = { ui: {} };
    } else if (!savedSearch.siren.ui) {
      savedSearch.siren.ui = {};
    }

    if (entities) {
      const parentId = sourceEntity.parentId === null ? sourceEntity.id : this.getRootSearchId(sourceEntity.parentId, entities);
      savedSearch.siren.parentId = parentId;
    }

    if (sourceEntity.instanceLabel) {
      const instanceLabel = sourceEntity.instanceLabel;
      if (sourceEntity.instanceLabel.type) {
        savedSearch.siren.ui.instanceLabelType = instanceLabel.type;
      }
      if (sourceEntity.instanceLabel.value) {
        savedSearch.siren.ui.instanceLabelValue = instanceLabel.value;
      }
    }

    if (sourceEntity.icon) {
      savedSearch.siren.ui.icon = sourceEntity.icon;
    }
    if (sourceEntity.color) {
      savedSearch.siren.ui.color = sourceEntity.color;
    }
    if (sourceEntity.color !== undefined && sourceEntity.color !== null) {
      savedSearch.siren.ui.color = sourceEntity.color;
    } else if (mappedColors) {
      // set a default color
      const entityKey = sourceEntity.label ? sourceEntity.label : sourceEntity.id;
      mappedColors.mapKeys([entityKey]);
      savedSearch.siren.ui.color = mappedColors.get(entityKey);
    }
    if (sourceEntity.shortDescription !== undefined && sourceEntity.shortDescription !== null) {
      savedSearch.siren.ui.shortDescription = sourceEntity.shortDescription;
    }

    if (sourceEntity.label) {
      savedSearch.title = sourceEntity.label;
    }
  }

  // Updates the passed savedRelation with the passed rel.
  // Returns true if any property
  _updateRelationIfNeeded(savedRelation, rel) {
    const isInverse = savedRelation.id !== rel.id;
    let isUpgradeable = false;

    if (!isInverse) {
      if (rel.domain.id !== savedRelation.domainId ||
        rel.directLabel !== savedRelation.directLabel ||
        rel.range.id !== savedRelation.rangeId) {
        savedRelation.title = `${rel.domain.label} -> ${rel.directLabel} -> ${rel.range.label}`;
        isUpgradeable = true;
      }
      if (rel.directLabel !== savedRelation.directLabel) {
        savedRelation.directLabel = rel.directLabel;
        isUpgradeable = true;
      }
      if (rel.domain.id !== savedRelation.domainId) {
        savedRelation.domainId = rel.domain.id;
        isUpgradeable = true;
      }
      if (rel.domainField !== savedRelation.domainField) {
        savedRelation.domainField = rel.domainField;
        isUpgradeable = true;
      }
      if (rel.inverseLabel !== savedRelation.inverseLabel) {
        savedRelation.inverseLabel = rel.inverseLabel;
        isUpgradeable = true;
      }
      if (rel.range.id !== savedRelation.rangeId) {
        savedRelation.rangeId = rel.range.id;
        isUpgradeable = true;
      }
      if (rel.rangeField !== savedRelation.rangeField) {
        savedRelation.rangeField = rel.rangeField;
        isUpgradeable = true;
      }
    } else {
      if (rel.domain.id !== savedRelation.domainId ||
        rel.directLabel !== savedRelation.directLabel ||
        rel.range.id !== savedRelation.rangeId) {
        savedRelation.title = `${rel.range.label} -> ${rel.inverseLabel} -> ${rel.domain.label}`;
        isUpgradeable = true;
      }
      if (rel.directLabel !== savedRelation.inverseLabel) {
        savedRelation.inverseLabel = rel.directLabel;
        isUpgradeable = true;
      }
      if (rel.range.id !== savedRelation.domainId) {
        savedRelation.domainId = rel.range.id;
        isUpgradeable = true;
      }
      if (rel.rangeField !== savedRelation.domainField) {
        savedRelation.domainField = rel.rangeField;
        isUpgradeable = true;
      }
      if (rel.inverseLabel !== savedRelation.directLabel) {
        savedRelation.directLabel = rel.inverseLabel;
        isUpgradeable = true;
      }
      if (rel.domain.id !== savedRelation.rangeId) {
        savedRelation.rangeId = rel.domain.id;
        isUpgradeable = true;
      }
      if (rel.domainField !== savedRelation.rangeField) {
        savedRelation.rangeField = rel.domainField;
        isUpgradeable = true;
      }
    }

    // Properties that are fixed regardless of the direction
    if (rel.timeout !== savedRelation.timeout) {
      savedRelation.timeout = rel.timeout;
      isUpgradeable = true;
    }
    if (rel.joinType !== savedRelation.joinType) {
      savedRelation.joinType = rel.joinType;
      isUpgradeable = true;
    }

    return isUpgradeable;
  }

  // === Entity Id CRUD ===

  createEntityIdentifier(entity) {
    return Promise.all([
      this.dataModelPermissions.checkEntityIdPermissions(CrudType.CREATE),
      this._initMappedColors()
    ])
      .then(([allowed, mappedColors]) => {
        if (allowed) {
          const entityToInsert = this.getEntityIdentifierToInsert(entity, mappedColors);

          return this._createEntityIdentifierObject(entityToInsert)
            .then(id => {
              this.notify.info(`Entity Id: "${entityToInsert.label}" successfully created.`);
              return id;
            });
        }
      });
  }

  updateEntityIdentifier(entity, savedEid) {
    return Promise.all([
      this.dataModelPermissions.checkEntityIdPermissions(CrudType.UPDATE),
      this._initMappedColors()
    ])
      .then(([allowed, mappedColors]) => {
        if (allowed) {
          // add additional properties to savedEid, like ui settings
          this._addPropertiesToSearch(entity, savedEid, null, mappedColors);

          return this.triggerBeforeSave()
            .then(() => savedEid.save())
            .then(() => {
              if (entity.invalidRelations) {
                delete entity.invalidRelations;
                return this.notify.warning(`Entity Id: "${savedEid.title}" updated but`
                  + ` some of relations were not saved due unsolved state.`);
              } else {
                return this.notify.info(`Entity Id: "${savedEid.title}" successfully updated.`);
              }
            })
            .then(() => savedEid.id);
        }
      });
  }

  deleteEntityIdentifier(entityToDelete) {
    return this.dataModelPermissions.checkEntityIdPermissions(CrudType.DELETE)
      .then(allowed => {
        if (allowed) {
          return this.savedEids.get(entityToDelete.id)
            .then(savedEid => savedEid.delete())
            .then(() => this.deleteRelationsByDomainOrRange(entityToDelete.id))
            .then(() => this.notify.info(`Entity Id: "${entityToDelete.label}" successfully deleted.`))
            .then(() => entityToDelete.id);
        }
      });
  }

  // === Index Pattern Search CRUD ===

  createIndexPatternSearch(entity, indexProperties) {
    return Promise.all([
      this.dataModelPermissions.checkIndexPatternSearchPermissions(CrudType.CREATE),
      this._initMappedColors()
    ])
      .then(([allowed, mappedColors]) => {
        if (allowed) {
          const entityToInsert = this.getIndexPatternSearchEntityToInsert(entity, mappedColors);
          const indexPropertiesToInsert = this.getIndexPropertiesToInsert(indexProperties);

          return this._createIndexPatternObject(indexPropertiesToInsert)
            .then(indexPatternId => {
              entityToInsert.indexPatternId = indexPatternId;
              return indexPatternId;
            })
            .then(indexPatternId => this._createSavedSearchObject(indexPatternId, entityToInsert))
            .then(savedSearchId => {
              this.notify.info(`Index Pattern Search: "${entityToInsert.label}" successfully created.`);
              return savedSearchId;
            });
        }
      });
  }

  // === Search CRUD ===

  createSearch(entity, savedSearch, entities) {
    return Promise.all([
      this.dataModelPermissions.checkSearchPermissions(CrudType.CREATE),
      this._initMappedColors()
    ])
      .then(([allowed, mappedColors]) => {
        if (allowed) {
          // add additional properties to savedsearch, like parentId
          this._addPropertiesToSearch(entity, savedSearch, entities);

          return savedSearch.save()
            .then(id => {
              this.notify.info(`Search: "${savedSearch.title}" successfully created.`);
              return id;
            });
        }
      });
  }

  updateSearch(entity, savedSearch, triggerBeforeSave = true) {
    const promise =
      this._isIndexPatternSearch(entity) ?
        this.dataModelPermissions.checkIndexPatternSearchPermissions(CrudType.UPDATE) :
        this.dataModelPermissions.checkSearchPermissions(CrudType.UPDATE);

    return Promise.all([
      promise,
      this._initMappedColors()
    ])
      .then(([allowed, mappedColors]) => {
        if (allowed) {
          // add additional properties to savedsearch, like parentId
          this._addPropertiesToSearch(entity, savedSearch);

          return this._getBeforeSavePromise(triggerBeforeSave)
            .then(() => savedSearch.save())
            .then(() => {
              if (entity.invalidRelations) {
                delete entity.invalidRelations;
                return this.notify.warning(`Search: "${savedSearch.title}" successfully updated.`
                  + ` Incomplete relations were not saved.`);
              } else {
                return this.notify.info(`Search: "${savedSearch.title}" successfully updated.`);
              }
            })
            .then(() => savedSearch.id);
        }
      });
  }

  deleteSearch(entity) {
    const promise =
      this._isIndexPatternSearch(entity) ?
        this.dataModelPermissions.checkIndexPatternSearchPermissions(CrudType.DELETE) :
        this.dataModelPermissions.checkSearchPermissions(CrudType.DELETE);

    return promise
      .then(allowed => {
        if (allowed) {
          if (entity._objects.errorType === DataModelErrorType.INDEX_PATTERN_MISSING_DATA_INDICES) {
            return this.savedObjectsClient.delete('search', entity.id)
              .then(() => {
                if (this._isIndexPatternSearch(entity) && entity.indexPattern) {
                  return this.savedObjectsClient.delete('index-pattern', entity.indexPattern)
                    .catch(err => {
                      // do nothing
                    });
                }
              })
              .then(() => this.deleteRelationsByDomainOrRange(entity.id))
              .then(() => this.notify.info(`Search: "${entity.label}" successfully deleted.`))
              .then(() => entity.id);
          } else {
            return entity._objects.savedSearch.delete()
              .then(() => {
                if (this._isIndexPatternSearch(entity) && entity.indexPattern) {
                  return this.savedObjectsClient.delete('index-pattern', entity.indexPattern);
                }
              })
              .then(() => this.deleteRelationsByDomainOrRange(entity.id))
              .then(() => this.notify.info(`Search: "${entity.label}" successfully deleted.`))
              .then(() => entity.id);
          }
        }
      });
  }

  // === Relation CRUD ===

  createRelation(relation) {
    return this.dataModelPermissions.checkRelationPermissions(CrudType.CREATE)
      .then(allowed => {
        if (allowed) {
          return this._createRelationObject(relation)
            .then(savedRelation => {
              return savedRelation.save({ checkIfDuplicateExists: false })
                .then(id => [id, savedRelation.inverseOf]);
            })
            .then(([id, inverseOf]) => {
              relation.id = id;
              relation.inverseOf = inverseOf;
              return id;
            });
        }
      });
  }

  updateRelation(relation) {
    return this.dataModelPermissions.checkRelationPermissions(CrudType.UPDATE)
      .then(allowed => {
        if (allowed) {
          return relation.save()
            .then(() => relation.id);
        }
      });
  }

  deleteRelation(relation) {
    return this.dataModelPermissions.checkRelationPermissions(CrudType.DELETE)
      .then(allowed => {
        if (allowed) {
          return relation.delete();
        }
      });
  }

  deleteRelationsByDomainOrRange(entityId) {
    return this.dataModelPermissions.checkRelationPermissions(CrudType.DELETE)
      .then(allowed => {
        if (allowed) {
          return this.savedRelations.find('', undefined, undefined, true, true)
            .then(data => {
              const relations = data.hits;
              // We need to delete the savedobject only once, for each couple of relations (direct and inverse)
              const deleted = new Set();
              return Promise.each(relations, relation => {
                if ((relation.domainId === entityId || relation.rangeId === entityId)
                  && !deleted.has(relation.id)) {
                  this.savedRelations.delete([relation.id]);
                  deleted.add(relation.inverseOf);
                }
              });
            });
        }
      });
  }

  // === Other methods ===

  createRelations(relations) {
    return Promise.map(relations,
      rel => this.createRelation(rel), { concurrency: 2 });
  }

  updateRelations(relations) {
    return Promise.map(relations, rel => {
      const savedRelation = rel._objects.savedRelation;
      const isToUpgrade = this._updateRelationIfNeeded(savedRelation, rel);
      if (isToUpgrade) {
        return this.updateRelation(savedRelation);
      }
    }, { concurrency: 2 });
  }

  deleteRelations(relationIds) {
    // Use the ontologymodel here to handle the case where the user
    // is deleting the inverse relation (for which we don't have a save object).
    return this.ontologyModel.getRelationMap()
      .then(relMap => {
        return Promise.map(relationIds, id => {
          const savedRel = relMap[id]._objects.savedRelation;
          return this.deleteRelation(savedRel);
        }, { concurrency: 2 });
      });
  }

  findIndexPatternSearchByPattern({ indexPropertiesName: pattern, silent = false }) {
    return this.ontologyModel.getEntityList()
      .then(entities => {
        const filtered = filter(entities, entity => this._isIndexPatternSearch(entity));
        // fetch index-patterns and filter by pattern
        const promises = filtered.map(entity => {
          return this.indexPatterns.get(entity.indexPattern)
            .then(indexPattern => {
              if (indexPattern.title === pattern) {
                return entity;
              } else {
                return null;
              }
            })
            .catch(err => {
              if (!silent) {
                this.notify.error(err);
              }
              return null;
            });
        });

        return Promise.all(promises)
          .then(entities => {
            return filter(entities, entity => entity !== null);
          });


      });
  }
}

uiModules
  .get('kibana')
  .service('dataModel', function ($injector, Private, config, createNotifier, indexPatterns, savedSearches, savedEids,
    savedRelations, ontologyModel) {
    const savedObjectsClient = Private(SavedObjectsClientProvider);
    const dataModelPermissions = Private(DataModelPermissionsProvider);
    const wrappedOntologyModel = Private(OntologyWrapperProvider).forOntologyModel(ontologyModel);
    const mappedColors = Private(VisColorMappedColorsProvider);
    const notify = createNotifier({
      location: 'Data Model'
    });
    return new DataModel(notify, config, wrappedOntologyModel, indexPatterns, savedSearches, savedEids, savedRelations,
      savedObjectsClient, dataModelPermissions, mappedColors);
  });
