import { CrudType } from './crud_type';

export const CREATE_INDEXPATTERN = [
  { type: 'index-pattern', permission: 'view' },
  { type: 'index-pattern', permission: 'create' }
];
export const UPDATE_INDEXPATTERN = [
  // To update an index pattern object we will remove it
  // and then re-create it - permissions reflect this.
  { type: 'index-pattern', permission: 'view' },
  { type: 'index-pattern', permission: 'delete' },
  { type: 'index-pattern', permission: 'create' }
];
export const DELETE_INDEXPATTERN = [
  { type: 'index-pattern', permission: 'view' },
  { type: 'index-pattern', permission: 'delete' },
];

export const CREATE_SEARCH = [
  { type: 'search', permission: 'view' },
  { type: 'search', permission: 'create' }
];
export const UPDATE_SEARCH = [
  { type: 'search', permission: 'view' },
  { type: 'search', permission: 'change' }
];
export const DELETE_SEARCH = [
  { type: 'search', permission: 'view' },
  { type: 'search', permission: 'delete' }
];

export const CREATE_RELATION = [
  { type: 'relation', permission: 'view' },
  { type: 'relation', permission: 'create' }
];
export const UPDATE_RELATION = [
  { type: 'relation', permission: 'view' },
  { type: 'relation', permission: 'change' }
];
export const DELETE_RELATION = [
  { type: 'relation', permission: 'view' },
  { type: 'relation', permission: 'delete' }
];

export const CREATE_EID = [
  { type: 'eid', permission: 'view' },
  { type: 'eid', permission: 'create' }
];
export const UPDATE_EID = [
  { type: 'eid', permission: 'view' },
  { type: 'eid', permission: 'change' }
];
export const DELETE_EID = [
  { type: 'eid', permission: 'view' },
  { type: 'eid', permission: 'delete' }
];


class DataModelPermissions {

  constructor(kibiAccessControl) {
    this.kibiAccessControl = kibiAccessControl;
  }

  checkSearchPermissions(crudType) {
    if (!this.kibiAccessControl) {
      return Promise.resolve(true);
    }

    let permissions = [];
    switch (crudType) {
      case CrudType.CREATE:
        permissions = permissions.concat(CREATE_SEARCH);
        break;
      case CrudType.UPDATE:
        permissions = permissions.concat(UPDATE_SEARCH);
        break;
      case CrudType.DELETE:
        permissions = permissions.concat(DELETE_SEARCH);
        break;
      default:
        throw new Error('Unsupported crud type ' + crudType + ' on saved searches');
    }
    return this.kibiAccessControl.checkPermissions(permissions);
  }


  checkIndexPatternSearchPermissions(crudType) {
    if (!this.kibiAccessControl) {
      return Promise.resolve(true);
    }

    let permissions = [];
    switch (crudType) {
      case CrudType.CREATE:
        permissions = permissions.concat(CREATE_INDEXPATTERN, CREATE_SEARCH);
        break;
      case CrudType.UPDATE:
        permissions = permissions.concat(UPDATE_INDEXPATTERN, UPDATE_SEARCH);
        break;
      case CrudType.DELETE:
        permissions = permissions.concat(DELETE_INDEXPATTERN, DELETE_SEARCH);
        break;
      default:
        throw new Error('Unsupported crud type ' + crudType + ' on saved indexpatterns');
    }
    return this.kibiAccessControl.checkPermissions(permissions);
  }

  checkEntityIdPermissions(crudType) {
    if (!this.kibiAccessControl) {
      return Promise.resolve(true);
    }

    let permissions = [];
    switch (crudType) {
      case CrudType.CREATE:
        permissions = permissions.concat(CREATE_EID);
        break;
      case CrudType.UPDATE:
        permissions = permissions.concat(UPDATE_EID);
        break;
      case CrudType.DELETE:
        permissions = permissions.concat(DELETE_EID);
        break;
      default:
        throw new Error('Unsupported crud type ' + crudType + ' on saved eids');
    }
    return this.kibiAccessControl.checkPermissions(permissions);
  }

  checkRelationPermissions(crudType) {
    if (!this.kibiAccessControl) {
      return Promise.resolve(true);
    }

    let permissions = [];
    switch (crudType) {
      case CrudType.CREATE:
        permissions = permissions.concat(CREATE_RELATION);
        break;
      case CrudType.UPDATE:
        permissions = permissions.concat(UPDATE_RELATION);
        break;
      case CrudType.DELETE:
        permissions = permissions.concat(DELETE_RELATION);
        break;
      default:
        throw new Error('Unsupported crud type ' + crudType + ' on saved relations');
    }
    return this.kibiAccessControl.checkPermissions(permissions);
  }

}



export function DataModelPermissionsProvider($injector) {
  let kibiAccessControl;
  if ($injector.has('kibiAccessControl')) {
    kibiAccessControl = $injector.get('kibiAccessControl');
  }
  return new DataModelPermissions(kibiAccessControl);
}
