import chrome from 'ui/chrome';
import { uiModules } from 'ui/modules';
import { get } from 'lodash';

uiModules
  .get('kibi_datasources/services/jdbc_datasources')
  .service('jdbcDatasources', function ($http, Promise, federateResolver) {

    const DEFAULT_DATASOURCE_TYPES = {
      'Dremio': {
        'driverClassName': 'com.dremio.jdbc.Driver',
        'defaultURL': 'jdbc:dremio:direct={{host}}:{{port}}{{databasename}}',
        'defaultPort': 31010,
        'disclaimer': 'This is a sample connection string, see the <a target="_blank" href="https://docs.dremio.com/drivers/dremio-jdbc-driver.html">Dremio JDBC documentation</a> for further information.'
      },
      'Impala': {
        'driverClassName': 'com.cloudera.impala.jdbc41.Driver',
        'defaultURL': 'jdbc:impala://{{host}}:{{port}}/default;UseNativeQuery=1',
        'defaultPort': 21050,
        'disclaimer': 'This is a sample connection string, see the <a target="_blank" href="https://www.cloudera.com/documentation/enterprise/5-9-x/topics/impala_jdbc.html">Impala JDBC documentation</a> for further information.'
      },
      'MySQL': {
        'driverClassName': 'com.mysql.jdbc.Driver',
        'defaultURL': 'jdbc:mysql://{{host}}:{{port}}{{databasename}}?useLegacyDatetimeCode=false',
        'defaultPort': 3306,
        'disclaimer': 'This is a sample connection string, see the <a target="_blank" href="https://dev.mysql.com/doc/connector-j/5.1/en/connector-j-reference.html">MySQL Connector/J documentation</a> for further information.'
      },
      'PostgreSQL': {
        'driverClassName': 'org.postgresql.Driver',
        'defaultURL': 'jdbc:postgresql://{{host}}:{{port}}{{databasename}}',
        'defaultPort': 5432,
        'disclaimer': 'This is a sample connection string, see the <a target="_blank" href="https://jdbc.postgresql.org/documentation/94/connect.html">PostgreSQL JDBC documentation</a> for further information.'
      },
      'SQL Server 2017': {
        'driverClassName': 'com.microsoft.sqlserver.jdbc.SQLServerDriver',
        'defaultURL': 'jdbc:sqlserver://{{host}}:{{port}}{{databasename}}',
        'defaultPort': 1433,
        'disclaimer': 'This is a sample connection string, see the <a target="_blank" href="https://docs.microsoft.com/en-us/sql/connect/jdbc/building-the-connection-url">SQL Server JDBC documentation</a> for further information.'
      },
      'SAP ASE 15.7': {
        'driverClassName': 'net.sourceforge.jtds.jdbc.Driver',
        'defaultURL': 'jdbc:jtds:sybase://{{host}}:{{port}}{{databasename}}',
        'defaultPort': 5000,
        'disclaimer': 'This is a sample connection string, see the <a target="_blank" href="http://jtds.sourceforge.net/faq.html#urlFormat">jTDS documentation</a> for further information.' +
                      ' If your setup is using the jConnect JDBC Driver, see the <a target="_blank" href="https://wiki.scn.sap.com/wiki/display/SYBCON/jConnect+Driver+Overview">jConnect documentation</a> for further information.'
      },
      'Oracle Database 12c': {
        'driverClassName': 'oracle.jdbc.OracleDriver',
        'defaultURL': 'jdbc:oracle:thin:@//{{host}}:{{port}}{{databasename}}',
        'defaultPort': 1521,
        'disclaimer': 'This is a sample connection string, see the <a target="_blank" href="https://docs.oracle.com/cd/E11882_01/java.112/e16548/urls.htm#JJDBC08200">Oracle JDBC documentation</a> for further information.'
      },
      'Spark SQL 2.2':  {
        'driverClassName': 'com.simba.spark.jdbc41.Driver',
        'defaultURL': 'jdbc:spark://{{host}}:{{port}}{{databasename}};UseNativeQuery=1',
        'defaultPort': 10000,
        'disclaimer': 'This is a sample connection string, see the <a target="_blank" href="https://www.simba.com/products/Spark/doc/JDBC_InstallGuide/content/jdbc/sp/using/connectionurl.htm">Simba Spark JDBC documentation</a> for further information.'
      },
      'Presto': {
        'driverClassName': 'com.facebook.presto.jdbc.PrestoDriver',
        'defaultURL': 'jdbc:presto://{{host}}:{{port}}{{databasename}}',
        'defaultPort': 8080,
        'disclaimer': 'This is a sample connection string, see the <a target="_blank" href="https://prestodb.io/docs/current/installation/jdbc.html">Presto JDBC documentation</a> for further information.'
      }
    };

    class JdbcDatasources {

      get datasourceDefaults() {
        return DEFAULT_DATASOURCE_TYPES;
      }

      constructor() {
        this.basePath = chrome.getBasePath();
        this.datasourceBaseUrl = `${this.basePath}/connector_elasticsearch/_siren/connector/datasource`;
        this.indexBaseUrl = `${this.basePath}/connector_elasticsearch/_siren/connector/index`;
      }

      get(id) {
        return $http.get(this.datasourceBaseUrl + '/' + id).then(resp => resp.data);
      }

      list() {
        return $http.get(this.datasourceBaseUrl + '/_search').then(res => {
          return res.data.hits.hits;
        });
      }

      save(datasource) {
        return $http.put(this.datasourceBaseUrl + '/' + datasource._id, datasource._source);
      }

      delete(id) {
        return $http.delete(this.datasourceBaseUrl + '/' + id);
      }

      validate(datasource) {
        if (datasource._source.elastic) {
          return $http.post(`${this.basePath}/api/console/proxy?path=%2F${datasource._id}%3A*%2F_search%3Fsize%3D0&method=GET`)
            .then(res => res.data)
            .catch(res => Promise.reject(res.data));
        }
        return $http.post(this.datasourceBaseUrl + '/' + datasource._id + '/_validate', datasource._source)
          .then(res => res.data)
          .catch(res => Promise.reject(res.data));
      }

      getRemoteClusters() {
        return $http.get(`${this.basePath}/elasticsearch/_cluster/settings`)
          .then(res => Object.keys(get(res, 'data.persistent.cluster.remote') || {}))
          .catch(res => Promise.reject(res.data));
      }

      /**
       * Return the indices present in a remote cluster
       * @param  {String} remoteEsCluster
       * @return {Array.<Object>}
       */
      getRemoteIndices(remoteEsCluster) {
        const query = {
          size: 0,
          aggs: {
            indices: {
              terms: {
                field: '_index',
                size: 10
              }
            }
          }
        };
        return $http.post(`${this.basePath}/api/console/proxy?path=%2F${remoteEsCluster}%3A*%2F_search&method=POST`, query)
          .then(res => (get(res, 'data.aggregations.indices.buckets') || []).map(ele => {
            return {
              name: ele.key.replace(`${remoteEsCluster}:`, ''),
              count: ele.doc_count
            };
          }))
          .catch(res => Promise.reject(res.data));
      }

      getVirtualIndex(id) {
        return $http.get(this.indexBaseUrl + '/' + id).then(res => res.data);
      }

      deleteVirtualIndex(id) {
        return $http.delete(this.indexBaseUrl + '/' + id);
      }

      createVirtualIndex(index) {
        return $http.put(this.indexBaseUrl + '/' + index._id, index._source).then(res => {
          if (res.status === 201) {
            return res.data;
          } else if (res.status === 200) {
            return {
              updated: true
            };
          } else {
            return Promise.reject(new Error('Could not create an index pattern ' + JSON.stringify(res)));
          }
        });
      }

      listVirtualIndices() {
        return $http.get(this.indexBaseUrl + '/_search').then(res => {
          return res.data.hits.hits;
        });
      }

      getMetadata(datasourceName, catalog, schema) {
        if (!datasourceName) {
          return Promise.resolve([]);
        }
        return $http({
          url: `${this.datasourceBaseUrl}/${datasourceName}/_metadata`,
          method: 'POST',
          params: {
            catalog,
            schema
          }
        })
          .then(res => {
            return res.data;
          });
      }

      getDatasourceTypes() {
        if (federateResolver.checkApiAvailability('DATASOURCE_TYPES_API_V1')) {
          return $http.get(this.datasourceBaseUrl)
            .then(res => res.data);
        }
        return Promise.resolve(this.datasourceDefaults);
      }

      getMetadataFields(datasourceName, catalog, schema, resource) {
        if (!datasourceName) {
          return Promise.resolve([]);
        }
        return $http({
          url: `${this.datasourceBaseUrl}/${datasourceName}/_resource_metadata`,
          method: 'POST',
          params: {
            catalog,
            schema,
            resource
          }
        })
          .then(res => {
            return res.data;
          });
      }
    }
    return new JdbcDatasources();
  });
