import { IndexPatternsGetProvider }  from 'ui/index_patterns/_get';
import { VisTypesRegistryProvider } from 'ui/registry/vis_types';
import { SavedObjectsClientProvider } from 'ui/saved_objects';
import _ from 'lodash';
import kibiUtils from 'kibiutils';

export default function ImportHelperFactory(config, es, kibiVersion, kbnIndex,
  queryEngineClient, Private, Promise, indexPatterns, dashboardGroups, ontologyModel) {

  const getIds = Private(IndexPatternsGetProvider);
  const visTypes = Private(VisTypesRegistryProvider);
  const savedObjectsClient = Private(SavedObjectsClientProvider);

  class ImportExportHelper {

    createIndexPattern(doc) {
      return savedObjectsClient.update('index-pattern', doc._id, doc._source);
    }

    loadIndexPatterns(indexPatternDocuments, notify) {
      if (indexPatternDocuments && indexPatternDocuments.length > 0) {
        const promises = [];
        _.each(indexPatternDocuments, (doc) => {
          // lets try to fetch the field mappings to check
          // that index-pattern matches any existing indices
          const promise = es.indices.getFieldMapping({
            index: doc._source.title,
            fields: '*',
            allowNoIndices: false,
            includeDefaults: true
          }).catch((err) => {
            notify.warning(
              'Imported index-pattern: [' + doc._source.title + '] did not match any indices. ' +
              'If you would like to remove it go to Settings->Indices'
            );
          }).finally(() => {
            return this.createIndexPattern(doc);
          });
          promises.push(promise);
        });
        return Promise.all(promises);
      } else {
        return Promise.resolve(true);
      }
    }

    loadConfig(configDocument, notify) {
      if (configDocument) {
        // kibi: siren uses 'siren' config
        if (configDocument._id === 'config:siren') {
          // override existing config values
          const promises = [];
          _.each(configDocument._source, function (value, key) {
            promises.push(config.set(key, value));
          });
          return Promise.all(promises);
        } else {
          notify.error(
            'Config object version [' + configDocument._id + '] in the import ' +
            'does not match current version [ siren ]\n' +
            'None of the advanced settings parameters were imported'
          );
          return Promise.resolve(true);
        }
      } else {
        // return Promise so we can chain the other part
        return Promise.resolve(true);
      }
    }

    loadOntologyModel(ontologyDocuments, notify) {
      if (ontologyDocuments.length === 0) {
        return Promise.resolve(true);
      } else if (ontologyDocuments.length === 1) {
        const ontologyDoc = ontologyDocuments[0];
        return savedObjectsClient.update('ontology-model', ontologyDoc._id, ontologyDoc._source);
      } else {
        notify.error('Detected more then one ontology document. Please remove one from import file');
        return Promise.resolve(false);
      }
    }

    clearCaches() {
      return queryEngineClient.clearCache() // reloads all in memory queries on the server in query engine
        .then(() => ontologyModel.clearCache()); // removes cached ontology (entities) on the client side
    }

    checkVisualizationTypeExists(doc, notify) {
      // extra check to make sure that required visualisation plugin is present
      try {
        const visState = JSON.parse(doc._source.visState);
        if (!visTypes.byName[visState.type]) {
          notify.error(
            'Unknown visualisation type [' + visState.type + '] for [' + doc._id + ']. ' +
            'Make sure that all required plugins are installed'
          );
          return false;
        } else {
          return true;
        }
      } catch (err) {
        notify.error('Unknown error while parsing visState of [' + doc._id + '] visualisation.');
      }
    }

    /*
     * Remove defaults scripts from the list of import objects
     */
    removeDefaultScripts(docs, notify) {

      // to grab this list do
      // grep -r title kibi-enterprise/kibi-plugins/graph_browser/scripts/*.json |  awk -F ''' '{printf('\'%s\',\n', $4)}'
      const sampleScriptTitles = [
        'Add geo-locations for map visualization.  (works only with Kibi Demo data)',
        'Add time fields. (works only with Kibi Demo data)',
        'Default Expansion Policy',
        'Expand by relation',
        'Advanced lens. - default implementation',
        'Associate records based on ontology lens - default implementation',
        'Color lens. - default implementation',
        'Graph metric lens. - default implementation',
        'Label lens. - default implementation',
        'Show record content on tooltips lens. - default implementation',
        'Size lens. - default implementation',
        'Time and location lens. - default implementation',
        'Select - All',
        'Select - By Edge Count',
        'Select - By Entity Type',
        'Select - Extend',
        'Select - Invert',
        'Shortest Path',
        'Show nodes count by type',
        'Signal Dead Companies'
      ];

      const sampleScriptsId = _.map(sampleScriptTitles, title => 'script:' + kibiUtils.slugifyId(title));

      const defaultScriptsInDocs = [];
      docs = _.filter(docs, function (doc) {
        if (doc._type === 'script') {
          if (_.includes(sampleScriptsId, doc._id)) {
            defaultScriptsInDocs.push(doc._source.title);
            return false;
          } else {
            return true;
          }
        } else {
          return true;
        }
      });

      if (defaultScriptsInDocs.length > 0) {
        notify.warning(
          'These scripts [' + defaultScriptsInDocs + '] are immutable scripts and cannot be modified'
        );
      }
      return docs;
    }

    /*
     * Add config and index patterns to the list of exported objects
     */
    addExtraObjectForExportAll(objectsToExport) {
      objectsToExport.push([{ id: 'config:siren', type: 'config' }]);

      return Promise.resolve(objectsToExport);
    }

    moveConfigToTop(docs) {
      docs.sort(function (a, b) {
        if (a.type === 'config' && b.type !== 'config') {
          return -1;
        } else if (a.type !== 'config' && b.type === 'config') {
          return 1;
        } else {
          if (a.type < b.type) {
            return -1;
          } else if (a.type > b.type) {
            return 1;
          } else {
            return 0;
          }
        }
      });
    }

    importSirenDocsAndSortDocuments(docs, notify) {
      // kibi: change the import to sequential to solve the dependency problem between objects
      // as visualisations could depend on searches
      // lets order the export to make sure that searches comes before visualisations
      // then also import object sequentially to avoid errors
      const configDocument = _.find(docs, function (o) {
        return o._type === 'config';
      });

      // kibi: added to manage index-patterns import
      const indexPatternDocuments = _.filter(docs, function (o) {
        return o._type === 'index-pattern';
      });

      // kibi: added to manage ontology-model import
      const ontologyDocuments = _.filter(docs, function (o) {
        return o._type === 'ontology-model';
      });

      const self = this;
      docs = _.filter(docs, function (doc) {
        if (doc._type === 'visualization') {
          return self.checkVisualizationTypeExists(doc, notify);
        } else {
          return doc._type !== 'config' && doc._type !== 'index-pattern' && doc._type !== 'ontology-model';
        }
      });

      // kibi: added to sort the docs by type
      docs.sort(function (a, b) {
        if (a._type === 'search' && b._type !== 'search') {
          return -1;
        } else if (a._type !== 'search' && b._type === 'search') {
          return 1;
        } else {
          if (a._type < b._type) {
            return -1;
          } else if (a._type > b._type) {
            return 1;
          } else {
            return 0;
          }
        }
      });

      return this.loadIndexPatterns(indexPatternDocuments, notify)
        .then(() => this.loadConfig(configDocument, notify))
        .then(() => this.loadOntologyModel(ontologyDocuments, notify))
        .then(() => docs);
    }
  }

  return new ImportExportHelper();
};
