'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _joi = require('joi');

var _joi2 = _interopRequireDefault(_joi);

var _model = require('../model');

var _model2 = _interopRequireDefault(_model);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function handleError(e) {
  // if we are unable to access for reasons other than security restiction
  // rethrow the error
  if (e.status !== 403 && e.status !== 500) {
    throw e;
  }
}

/**
 * Model for index pattern objects.
 */
class IndexPatternModel extends _model2.default {
  constructor(server) {

    const schema = _joi2.default.object().keys({
      title: _joi2.default.string(),
      timeFieldName: _joi2.default.string(),
      intervalName: _joi2.default.string(),
      sourceFilters: _joi2.default.string(),
      paths: _joi2.default.object().default({}),
      mappingFormats: _joi2.default.object().default({}),
      fields: _joi2.default.object().default({}),
      fieldFormatMap: _joi2.default.string(),
      excludeIndices: _joi2.default.boolean() // kibi: added by Kibi
    });

    super(server, 'index-pattern', schema, 'Index pattern');
  }

  async filterSearchResponse(request = {}, searchResponse, excludedIndices = [], ignoreBroken) {
    const filteredHits = [];

    const excludedObj = await this._buildExcludedIndexSet(excludedIndices);

    if (searchResponse && searchResponse.hits && searchResponse.hits.hits) {
      for (const indexPattern of searchResponse.hits.hits) {
        const title = indexPattern._source['index-pattern'].title;
        if (excludedObj.indexSet.has(title)) {
          continue;
        }

        if (title.indexOf(',') > -1) {
          // handle comma separated index-patterns
          const subTitles = title.split(',');

          for (const subTitle of subTitles) {
            if (excludedObj.indexSet.has(subTitle)) {
              continue;
            }
            try {
              const resp = await this._testIndexPattern(subTitle, request, excludedObj);

              if (resp) {
                filteredHits.push(indexPattern);
                // We just need one single subTitle to work to return the index-pattern
                break;
              }
            } catch (e) {
              handleError(e);
            }
          }
        } else {
          try {
            const resp = await this._testIndexPattern(title, request, excludedObj);

            if (resp) {
              filteredHits.push(indexPattern);
            } else if (ignoreBroken) {
              filteredHits.push(indexPattern);
            }
          } catch (e) {
            handleError(e);
          }
        }
      }
    }

    searchResponse.hits.hits = filteredHits;
    searchResponse.hits.total = filteredHits.length;

    return searchResponse;
  }

  // Utility function to handle exluded indices.
  // In case the are star pattern we build the _cat indices suffix,
  // otherwise we add the indexes to a set. returns an object like
  // {
  //   catSuffix: '-starpatter*,...',
  //   indexSet: Set()
  // }
  async _buildExcludedIndexSet(excludedIndices) {
    const excludedObject = {
      indexSet: new Set(),
      catSuffix: ''
    };

    const excludedStarPatterns = [];
    for (const excludedIndex of excludedIndices) {
      if (excludedIndex.indexOf('*') > -1) {
        excludedObject.catSuffix += ',-' + excludedIndex;
      } else {
        excludedObject.indexSet.add(excludedIndex);
      }
    }

    return excludedObject;
  }

  // Test if the user has permission to see the passed index
  async _testIndexPattern(title, req, excludedObj) {
    if (title.indexOf('*') > -1) {
      return await this._testStarPattern(title, req, excludedObj);
    } else {
      // Need to add the catSuffix here to remove potentially excluded star patterns.
      // Eg: we're testing 'investor' and our exclusion list is [ 'inv*' ]
      const catTitle = title + excludedObj.catSuffix;
      return await this._cluster.callWithRequest(req, 'indices.exists', { index: catTitle });
    }
  }

  // Test if the user has permissions to see the passed star indexpattern
  async _testStarPattern(starTitle, req, excludedObj) {
    // Need to add the exclusion list as we may have excluded a subset of the star pattern we're expanding.
    // Eg: we're testing '*' and our exclusion list is [ 'inv*' ]
    const catStarTitle = starTitle + excludedObj.catSuffix;
    const allIndices = await this._cluster.callWithInternalUser('cat.indices', { format: 'json', h: 'index', index: catStarTitle });

    for (const indexItem of allIndices) {
      const index = indexItem.index;
      if (excludedObj.indexSet.has(index)) {
        continue;
      }

      let resp;
      try {
        // No need to add the catSuffix here as the index comes from the allIndices,
        // which are already filtered using the suffix
        resp = await this._cluster.callWithRequest(req, 'indices.exists', { index });
      } catch (e) {
        handleError(e);
      }
      if (resp) {
        return true;
      }
    }

    return false;
  }
}
exports.default = IndexPatternModel;
module.exports = exports['default'];
