import _ from 'lodash';

import { RequestQueueProvider } from '../_request_queue';
import { FetchTheseProvider } from './fetch_these';
import { CallResponseHandlersProvider } from './call_response_handlers';
import { ReqStatusProvider } from './req_status';

export function FetchProvider(Private, Promise, config) {

  const requestQueue = Private(RequestQueueProvider);
  const immediatelyFetchThese = Private(FetchTheseProvider);
  const callResponseHandlers = Private(CallResponseHandlersProvider);
  const INCOMPLETE = Private(ReqStatusProvider).INCOMPLETE;

  // Note:
  // This behave slightly different than Kibana
  // We make sure that each request is send separatelly
  const fetch = () => {
    const requests = requestQueue.getPending();
    if (requests) {
      for (let i = 0; i < requests.length; i++) {
        immediatelyFetchThese([requests[i]]);
      }
    }
  };

  // Note:
  // This one was implemented to take advantage of federate cache on complex dashboards 360 with many joins
  const fetchSequentially = () => {
    const requests = requestQueue.getPending();
    const resolvedArray = [];
    // Note: here we are mapping to an array of functions which return promise on call not array of promises)
    const requestsArray = requests.map(req => {
      return () => {
        return new Promise((resolve, reject) => {
          immediatelyFetchThese([req]);
          const resolved = req.getCompletePromise();
          resolvedArray.push(resolved);
          resolved.then(() => {
            resolve();
          });
        });
      };
    });

    const lastPromise = requestsArray.reduce(function (prev, curr, i) {
      return prev.then(function (res) {
        return curr(res);
      });
    }, Promise.resolve(0));

    return lastPromise.then(() => resolvedArray);
  };

  const debouncedFetch = _.debounce(() => {
    immediatelyFetchThese(requestQueue.getPending());
  }, {
    wait: 10,
    maxWait: 50
  });

  this._fetchTheseSoon = function (requests) {
    requests.forEach(req => req._setFetchRequested());

    if (config.get('courier:batchSearches')) {
      debouncedFetch();
      return Promise.all(requests.map(req => req.getCompletePromise()));
    } else {
      // Note: check courier:sequentialSearches only if 'courier:batchSearches' is false
      if (config.get('courier:sequentialSearches')) {
        return fetchSequentially();
      } else {
        fetch();
        return Promise.all(requests.map(req => req.getCompletePromise()));
      }
    }
  };

  this.fetchQueued = (strategy) => {
    // siren: start
    // If any requests are in progress when a new request is sent
    // (inProgress means request._fetchRequested = true and request.started = true)
    // then the request is stopped and restarted to make sure the latest changes
    // are reflected in the response. Previously a new request for the same source was
    // ignored if the last request was in progress
    requestQueue.get(strategy).forEach(request => {
      const isMultichartRequest = request.source && request.source.vis &&
      (request.source.vis.$$sirenSingleCall === true || request.source.vis.$$sirenMultichart === true);

      if (
        request.inProgress() &&
          request.strategy.clientMethod === 'msearch' &&
          !isMultichartRequest
      ) {
        request.retry();
      }
    });
    // siren: end

    //siren: Multichart need this to filter out duplicated requests
    const queuedRequests = _.filter(requestQueue, function (req) {
      return req && req.source && req.source.vis && req.source.vis.$$sirenMultichart;
    });

    if (queuedRequests.length > 0) {
      const index = {};
      const duplicated = queuedRequests.filter(req => {
        const iid = req.source._instanceid;
        if (!index[iid]) {
          index[iid] = req;
          return false;
        }
        return true;
      });

      duplicated.forEach(req => {
        const iid = req.source._instanceid;
        if (index[iid]) {
          index[iid].abort();
        }
      });
    }
    //siren: end

    const requests = requestQueue.getStartable(strategy);

    //siren: Adding $$sirenSingleCall = true to a member of a visualization allows to avoid all other requests
    //siren i.e. This is usefull for multi chart plugin
    const singleCallRequests = _.filter(requests, function (req) {
      return req && req.source && req.source.vis && req.source.vis.$$sirenSingleCall;
    });

    const nonSingleCallRequests = _.filter(requests, request => {
      if (!_.get(request, 'source.vis')) return true;

      return !request.source.vis.$$sirenSingleCall;
    });

    if (singleCallRequests.length > 0) {
      for (let i = 0; i < singleCallRequests.length; i++) {
        const req = singleCallRequests[i];
        req.source.vis.$$sirenSingleCall = false;
        if (i !== singleCallRequests.length - 1) {
          req.abort();
        } else {
          // once all the previous requests has been aborted,
          // push the multichart request back onto the other remaining requests
          nonSingleCallRequests.push(req);
        }
      }
    }
    //siren: end

    return this._fetchTheseSoon(nonSingleCallRequests);
  };

  function fetchASource(source) {
    const defer = Promise.defer();

    this._fetchTheseSoon([
      source._createRequest(defer)
    ]);

    return defer.promise;
  }

  /**
   * Fetch a single doc source
   * @param {DocSource} source - The DocSource to request
   * @async
   */
  this.doc = fetchASource;

  /**
   * Fetch a single search source
   * @param {SearchSource} source - The SearchSource to request
   * @async
   */
  this.search = fetchASource;

  /**
   * Fetch a list of requests
   * @param {array} reqs - the requests to fetch
   * @async
   */
  this.these = this._fetchTheseSoon;

  /**
   * Send responses to a list of requests, used when requests
   * should be skipped (like when a doc is updated with an index).
   *
   * This logic is a simplified version of what fetch_these does, and
   * could have been added elsewhere, but I would rather the logic be
   * here than outside the courier/fetch module.
   *
   * @param {array[Request]} requests - the list of requests to respond to
   * @param {array[any]} responses - the list of responses for each request
   */
  this.fakeFetchThese = function (requests, responses) {
    return Promise.map(requests, function (req) {
      return req.start();
    })
      .then(function () {
        return callResponseHandlers(requests, responses);
      })
      .then(function (requestStates) {
        if (_.contains(requestStates, INCOMPLETE)) {
          throw new Error('responding to requests did not complete!');
        }
      });
  };
}
