import _ from 'lodash';
import template from 'ui/filter_bar/filter_bar.html';
import 'ui/directives/json_input';
import '../filter_editor';
import { filterAppliedAndUnwrap } from 'ui/filter_bar/lib/filter_applied_and_unwrap';
import { FilterBarLibMapAndFlattenFiltersProvider } from 'ui/filter_bar/lib/map_and_flatten_filters';
import { FilterBarLibMapFlattenAndWrapFiltersProvider } from 'ui/filter_bar/lib/map_flatten_and_wrap_filters';
import {
  FilterBarLibExtractTimeFilterIfFromMainVisualizationProvider
} from 'ui/filter_bar/lib/extract_time_filter_if_from_main_visualization';
import { FilterBarLibChangeTimeFilterProvider } from 'ui/filter_bar/lib/change_time_filter';
import { FilterBarQueryFilterProvider } from 'ui/filter_bar/query_filter';
import { compareFilters } from './lib/compare_filters';
import { uiModules } from 'ui/modules';

export { disableFilter, enableFilter, toggleFilterDisabled } from './lib/disable_filter';


// kibi: imports
import 'ui/kibi/directives/kibi_entity_clipboard';
import 'ui/kibi/styles/explanation.less';
import { onDashboardPage } from 'ui/kibi/utils/on_page';
import { mergeFilters, isMultiSelectEnabled } from 'ui/filter_bar/lib/multi_filter'; // kibi: #4881 OR multiselection patch
import { getSearchPropertiesForFilters } from './get_search_prop_for_filters';
// kibi: end

const module = uiModules.get('kibana');

module.directive('filterBar', function ($rootScope, Private, Promise, getAppState,
  kibiState, config, createNotifier, joinExplanation, timefilter) {
  const mapAndFlattenFilters = Private(FilterBarLibMapAndFlattenFiltersProvider);
  const mapFlattenAndWrapFilters = Private(FilterBarLibMapFlattenAndWrapFiltersProvider);
  const extractTimeFilterIfFromMainVisualization = Private(FilterBarLibExtractTimeFilterIfFromMainVisualizationProvider);
  const changeTimeFilter = Private(FilterBarLibChangeTimeFilterProvider);
  const queryFilter = Private(FilterBarQueryFilterProvider);

  const notify = createNotifier({
    location: 'Siren Navigation Bar'
  });
  //: kibi: end

  return {
    template,
    restrict: 'E',
    scope: {
      indexPatterns: '=',
      coat: '=?'
    },
    link: function ($scope) {
      // bind query filter actions to the scope
      [
        'addFilters',
        'toggleFilter',
        'toggleAll',
        'pinFilter',
        'pinAll',
        'invertFilter',
        'invertAll',
        'removeFilter',
        'removeAll'
      ].forEach(function (method) {
        $scope[method] = queryFilter[method];
      });

      // kibi: used in listening timefilter fetch to detect first load
      let initializing = true;

      $scope.state = getAppState();

      $scope.newFilters = []; // kibi: #4881 OR multiselection patch

      // NOTE:
      // Watching this is the only reliable way of knowing that there are some
      // OR filters present, the watcher below correctly detects when the user pressed apply or cancel button
      // we use $rootScope.$emit as this event is meant to be caught only by dashboard controller
      $scope.$watch('newFilters', (filters) => {
        const orFiltersPresent = filters && filters.length > 0;
        $rootScope.$emit('filterbar:orFiltersPresent', orFiltersPresent);
      });

      $scope.showAddFilterButton = () => {
        return _.compact($scope.indexPatterns).length > 0;
      };

      $scope.applyFilters = function (filters) {
        addAndInvertFilters(mergeFilters(filterAppliedAndUnwrap(filters))); // kibi: #4881 OR multiselection patch
        $scope.newFilters = [];

        // change time filter
        if ($scope.changeTimeFilter && $scope.changeTimeFilter.meta && $scope.changeTimeFilter.meta.apply) {
          changeTimeFilter($scope.changeTimeFilter);
        }
      };

      // kibi: added newFilter parameter to spawn a filter editor with given data
      $scope.addFilter = (newFilter = true) => {
        $scope.editingFilter = {
          meta: { isNew: newFilter }
        };
      };
      // kibi: end

      $scope.deleteFilter = (filter) => {
        $scope.removeFilter(filter);
        if (filter === $scope.editingFilter) $scope.cancelEdit();
      };

      $scope.editFilter = (filter) => {
        $scope.editingFilter = filter;
      };

      $scope.cancelEdit = () => {
        delete $scope.editingFilter;
      };

      $scope.saveEdit = (filter, newFilter, isPinned) => {
        if (!filter.isNew) $scope.removeFilter(filter);
        delete $scope.editingFilter;
        $scope.addFilters([newFilter], isPinned);
      };

      $scope.clearFilterBar = function () {
        $scope.newFilters = [];
        $scope.changeTimeFilter = null;
      };

      // update the scope filter list on filter changes
      $scope.$listen(queryFilter, 'update', function () {
        updateFilters();
      });

      // when appState changes, update scope's state
      $scope.$watch(getAppState, function (appState) {
        $scope.state = appState;
      });

      // kibi: Added custom event to open a new filter with given filter data
      $scope.$on('NewFilterEditor', (e, filter) => $scope.addFilter(filter));

      /**
       * kibi: replace '$COUNT' to '...' in filters
       */
      const removeCountFromFilterMetaAlias = function () {
        _.each($scope.state.filters, (filter) => {
          if (filter.join_sequence && filter.meta.alias_tmpl) {
            const base = filter.meta.alias.replace(/[0-9]+/, '');
            // make sure the alias was unchanged
            if (filter.meta.alias_tmpl.replace('$COUNT', '') === base) {
              filter.meta.alias = filter.meta.alias_tmpl.replace('$COUNT', '...');
              delete filter.meta.alias_tmpl;
            }
          }
        });
      };

      /**
       * kibi: If dashboard time is changed then the count from the join_sequence filter alias is removed.
       * If left alone, the count would be misleading.
       */
      $scope.$listen(timefilter, 'fetch', () => {
        if (initializing) {
          initializing = false;
        } else {
          removeCountFromFilterMetaAlias();
        }
      });

      /**
       * kibi: Watch the filters and if anything is added/removed then the count from the join_sequence filter alias is removed.
       * If left alone, the count would be misleading.
       */
      $scope.$watchCollection('state.filters', function (filters, oldFilters) {
        if (!filters || filters.length === (oldFilters && oldFilters.length || 0)) {
          return;
        }
        removeCountFromFilterMetaAlias();
      });

      $scope.$watch('state.$newFilters', function (filters) {
        // kibi: #4881 OR multiselection patch - Start
        if (!filters || filters.length === 0) return;
        const dashboard = kibiState.getDashboardOnView();
        let coatItems;
        // there is no dashboard when we are in vis editor
        if (dashboard && _.get(dashboard, 'siren.coat.items')) {
          coatItems = dashboard.siren.coat.items;
        }

        // Allow the users to decide what filters to apply when
        // - more than one new filter getting added
        // - new filters exist that have not been applied
        // - multiselect is enabled
        if (filters.length > 1 || $scope.newFilters.length > 0 || isMultiSelectEnabled()) {
          // kibi: #4881 OR multiselection patch - End
          return mapFlattenAndWrapFilters(filters)
            .then(function (results) {
              // Note:
              // check if it is a timefilter and if it is
              // check if it was generated by main or linked visualization
              // if by main one - modify a global timefilter
              // if not just add the filter to the state
              return extractTimeFilterIfFromMainVisualization(results, coatItems)
                .then(res => {
                  // if extrated
                  if (res.timeFilters) {
                    $scope.changeTimeFilter = res.timeFilters[0];
                    if (res.timeFilters.length > 1 && console) {
                      console.log('More than one time filter detected.'); // eslint-disable-line no-console
                      console.log(res.timeFilters); // eslint-disable-line no-console
                      console.log('Only first filter was used to modify the global timefiler'); // eslint-disable-line no-console
                    }
                  }
                  $scope.newFilters = $scope.newFilters.concat(res.notTimeFilters); // kibi: #4881 OR multiselection patch
                });
            });
        }

        // Just add single filters to the state.
        if (filters.length === 1) {
          // Note:
          // check if it is a timefilter and if it is
          // check if it was generated by main or linked visualization
          // if by main one - modify a global timefilter
          // if not just add the filter to the state
          return extractTimeFilterIfFromMainVisualization(filters, coatItems)
            .then(res => {
              if (res.timeFilters) {
                changeTimeFilter(res.timeFilters[0]); // here it is safe to assume only one filter
              }
              addAndInvertFilters(res.notTimeFilters);
            });
        }
      });

      function addAndInvertFilters(filters) {
        const existingFilters = queryFilter.getFilters();
        const inversionFilters = _.filter(existingFilters, (existingFilter) => {
          const newMatchingFilter = _.find(filters, _.partial(compareFilters, existingFilter));
          return newMatchingFilter
            && newMatchingFilter.meta
            && existingFilter.meta
            && existingFilter.meta.negate !== newMatchingFilter.meta.negate;
        });
        const newFilters = _.reject(filters, (filter) => {
          return _.find(inversionFilters, _.partial(compareFilters, filter));
        });

        _.forEach(inversionFilters, $scope.invertFilter);
        $scope.addFilters(newFilters);
      }

      function updateFilters() {
        const filters = queryFilter.getFilters();

        mapAndFlattenFilters(filters).then(function (results) {
          // used to display the current filters in the state
          $scope.filters = _.sortBy(results, function (filter) {
            return !filter.meta.pinned;
          });
        })
          .then(function () {
            $scope.searchPropertiesForFilters = getSearchPropertiesForFilters($scope.filters, $scope.coat);
          })
        // kibi: join filter explanation
          .then(function () {
            return joinExplanation.getFilterExplanations(filters);
          })
          .then(function (explanations) {
            return joinExplanation.initQtip(explanations);
          })
          .then(() => {
            $scope.$emit('filterbar:updated');
          });
      }

      updateFilters();

      // kibi: needed to recreate filter label.
      // as we do not want to store the meta info in filter join definition
      // we have to recreate it.
      // it should support the following filters:
      // .query
      // .dbfilter
      // .geo_bounding_box
      // .range
      // .not
      // .or
      // .exists
      // .missing
      // .script
      $scope.recreateFilterLabel = joinExplanation.createFilterLabel;


      // kibi: needed to show filterbar when kibiEntityClipboard contains an entity
      $scope.showKibiEntityClipboard = onDashboardPage() && Boolean(kibiState.getEntityURI());

      // kibi: listen to changes to the kibiState
      $scope.$listen(kibiState, 'save_with_changes', (diff) => {
        const currentDashboard = kibiState.getDashboardOnView();

        if (!currentDashboard) {
          $scope.showKibiEntityClipboard = false;
          return;
        }

        const promise = Promise.resolve();

        // the selected entity changed
        if (diff.indexOf(kibiState._properties.selected_entity) !== -1 ||
            diff.indexOf(kibiState._properties.selected_entity_disabled) !== -1) {
          $scope.showKibiEntityClipboard = Boolean(kibiState.getEntityURI());
          promise.then(() => updateFilters.call(this)).catch(notify.error);
        }
      });
    }
  };
});
