// kibi: #4881 OR multiselection patch - Start
import _ from 'lodash';
import { filterHelper } from 'ui/kibi/components/dashboards360/filter_helper';

/**
 * Create single filter per data field. Fields with multiple filters are merged into a bool query.
 * Filters without meta.key can not be merged because field is unknown.
 *
 * @param {array} filters to merge.
 * @returns {array} merged filters
 */
export function mergeFilters(filters) {

  // Group filters by key
  const filterGroups = new Map();
  const filtersWithNoKey = [];
  filters.forEach(f => {

    // first make sure that we translate all old deprecated filters which could be created by not up to date plugins etc
    const filter = filterHelper.translateToQuery(f);

    const key = _.get(filter, 'meta.key');
    if (!key) {
      filtersWithNoKey.push(filter);
      return;
    }

    if (!filterGroups.has(key)) {
      filterGroups.set(key, [filter]);
    } else {
      filterGroups.set(key, filterGroups.get(key).concat(filter));
    }
  });

  const filtersWithKey = [];
  filterGroups.forEach((filterGroup, key) => {
    if (filterGroup.length === 1) {
      // Key with only one filter
      filtersWithKey.push(filterGroup[0]);
    } else {
      // Key with multiple filters - combine into OR query
      const values = [];
      let shouldFilters = [];
      const mustNotFilters = [];
      let _siren;
      filterGroup.forEach((filter) => {
        // grab _siren meta from first filter in a group
        if (!_siren && filter.meta._siren) {
          _siren = filter.meta._siren;
        }
        const value = _.get(filter, 'meta.value', '');
        if (_.get(filter, 'meta.negate', false)) {
          values.push('!' + value);
          mustNotFilters.push(cleanFilter(filter));
        } else {
          values.push(value);
          shouldFilters.push(cleanFilter(filter));
        }
      });
      if (mustNotFilters.length >= 1) {
        shouldFilters = shouldFilters.concat({
          bool: {
            must_not: mustNotFilters
          }
        });
      }
      const orFilter = {
        bool: {
          should: shouldFilters
        },
        meta: {
          alias: `${key}: ${values.join()}`,

        }
      };
      if (_siren) {
        orFilter.meta._siren = _siren;
      }
      filtersWithKey.push(orFilter);
    }
  });

  return filtersWithKey.concat(filtersWithNoKey);
}

function cleanFilter(filter) {
  if (_.has(filter, 'query')) {
    return filter.query;
  }

  delete filter.meta;
  return filter;
}

const CTRL_KEY = 17;

// OSX Chrome and Safari the left and right command keys have different keyCode property
const LEFT_MACOS_COMMAND_KEY_CODE = 91;
const RIGHT_MACOS_COMMAND_KEY_CODE = 93;

// OSX Firefox the left and right command keys have same keyCode but different code
const FIREFOX_OS_RIGHT_CODE  = 'OSRight';
const FIREFOX_OS_LEFT_CODE  = 'OSLeft';
const FIREFOX_MACOS_COMMAND_KEY_CODE = 224;

function isCtrlKey(event) {
  return event.keyCode === CTRL_KEY ||
    // OSX Chrome and Safari
    event.keyCode === LEFT_MACOS_COMMAND_KEY_CODE || event.keyCode === RIGHT_MACOS_COMMAND_KEY_CODE ||
    // OSX Firefox
    ((event.code === FIREFOX_OS_RIGHT_CODE || event.code === FIREFOX_OS_LEFT_CODE) && event.keyCode === FIREFOX_MACOS_COMMAND_KEY_CODE);
}

let isMultiSelect = false;

window.addEventListener('keydown', (event) => {
  if (isCtrlKey(event)) {
    isMultiSelect = true;
  }
}, false);

window.addEventListener('keyup', (event) => {
  if (isCtrlKey(event)) {
    isMultiSelect = false;
  }
}, false);

window.addEventListener('blur', (event) => {
  isMultiSelect = false;
}, false);

/**
 * Use this function to know when multi-select functionallity is enabled.
 *
 * @returns {boolean} Is multi-select functionallity enabled?
 */
export function isMultiSelectEnabled() {
  return isMultiSelect;
}
// kibi: #4881 OR multiselection patch - End
