import _ from 'lodash';
import { uiModules } from 'ui/modules';
import { FILTER_OPERATOR_TYPES } from './lib/filter_operators';
import template from './filter_editor.html';
import { documentationLinks } from '../documentation_links/documentation_links';
import './filter_query_dsl_editor';
import './filter_field_select';
import './filter_operator_select';
import './params_editor/filter_params_editor';
import './filter_editor.less';
import {
  getQueryDslFromFilter,
  getFieldFromFilter,
  getOperatorFromFilter,
  getParamsFromFilter,
  isFilterValid,
  buildFilter
} from './lib/filter_editor_utils';
import * as filterBuilder from '../filter_manager/lib';
import { keyMap } from '../utils/key_map';

const module = uiModules.get('kibana');
module.directive('filterEditor', function ($timeout, indexPatterns, $document) {
  return {
    restrict: 'E',
    template,
    scope: {
      indexPatterns: '=',
      filter: '=',
      onDelete: '&',
      onCancel: '&',
      onSave: '&'
    },
    controllerAs: 'filterEditor',
    bindToController: true,
    controller: function ($scope, $element) {
      this.init = () => {
        const { filter } = this;
        this.docLinks = documentationLinks;
        this.alias = filter.meta.alias;
        this.isEditingQueryDsl = false;
        this.queryDsl = getQueryDslFromFilter(filter);
        if (filter.meta.isNew) {
          // kibi: Reading new filter requests with some data already filled in
          const isNew = filter.meta.isNew;
          let focus = 'field';

          if (isNew.field) {
            this.setField(isNew.field);
            focus = 'operator';
          }
          if (isNew.operator) {
            this.setOperator(isNew.operator);
            focus = 'params';
          }
          if (isNew.params) {
            this.setParams(isNew.params);
            focus = '';
          }

          if (focus) { this.setFocus(focus); }
          // kibi: end
        } else {
          getFieldFromFilter(filter, indexPatterns)
            .then((field) => {
              this.setField(field);
              this.setOperator(getOperatorFromFilter(filter));
              this.params = getParamsFromFilter(filter);
            });
        }
      };

      $scope.$watch(() => this.filter, this.init);
      $scope.$watchCollection(() => this.filter.meta, this.init);

      this.setQueryDsl = (queryDsl) => {
        this.queryDsl = queryDsl;
      };

      this.setField = (field) => {
        this.field = field;
        this.operator = null;
        this.params = {};
      };

      this.onFieldSelect = (field) => {
        this.setField(field);
        this.setFocus('operator');
      };

      this.setOperator = (operator) => {
        this.operator = operator;
      };

      this.onOperatorSelect = (operator) => {
        this.setOperator(operator);
        this.setFocus('params');
      };

      this.setParams = (params) => {
        this.params = params;
      };

      this.setFocus = (name) => {
        $timeout(() => $scope.$broadcast(`focus-${name}`));
      };

      this.showQueryDslEditor = () => {
        const { type, isNew } = this.filter.meta;
        return this.isEditingQueryDsl || (!isNew && !FILTER_OPERATOR_TYPES.includes(type));
      };

      this.isValid = () => {
        if (this.showQueryDslEditor()) {
          return _.isObject(this.queryDsl);
        }
        const { field, operator, params } = this;
        // kibi: added to disable/enable 'Save' button after validation
        if (this.field && this.field.type === 'date' && (params.phrase || params.range)) {
          return isFilterValid({ field, operator, params }) && this.isTimeFilterValid;
        };
        // kibi: end
        return isFilterValid({ field, operator, params });
      };

      this.save = () => {
        const { filter, field, operator, params, alias } = this;

        let newFilter;
        if (this.showQueryDslEditor()) {
          const meta = _.pick(filter.meta, ['negate', 'index']);
          meta.index = meta.index || this.indexPatterns[0].id;
          newFilter = Object.assign(this.queryDsl, { meta });
        } else {
          const indexPattern = field.indexPattern;
          newFilter = buildFilter({ indexPattern, field, operator, params, filterBuilder });
        }
        newFilter.meta.disabled = filter.meta.disabled;
        newFilter.meta.alias = alias;
        if (filter.meta.version) {
          newFilter.meta.version = filter.meta.version;
        }

        // kibi: preserve _siren meta
        if (filter.meta && filter.meta._siren) {
          newFilter.meta._siren = filter.meta._siren;
        }
        // kibi: end

        const isPinned = _.get(filter, ['$state', 'store']) === 'globalState';
        return this.onSave({ filter, newFilter, isPinned });
      };

      // kibi: added to make 'Add filter' window resizable
      function resetEditorStyles(element) {
        element.removeAttr('was-dragged');
        element.find('.filterEditor').removeAttr('style');
        element.closest().find('.filter-edit-container').removeAttr('style');
        element.find('.ace_editor').removeAttr('style');
        element.find('#editorControl').removeAttr('style');
        element.find('#labelInput').removeAttr('style');
      };

      function updateEditorStyles(element, event, height, width) {
        const editorControl = element.find('#editorControl');
        const filterEditContainer = element.closest('.filter-edit-container');
        const filterEditor = element.find('.filterEditor');
        const aceEditor = element.find('.ace_editor');
        const labelInput = element.find('#labelInput');
        const editorPosition = aceEditor[0].getBoundingClientRect();
        const editorHeight = event.originalEvent.pageY - editorPosition.top - pageYOffset
          - 2 * editorControl[0].offsetHeight - labelInput[0].offsetHeight;
        editorControl
          .css('position', 'absolute')
          .css('right', '0px')
          .css('bottom', '0px');
        filterEditContainer
          .css('position', 'absolute')
          .css('height', event.originalEvent.pageY - 50 + 'px')
          .css('width', event.originalEvent.pageX - 50 + 'px')
          .css('cursor', 'se-resize');
        filterEditor
          .css('height', height + 'px')
          .css('width', width + 'px');
        aceEditor
          .css('height', editorHeight + 'px');
      };

      this.switchFilters = () => {
        if ($element.attr('was-dragged')) {
          resetEditorStyles($element);
        };
        this.isEditingQueryDsl = !this.isEditingQueryDsl;
      };

      $element.find('.filterEditor__expand').on('mousedown.activateResize', (event) => {
        if (event.which !== 1) {
          return;
        };
        const filterEditor = $element.find('.filterEditor');
        $document.on('mousemove.resizeEditor', (event) => {
          event.preventDefault();
          const elementPosition = filterEditor[0].getBoundingClientRect();
          const newEditorHeight = event.originalEvent.clientY - elementPosition.top;
          const newEditorWidth = event.originalEvent.clientX - elementPosition.left;
          const minEditorHeight = 254;
          const minEditorWidth = 600;
          if (
            newEditorHeight < minEditorHeight
            || newEditorWidth < minEditorWidth
            || newEditorWidth > document.body.clientWidth - elementPosition.left
          ) {
            return;
          };
          $element.attr('was-dragged', true);
          updateEditorStyles($element, event, newEditorHeight, newEditorWidth);
        });
        $document.on('mouseup.finishResize', (event) => {
          $document.off('.resizeEditor');
          $element.closest('.filter-edit-container').removeAttr('style');
        });
      });

      const keydownHandler = (event) => {
        if (keyMap[event.keyCode] === 'escape') {
          $timeout(() => this.onCancel());
        }
      };

      $element.on('keydown', keydownHandler);

      $scope.$on('$destroy', () => {
        $element.off('keydown', keydownHandler);
        $element.find('.filterEditor__expand').off('.activateResize');
        $document.off('.finishResize');
      });
      // kibi: end

      // kibi: added to disable/able button after validation
      $scope.$on('siren:filterEditor:checkValidation', (event, validation) => {
        this.isTimeFilterValid = validation;
      });
      // kibi: end
    }
  };
})
  .filter('prettyJSON', () => json => JSON.stringify(json, null, ' '));
