import _ from 'lodash';
import moment from 'moment-timezone'; // kibi: added

// Takes a hit, merges it with any stored/scripted fields, and with the metaFields
// returns a flattened version
export function IndexPatternsFlattenHitProvider(config, elasticsearchVersion) { // kibi: added elasticsearchVersion
  let metaFields = config.get('metaFields');

  config.watch('metaFields', value => {
    metaFields = value;
  });

  function flattenHit(indexPattern, hit, deep) {
    const flat = {};

    // recursively merge _source
    const fields = indexPattern.fields.byName;
    (function flatten(obj, keyPrefix) {
      keyPrefix = keyPrefix ? keyPrefix + '.' : '';
      _.forOwn(obj, function (val, key) {
        key = keyPrefix + key;

        if (deep) {
          const isNestedField = fields[key] && fields[key].type === 'nested';
          const isArrayOfObjects = _.isArray(val) && _.isPlainObject(_.first(val));
          if (isArrayOfObjects && !isNestedField) {
            _.each(val, v => flatten(v, key));
            return;
          }
        } else if (flat[key] !== void 0) {
          return;
        }

        const hasValidMapping = fields[key] && fields[key].type !== 'conflict';
        const isValue = !_.isPlainObject(val);

        if (hasValidMapping || isValue) {
          if (!flat[key]) {
            flat[key] = val;
          } else if (_.isArray(flat[key])) {
            flat[key].push(val);
          } else {
            flat[key] = [ flat[key], val ];
          }
          return;
        }

        flatten(val, key);
      });
    }(hit._source));

    // assign the meta fields
    _.each(metaFields, function (meta) {
      if (meta === '_source') return;
      flat[meta] = hit[meta];
    });

    // unwrap computed fields
    _.forOwn(hit.fields, function (val, key) {
      if (key[0] === '_' && !_.contains(metaFields, key)) return;
      // kibi: if Elasticsearch version 6, convert formatted timestamps in fields to ms since epoch.
      // If not, we assume that since the fields uses the formatting specified in the mapping filters will work without
      // having to be converted to milliseconds explicitly.
      // See https://github.com/elastic/elasticsearch/pull/29639/commits/b25ccfc1ae2e245ccf020ef773e14181003645ac .
      let hitValue = _.isArray(val) && val.length === 1 ? val[0] : val;
      if (fields.hasOwnProperty(key) && fields[key].type === 'date' && _.isString(hitValue) && hitValue.indexOf('Z') === hitValue.length - 1
          && elasticsearchVersion.getMajor() === 6) {
        const m = moment.tz(hitValue, 'YYYY-MM-DDTHH:mm:ss.SSSZ', true, 'UTC');
        if (m.isValid()) {
          hitValue = m.valueOf();
        }
      }
      flat[key] = hitValue;
      // kibi: end
    });

    return flat;
  }

  return function flattenHitWrapper(indexPattern) {
    return function cachedFlatten(hit, deep = false) {
      return hit.$$_flattened || (hit.$$_flattened = flattenHit(indexPattern, hit, deep));
    };
  };
}
