import { EntityType } from 'ui/kibi/components/ontology/entity_type';
import { JoinAlgorithmType } from './join_algorithm_type';
import { getMaxDepth } from './get_max_depth';
import { computeJoinFilter } from './compute_join_filter';
import { findItemById } from './find_item_by_id';

function getSearchNodesFromLevel(treeItems, level) {
  return treeItems
    .filter(item => item.type === 'node' && item.depth === level && item.d.entity.type === EntityType.SAVED_SEARCH);
}

function getIncomingLinks(treeItems, nodeId) {
  return treeItems
    .filter(item => item.type === 'link' && item.id2 === nodeId);
}

function containsFilterFromLinkedVis(filters) {
  if (!filters || filters.length === 0 || !(filters instanceof Array)) {
    return false;
  }
  for (let i = 0; i < filters.length; i++) {
    if (
      filters[i].meta && filters[i].meta._siren && filters[i].meta._siren.vis &&
      filters[i].meta._siren.vis.id && filters[i].meta._siren.vis.panelIndex
    ) {
      return true;
    }
  }
  return false;
}

function areIndicesTheSame(indicesA, indicesB) {
  if (indicesA.length !== indicesB.length) {
    return false;
  }
  const sortedA = indicesA.slice(0).sort();
  const sortedB = indicesB.slice(0).sort();
  for (let i = 0; i < sortedA.length; i++) {
    if (sortedA[i] !== sortedB[i]) {
      return false;
    }
  }
  return true;
}

function shouldComputeJoinForFilterBasedAlgorithm(sourceNode) {
  const isMainSource = sourceNode.d.isRoot === true;
  const containsAnyFilterFromLinkedVis = containsFilterFromLinkedVis(sourceNode.d.entity.filters);
  const haveJoinFilterComputedInPreviousStep = sourceNode.d.computed === true;
  return isMainSource || containsAnyFilterFromLinkedVis || haveJoinFilterComputedInPreviousStep;
}

// Traverse the focused tree backwords level by level
// Below the tree is focused on node a
// we start not from the deepest level but from one above
// e.g. in below tree the deepest level is 2 (nodes c and d)
// so we start processing from level 1 and compute filters for nodes b and b1
// then we move one level up and compute filter for node a
//
//     a*
//    / \
//   b   b1  <-- we start on this level and go up
//  / \
// c   d

function computeFiltersFull(focusedTree) {
  const maxDepth = getMaxDepth(focusedTree);
  let level = maxDepth - 1;


  while (level >= 0) {
    // process all search nodes on level
    // compute filter or filters for each node on level n
    const targetNodes = getSearchNodesFromLevel(focusedTree, level);
    targetNodes.forEach(targetNode => {
      if (!targetNode.d.entity.filters) {
        targetNode.d.entity.filters = [];
      }

      // compute a filter for this node
      // e.g lets asume we are on node b
      //
      //   b
      //  / \
      // c   d
      //
      // the result filters on "a" should contain to join filters join(c->b) AND join(d, b)

      // first find all links pointing to node
      const incomingLinks = getIncomingLinks(focusedTree, targetNode.id);
      incomingLinks.forEach(link => {
        const targetIndices = targetNode.d.entity.indices;
        const targetField = link.d.rel.range.field;

        // check if source is a search or eid
        const sourceNode = findItemById(focusedTree, link.id1);
        if (sourceNode.d.entity.type === EntityType.SAVED_SEARCH) {

          const sourceField = link.d.rel.domain.field;
          const sourceIndices = sourceNode.d.entity.indices;
          const sourceFilters = sourceNode.d.entity.filters || [];

          const joinFilter = computeJoinFilter(sourceField, sourceIndices, sourceFilters, targetField, targetIndices);
          targetNode.d.entity.filters.push(joinFilter);
        } else if (sourceNode.d.entity.type === EntityType.VIRTUAL_ENTITY) {
          // here do the same just target is one more hop away
          const incomingEIDLinks = getIncomingLinks(focusedTree, sourceNode.id);
          if (incomingEIDLinks.length === 0) {
            return;
          }

          incomingEIDLinks.forEach(incomingEIDLink => {
            const sourceEIDNode = findItemById(focusedTree, incomingEIDLink.id1);

            if (sourceEIDNode.d.entity.type === EntityType.SAVED_SEARCH) {
              const sourceField = incomingEIDLink.d.rel.domain.field;
              const sourceIndices = sourceEIDNode.d.entity.indices;
              const sourceFilters = sourceEIDNode.d.entity.filters || [];
              const joinFilter = computeJoinFilter(sourceField, sourceIndices, sourceFilters, targetField, targetIndices);
              targetNode.d.entity.filters.push(joinFilter);
            } else {
              throw new Error('Expected a search node after eid node');
            }
          });
        } else {
          throw new Error('Unexpected type while traversing the focused tree');
        }
      });
    });
    level--;
  }
}

// Traverse the focused tree backwords level by level
// Below the tree is focused on node a
// we start not from the deepest level but from one above
// e.g. in below tree the deepest level is 2 (nodes c and d)
// so we start processing from level 1 and compute filters for nodes b and b1
// then we move one level up and compute filter for node a
//
//     a*
//    / \
//   b   b1  <-- we start on this level and go up
//  / \
// c   d

function computeFiltersFilterBased(focusedTree) {
  const maxDepth = getMaxDepth(focusedTree);
  let level = maxDepth - 1;


  while (level >= 0) {
    // process all nodes on level
    // compute filter or filters for each node on level n
    const targetNodes = getSearchNodesFromLevel(focusedTree, level);
    targetNodes.forEach(targetNode => {
      if (!targetNode.d.entity.filters) {
        targetNode.d.entity.filters = [];
      }
      // compute a filter for this node
      // e.g lets asume we are on node b
      //
      //   b
      //  / \
      // c   d
      //
      // the result filters on "a" should contain to join filters join(c->b) AND join(d, b)

      // first find all links pointing to node
      const incomingLinks = getIncomingLinks(focusedTree, targetNode.id);
      incomingLinks.forEach(link => {
        const targetIndices = targetNode.d.entity.indices;
        const targetField = link.d.rel.range.field;

        // check if source is a search or eid
        const sourceNode = findItemById(focusedTree, link.id1);
        if (sourceNode.d.entity.type === EntityType.SAVED_SEARCH) {
          if (!shouldComputeJoinForFilterBasedAlgorithm(sourceNode)) {
            return;
          }
          const sourceField = link.d.rel.domain.field;
          const sourceIndices = sourceNode.d.entity.indices;
          const sourceFilters = sourceNode.d.entity.filters || [];

          const joinFilter = computeJoinFilter(sourceField, sourceIndices, sourceFilters, targetField, targetIndices);
          targetNode.d.entity.filters.push(joinFilter);
          targetNode.d.computed = true;

        } else if (sourceNode.d.entity.type === EntityType.VIRTUAL_ENTITY) {
          // here do the same just target is one more hop away
          const incomingEIDLinks = getIncomingLinks(focusedTree, sourceNode.id);
          if (incomingEIDLinks.length === 0) {
            return;
          }
          incomingEIDLinks.forEach(incomingEIDLink => {
            const sourceEIDNode = findItemById(focusedTree, incomingEIDLink.id1);

            if (sourceEIDNode.d.entity.type === EntityType.SAVED_SEARCH) {
              if (!shouldComputeJoinForFilterBasedAlgorithm(sourceEIDNode)) {
                return;
              }

              const sourceField = incomingEIDLink.d.rel.domain.field;
              const sourceIndices = sourceEIDNode.d.entity.indices;
              const sourceFilters = sourceEIDNode.d.entity.filters || [];
              const joinFilter = computeJoinFilter(sourceField, sourceIndices, sourceFilters, targetField, targetIndices);
              targetNode.d.entity.filters.push(joinFilter);
              targetNode.d.computed = true;
            } else {
              throw new Error('Expected a search node after eid node');
            }
          });
        } else {
          throw new Error('Unexpected type while traversing the focused tree');
        }
      });
    });
    level--;
  }
}

export function computeFilters(focusedTree, joinAlgorithmType) {
  if (joinAlgorithmType === JoinAlgorithmType.INNER_FILTER_JOIN) {
    computeFiltersFull(focusedTree);
  } else if (joinAlgorithmType === JoinAlgorithmType.LEFT_FILTER_JOIN) {
    computeFiltersFilterBased(focusedTree);
  } else {
    throw new Error(`Not supported JoinAlgorithmType: ${joinAlgorithmType} use one of: ${Object.keys(JoinAlgorithmType)}`);
  }
}

