import _ from 'lodash';
import { uiModules } from 'ui/modules';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';

uiModules
  .get('kibana/ontology_model')
  .service('ontologyModelHelper', function (Private, queryEngineClient, createNotifier, ontologyModel, savedDashboards, savedSearches) {
    const notify = createNotifier({
      location: 'Kibi Ontology Model Helper'
    });

    function _findRootEntityId(entityId, entities) {
      const entity = _.find(entities, 'id', entityId);
      return entity.parentId ? _findRootEntityId(entity.parentId, entities) : entityId;
    }

    function _addIndexPatternTo(relationEntity, savedSearches, entities) {
      if (relationEntity.type === EntityType.VIRTUAL_ENTITY) {
        relationEntity.indexPatternId = relationEntity.id;
      } else if (relationEntity.type === EntityType.SAVED_SEARCH) {
        // find root entity
        const rootEntityId = _findRootEntityId(relationEntity.id, entities);
        const rootSavedSearch = _.find(savedSearches, 'id', rootEntityId);

        if (rootSavedSearch) {
          relationEntity.indexPatternId = JSON.parse(rootSavedSearch.kibanaSavedObjectMeta.searchSourceJSON).index;
        }
      }
    }

    // introduced to break circular dependency
    class OntologyModelHelper {

      /*
       * Returns a list of dashboards that can be the target of a join for the given entity id.
       */
      getDashboardsByEntity(entity) {
        return ontologyModel.getRangesForEntityId(entity.id)
          .then((ranges) => {
            const indexPatterns = new Set(_.map(ranges, 'indexPattern'));
            return savedDashboards.find()
              .then(function (dashboards) {
                const promises = _.map(dashboards.hits, (dashboard) => {
                  const coat = JSON.parse(dashboard.coatJSON);
                  const rootNode = _.find(coat.items, item => item.d.isRoot === true);
                  const savedSearchId = rootNode ? rootNode.d.entity.id : null;
                  return savedSearches.get(savedSearchId)
                    .then((savedSearch) => {
                      if (savedSearch.kibanaSavedObjectMeta && savedSearch.kibanaSavedObjectMeta.searchSourceJSON) {
                        const index = JSON.parse(savedSearch.kibanaSavedObjectMeta.searchSourceJSON).index;
                        if (indexPatterns.has(index)) {
                          return dashboard;
                        }
                      }
                    });
                });
                return Promise.all(promises)
                  .then(dashboards => {
                    return _.filter(dashboards, (dashboard) => {
                      return !!dashboard;
                    });
                  });
              });
          });
      };

      addIndexPatternsToRelations(relations) {
        return Promise.all([ savedSearches.find(), ontologyModel.getEntityList() ])
          .then(([ savedSearches, entities ]) => {
            _.each(relations, rel => {
              _addIndexPatternTo(rel.domain, savedSearches.hits, entities);
              _addIndexPatternTo(rel.range, savedSearches.hits, entities);
            });
            return relations;
          });
      }

    }

    return new OntologyModelHelper();
  });
