import { uiModules } from 'ui/modules';
import Promise from 'bluebird';
import handleMissingIndexPattern from 'ui/kibi/index_patterns/handle_missing_index_pattern';
import _ from 'lodash';

uiModules
  .get('kibana')
  .factory('mappings', (es, config) => {

    const promisesCache = new Map();
    let excludedIndicesSuffix;

    function initExcludedIndicesSuffix() {
      if (!excludedIndicesSuffix) {
        excludedIndicesSuffix = ',-' + config.get('siren:excludedIndices').split(',').join(',-');
      }
    }

    class Mappings {

      constructor() {
      }

      // Takes an actual indexPattern (the one usually stored in the title)
      getMapping(indexPattern) {
        initExcludedIndicesSuffix();

        if (!promisesCache.has(indexPattern)) {
          const indicesArray = indexPattern.split(',');
          const indices = _.reduce(indicesArray, (total, index) => {
            if (index.startsWith('-')) {
              total.excluded.push(index);
            } else {
              total.included.push(index);
            }
            return total;
          }, { included: [], excluded: [] });
          const excludedSuffix = excludedIndicesSuffix + (indices.excluded.length ? ',' + indices.excluded.join(',') : '');

          const promise = Promise.map(indices.included, index => {
            return es.indices.getMapping({
              index: index + excludedSuffix,
              // Set to true to not return an error if one of the excluded indices is not available
              ignoreUnavailable: true,
              allowNoIndices: true
            })
              .catch(err => handleMissingIndexPattern(err, indexPattern, {}));
          })
            .then(responses => {
              return _.reduce(
                responses,
                (total, resp) => _.merge(total, resp),
                {}
              );
            });

          promisesCache.set(indexPattern, promise);
        }

        return promisesCache.get(indexPattern);
      }

      clearCache() {
        promisesCache.clear();
        excludedIndicesSuffix = undefined;
      }
    }

    return new Mappings();
  });
