import findRelationsReportTemplate from './findrelations_report.html';
import findRelationsErrorTemplate from './findrelations_error.html';

import { RelationsHelperProvider } from './relations_helper';
import { BaseModalProvider } from 'ui/kibi/modals/base_modal';

import './relation_explorer';

import './quickrelations_modal.less';
import './layout.less';

import { allSelected } from 'ui/kibi/directives/tristate_checkbox';
import { sortContext, sortSequence } from 'ui/kibi/directives/sort_icon';

import { promiseMapSeries } from 'ui/kibi/utils/promise';
import { EntityType } from 'ui/kibi/components/ontology/entity_type';

import _ from 'lodash';
import $ from 'jquery';


export function ReportProvider(Private, $timeout) {
  const baseModal = Private(BaseModalProvider);

  const {
    pairText, pairHash, ssearchIndexPattern, hydrateLink
  } = Private(RelationsHelperProvider);

  function toHydratedEidEndpoint(ep) {
    return {
      eid: { name: ep.eid },
      get text() { return this.eid.name; },
      get hash() { return this.eid.name; },
      selected: true
    };
  }

  function toHydratedFieldEndpoint(fieldData, link) {
    const field = fieldData.orig;

    return {
      field,
      data: fieldData,
      text: fieldData.text,
      hash: fieldData.id,
      selected: link.selected
    };
  }

  function toHydratedEndpoint(ep, link) {
    return ep.eid
      ? toHydratedEidEndpoint(ep, link) : toHydratedFieldEndpoint(ep, link);
  }

  function linksToConnections(links) {
    return _(links)
      .map(link => hydrateLink(link, toHydratedEndpoint))
      .value();
  }

  function linksToRelationGroups(links) {
    return _(links)
      .map(link => hydrateLink(link, toHydratedEndpoint))
      .groupBy('target.hash')
      .map(groupLinks => {
        const target = groupLinks[0].target;
        target.selected = _.some(groupLinks, 'selected');

        const sources = _.map(groupLinks, 'source');
        const endpoints = [ target ].concat(sources);

        return { target, sources, endpoints };
      })
      .value();
  }

  function relationGroupsToConnections(relationGroups) {
    return _(relationGroups)
      .map(group => _.map(group.sources, src => [group.target, src]))
      .flatten()
      // Connections could be duplicated if the user changes targets - filter duplicates
      .indexBy(pair => _(pair).map('hash').sortBy().map(pairHash).value())
      .map(pair => ({ target: pair[0], source: pair[1] }))
      .value();
  }

  function show(args) {
    function toTypeField(field) {
      const result = _.clone(field);
      result.name = '';

      return result;
    }

    function targetText(dst) {
      return dst.eid ? '[EID] ' + dst.text : dst.text;
    }

    function targetLabel(dst) {
      let result = targetText(dst);
      if (dst === this.endpoints[0]) { result = '* ' + result; }

      return result;
    }

    function toReportGroup(relationGroup) {
      const { endpoints, target, sources } = relationGroup;

      const typeField = toTypeField(sources[0].field);
      const fieldEndpoints = _.filter(endpoints, 'field');

      const targetCandidates = [ target ].concat(_.sortBy(sources));
      if (!target.eid) {
        targetCandidates.splice(1, 0, toHydratedEidEndpoint({ eid: '<EID>' }));
      }

      const result = {
        endpoints, target, sources,
        typeField, fieldEndpoints, targetCandidates,
        targetExpanded: false,

        targetText,
        targetLabel
      };

      result.allSelected = allSelected(() => result.sources);
      return result;
    }

    function toExplorerEndpoint(args, endpoint) {
      const { savedSearchesByIndexTitle } = args;

      return endpoint.eid
        ? { type: EntityType.VIRTUAL_ENTITY, id: endpoint.eid.name }
        : {
          type: EntityType.SAVED_SEARCH,
          id: savedSearchesByIndexTitle[endpoint.field.indexPattern.title].id,
          field: endpoint.field.name
        };
    }

    function relationsScope(args, relationGroups) {
      const { mediator } = args;

      const dstSortControls = {};
      const srcSortControls = {};

      const sort = {
        src: sortContext({
          text: endpoint => endpoint.text
        }, { controls: srcSortControls }),
        dst: sortContext({
          text: relGroup => relGroup.target.text,
          typeField: relGroup => relGroup.typeField.type
        }, { controls: dstSortControls })
      };

      sort.src.text.setOrder('asc');
      sort.dst.text.setOrder('asc');


      const scope = _.assign({
        relationGroups: dstSortControls.sort(relationGroups),
        sort,
        selectedGroup: null,
        selCount: '',
        allSelected: allSelected(relationGroups, 'allSelected.value'),
        mediator,


        reset() {
          this.viewMode = 'relations';
          this.explorerDomain = null;
          this.explorerTarget = null;
        },

        updateCount() {
          let selected = 0;
          let total = 0;

          _.chain(this.relationGroups).map('sources').flatten()
            .forEach(src => {
              selected += +src.selected;
              ++total;
            })
            .commit();

          this.selCount = selected + '/' + total;
        },

        selectGroup(group, $event) {
          if ($event) { $event.stopPropagation(); }

          if (group === this.selectedGroup) { return; }
          this.selectedGroup = group;

          if (!group) { return; }

          group.sources = srcSortControls.sort(group.sources);
        },

        showRelationSamples(src, $event) {
          if ($event) { $event.stopPropagation(); }

          this.viewMode = 'samples';

          this.explorerDomain = toExplorerEndpoint(args, src);
          this.explorerRange = toExplorerEndpoint(args, this.selectedGroup.target);
        },

        toggleTargetCandidates(group, $event) {
          this.selectGroup(group, $event);

          group.targetExpanded = !group.targetExpanded;
          if (!group.targetExpanded) { return; }

          const cell = $($event.target).closest('td');
          const select = cell.find('select')[0];

          $timeout(() => {
            select.size = Math.min(group.targetCandidates.length, 10);
            select.focus();
          });
        },

        closeTargetCandidates(group, $event) {
          if ($event) { $event.stopPropagation(); }
          group.targetExpanded = false;
        },

        changedTarget(group) {
          group.sources = srcSortControls.sort(
            _.without(group.fieldEndpoints, group.target));
        }
      });

      scope.selectGroup(scope.relationGroups[0] || null);
      scope.reset();
      scope.updateCount();

      return scope;
    }

    function knownRelationsScope(args, relationGroups) {
      const {
        savedSearches, allSavedSearches, existingConnections, ontologyRelations
      } = args;

      const sSearchesById = _.indexBy(savedSearches, 'id');
      const allSSearchesById = _.indexBy(allSavedSearches, 'id');

      const sortControls = {};
      const sort = sortContext({
        srcText: 'srcText',
        dstText: 'dstText',
        status: 'status'
      }, { controls: sortControls });

      sort.srcText.setOrder('asc');
      sort.dstText.setOrder('asc');
      sort.status.setOrder('asc');

      function relEndpointText(endpoint) {
        if (endpoint.type === EntityType.VIRTUAL_ENTITY) { return endpoint.label; }

        const savedSearch = allSSearchesById[endpoint.id];
        return savedSearch && pairText(savedSearch.title, endpoint.field);
      }

      function initialRelStatus(rel) {
        const processedDomain =
          rel.domain.type === EntityType.VIRTUAL_ENTITY ||
          sSearchesById[rel.domain.id];

        const processedRange =
          rel.range.type === EntityType.VIRTUAL_ENTITY ||
          sSearchesById[rel.range.id];

        return (processedDomain && processedRange) ? 'not-found' : 'unknown';
      }

      const hydratedOntologyRelations = _(ontologyRelations)
        .map(rel => ({
          relation: rel,
          srcText: relEndpointText(rel.domain),
          dstText: relEndpointText(rel.range),
          status: initialRelStatus(rel),
          withEntity:
            rel.domain.type === EntityType.VIRTUAL_ENTITY ||
            rel.range.type === EntityType.VIRTUAL_ENTITY,
        }))
        .filter(rel => rel.srcText < rel.dstText)
        .map(rel => rel.relation.domain.type === EntityType.VIRTUAL_ENTITY
          ? _.assign(rel, { srcText: rel.dstText, dstText: rel.srcText })
          : rel)
        .value();

      return {
        sort,
        relations: [],
        showLegend: true,

        reset() {
          const connections = relationGroupsToConnections(relationGroups)
            .concat(existingConnections);

          const srcMap = _(connections)
            .groupBy(conn => conn.source.text)
            .mapValues(group => _.indexBy(group, conn => conn.target.text))
            .value();

          const eidByEndpointText = _(connections)
            .filter('target.eid')
            .reduce(function (memo, conn) {
              memo[conn.source.text] = conn.target.text;
              return memo;
            }, {});

          this.relations = hydratedOntologyRelations.map(rel => {
            let { srcText, dstText, status } = rel;

            if (status === 'unknown') { return rel; }

            const dirDstMap = srcMap[srcText];
            const invDstMap = srcMap[dstText];
            const srcEid = eidByEndpointText[srcText];
            const dstEid = eidByEndpointText[dstText];

            const dirFound = (dirDstMap && dirDstMap[dstText]);
            const invFound = (invDstMap && invDstMap[srcText]);
            const eidFound = srcEid && (srcEid === dstEid);

            if (dirFound || invFound) { status = 'found'; }
            else if (eidFound) { status = 'found-eid'; }
            else { status = 'not-found'; }

            if (invFound) { srcText = rel.dstText; dstText = rel.srcText; }

            if (rel.withEntity) { dstText = '[EID] ' + dstText; }

            return { srcText, dstText, status };
          });

          this.relations = sortControls.sort(this.relations);
        }
      };
    }

    function notesScope(args) {
      let { savedSearches } = args;
      savedSearches = _.clone(savedSearches);

      const fieldsSortControls = {};
      const sort = {
        searches: sortContext({ text: 'title' }),
        fields: sortContext({
          text: 'fName',
          typeField: 'typeField.type'
        }, { controls: fieldsSortControls })
      };

      sort.searches.text.setOrder('asc', savedSearches);
      sort.fields.text.setOrder('asc');

      const scope = {
        savedSearches,
        sort,
        selectedSearch: null,
        searchFields: null,

        selectSearch(ssearch) {
          const fields = args.dataset
            .indices[ssearchIndexPattern(ssearch).title]
            .fields;

          this.selectedSearch = ssearch;
          this.searchFields = _.map(fields, field => _.assign({
            typeField: toTypeField(field.orig)
          }, field));

          this.searchFields = fieldsSortControls.sort(this.searchFields);
        }
      };

      scope.selectSearch(savedSearches[0]);
      return scope;
    }


    // Body

    let { relationGroups } = args;
    relationGroups = _.map(relationGroups, toReportGroup);

    return baseModal(findRelationsReportTemplate, {
      overlayClass: 'kibi-findrel-overlay',
      onEscKey: _.noop,
      tabs: {
        foundRelations: true, knownRelations: false,
        fields: false, log: false
      },
      f: relationsScope(args, relationGroups),
      k: knownRelationsScope(args, relationGroups),
      n: notesScope(args),
      l: { log: args.logString, errorsCount: args.errorsCount }
    })
      .show()
      .then(action => {
        if (!action) { return Promise.reject('quit'); }
        if (action !== 'ok') { return Promise.reject(); }

        args.action = action;
        args.relationGroups = _.map(relationGroups,
          group => _.pick(group, 'endpoints', 'sources', 'target'));

        return args;
      });
  }

  function showError(err, log) {
    return baseModal(findRelationsErrorTemplate, {
      overlayClass: 'kibi-findrel-overlay',
      err, log
    })
      .show();
  }

  return {
    linksToConnections,
    linksToRelationGroups,
    relationGroupsToConnections,

    show,
    showError
  };
};
