import template from './relation_explorer.html';
import './relation_explorer.less';

import './partials/kibi_relation_explorer';
import './partials/kibi_terms_explorer';

import { RelationsHelperProvider } from './relations_helper';

import { BaseModalProvider } from 'ui/kibi/modals/base_modal';
import { promiseMapSeries } from 'ui/kibi/utils/promise';

import { EntityType } from 'ui/kibi/components/ontology/entity_type';

import Bluebird from 'bluebird';
import _ from 'lodash';

import { uiModules } from 'ui/modules';


export function RelationExplorerProvider(Private, savedSearches) {
  const relationsHelper = Private(RelationsHelperProvider);
  const baseModal = Private(BaseModalProvider);

  const termsCount = 100;

  const { pairText } = relationsHelper;
  const phases = {};


  function initScope(args) {
    const { scope } = args;

    let cancelRequests;
    const canceledPromise = new Promise(function (resolve) {
      cancelRequests = resolve;
    });

    _.assign(scope, {
      status: 'loading',
      errorType: '',
      error: '',

      childModifiable: {},

      direction: null,
      sourceData: null,
      targetData: null,
      leftData: null,
      rightData: null,

      terms: [],
      docs: [],
      docIndex: -1,
      matchRatio: 0,


      nextDoc() {
        this.docIndex = Math.min(this.docIndex + 1, this.docs.length - 1);
      },
      prevDoc() {
        this.docIndex = Math.max(this.docIndex - 1, 0);
      },

      cancelAndClose() {
        cancelRequests();
        if (this.onClose) { this.onClose(); }
      },

      switchDirection() {
        cancelRequests();
        args.direction = args.invDirection;
        scope.waitPreparation = phases.prepareScope(args);
      }
    });

    args.canceledPromise = canceledPromise;
    return args;
  }

  function preprocessInput(args) {
    const [ direction, invDirection ] = [ args.direction, args.invDirection ] =
      (args.direction === 'left') ? [ 'left', 'right' ] : [ 'right', 'left' ];

    let domain = args[invDirection];
    let range = args[direction];

    if (!domain.field) {
      [ domain, range ] = [ range, domain ];
    }

    args.domain = domain;
    args.range = range;

    return args;
  }

  function validateInput(args) {
    const { domain, range, scope } = args;

    const okDomain = domain.id && domain.field;
    const okRange = range.id && (range.type === EntityType.VIRTUAL_ENTITY || range.field);

    if (okDomain && okRange) {
      return args;
    }

    scope.status = 'error';
    scope.errorType = 'The relation is incomplete!';

    return Promise.reject();
  }

  function ssearchIndexPattern(ssearch) {
    return ssearch.searchSource.get('index');
  }

  function ssearchField(ssearch, endpoint) {
    return ssearchIndexPattern(ssearch).fields.byName[endpoint.field];
  }

  function fieldText(ssearch, field) {
    return pairText(ssearch.title, field.displayName);
  }

  function getFieldData(endpoint) {
    return savedSearches.get(endpoint.id)
      .then(ssearch => {
        const field = ssearchField(ssearch, endpoint);
        const text = fieldText(ssearch, field);

        return { field, text };
      });
  }

  function getFieldsData(args) {
    const { domain, range } = args;

    return Promise.all([
      getFieldData(domain),
      range.field && getFieldData(range)
    ]).then(([ sourceData, targetData ]) => {
      [ args.sourceData, args.targetData ] = [ sourceData, targetData ];

      args[args.invDirection + 'Data'] = sourceData;
      args[args.direction + 'Data'] = targetData;

      return args;
    });
  }

  function validateFields(args) {
    const { sourceData, targetData, scope } = args;

    if (!targetData || sourceData.field.type === targetData.field.type) {
      return args;
    }

    scope.status = 'error';
    scope.errorType = 'Fields have different types!';

    return Promise.reject();
  }

  function makeRequestsMediator(args) {
    const { sourceData, targetData, requestsMediator } = args;

    if (requestsMediator) { return args; }

    const indexPatterns = _([ sourceData, targetData ])
      .compact()
      .map('field.indexPattern')
      .uniq()
      .value();

    return relationsHelper.getIndicesByPattern(indexPatterns)
      .then(relationsHelper.makeRequestsMediator)
      .then(requestsMediator => {
        args.requestsMediator = requestsMediator;
        return args;
      });
  }

  function loadTerms(args) {
    const { sourceData, requestsMediator, canceledPromise } = args;

    return relationsHelper.getTerms(
      sourceData.field, termsCount, requestsMediator, { canceledPromise }
    ).then(terms => {
      args.terms = terms;
      if (!terms.length) { args.scope.status = 'no-terms'; }

      return args;
    });
  }

  function toScope(args) {
    const { scope } = args;

    _.assign(scope,  _.pick(args, [
      'direction', 'sourceData', 'targetData', 'leftData', 'rightData', 'terms'
    ]));

    scope.nextDoc();
    return args;
  }

  function loadDocs(args) {
    const {
      scope, sourceData, targetData, direction, invDirection, terms,
      requestsMediator, canceledPromise
    } = args;

    // Docs are accumulated as they get retrieved from the backend, but
    // presented in the UI at 500ms intervals
    let storedDocs;


    function matchRatio(nestedDocs) {
      const matchedDocsCount = _.sum(nestedDocs, fieldDocs => +_.all(fieldDocs));
      return _.round(100 * matchedDocsCount / nestedDocs.length, 0);
    }

    function toScopeDocs(nestedDocs) {
      const [ sourceDocs, targetDocs ] = _.zip(...nestedDocs);

      return _(0).range(nestedDocs.length)
        .map(t => ({
          term: terms[t],
          value: sourceDocs[t],
          [invDirection]: sourceDocs[t],
          [direction]: targetDocs && targetDocs[t]
        }))
        .value();
    }

    function updateDocsPresentation() {
      if (!storedDocs) { return; }

      scope.status = 'ready';
      scope.docIndex = Math.max(scope.docIndex, 0);
      scope.matchRatio = matchRatio(storedDocs);
      scope.docs = toScopeDocs(storedDocs);
    }

    const throttledUpdateDocsPresentation =
      _.throttle(updateDocsPresentation, 500);

    function onTermDocsReceived(nestedDocs) {
      storedDocs = nestedDocs;
      throttledUpdateDocsPresentation();
    }


    const fields = _([ sourceData, targetData ])
      .compact().map('field').value();

    return relationsHelper.getTermDocs(fields, terms, requestsMediator, {
      canceledPromise, onTermDocsReceived
    })
      .then(() => {
      // Ensure UI is up-to-date before returning
        throttledUpdateDocsPresentation.cancel();
        updateDocsPresentation();

        return args;
      });
  }

  function onError(args, err) {
    const { scope } = args;

    scope.status = 'error';
    scope.errorType = 'Error while retrieving samples';
    scope.error = err.message || err;

    throw err;
  }


  _.assign(phases, {
    prepareScope(args) {
      return Promise.resolve(args)
        .then(initScope)
        .then(preprocessInput)
        .then(validateInput)
        .then(getFieldsData)
        .then(validateFields)
        .then(makeRequestsMediator)
        .then(loadTerms)
        .then(toScope)
        .then(loadDocs)
        .catch(err => err && onError(args, err));
    }
  });

  return {
    prepareScope: phases.prepareScope,

    spawnModal(left, right, direction = 'right') {
      const modal = baseModal(template, {});

      modal.scope.onClose = modal.scope.onCancel;
      this.prepareScope({ scope: modal.scope, left, right, direction });

      return modal.show();
    }
  };
}


uiModules
  .get('kibana')
  .directive('kibiRelationExplorerModal', function (Private) {
    return {
      restrict: 'E',
      scope: {
        left: '=',
        right: '=',
        direction: '=?',
        requestsMediator: '=?',
        onClose: '&'
      },
      template,
      controller($scope) {
        $scope.waitPreparation = Private(RelationExplorerProvider)
          .prepareScope(_.assign({ scope: $scope }, _.pick($scope, [
            'left', 'right', 'direction', 'requestsMediator'
          ])));
      }
    };
  });
