'use strict';

var _ui_bundle = require('./ui_bundle');

var _ui_bundle2 = _interopRequireDefault(_ui_bundle);

var _app_entry_template = require('./app_entry_template');

var _app_entry_template2 = _interopRequireDefault(_app_entry_template);

var _lodash = require('lodash');

var _bluebird = require('bluebird');

var _minimatch = require('minimatch');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const mkdirp = (0, _bluebird.promisify)(require('mkdirp'));

class UiBundleCollection {
  constructor(bundlerEnv, filter) {
    this.each = [];
    this.env = bundlerEnv;
    this.filter = (0, _minimatch.makeRe)(filter || '*', {
      noglobstar: true,
      noext: true,
      matchBase: true
    });
  }

  add(bundle) {
    if (!(bundle instanceof _ui_bundle2.default)) {
      throw new TypeError('expected bundle to be an instance of UiBundle');
    }

    if (this.filter.test(bundle.id)) {
      this.each.push(bundle);
    }
  }

  addApp(app) {
    this.add(new _ui_bundle2.default({
      id: app.id,
      modules: app.getModules(),
      template: _app_entry_template2.default,
      env: this.env
    }));
  }

  desc() {
    switch (this.each.length) {
      case 0:
        return '0 bundles';
      case 1:
        return `bundle for ${this.each[0].id}`;
      default:
        const ids = this.getIds();
        const last = ids.pop();
        const commas = ids.join(', ');
        return `bundles for ${commas} and ${last}`;
    }
  }

  async ensureDir() {
    await mkdirp(this.env.workingDir);
  }

  async writeEntryFiles() {
    await this.ensureDir();

    for (const bundle of this.each) {
      const existing = await bundle.readEntryFile();
      const expected = bundle.renderContent();

      if (existing !== expected) {
        await bundle.writeEntryFile();
        await bundle.clearBundleFile();
      }
    }
  }

  async getInvalidBundles() {
    const invalids = new UiBundleCollection(this.env);

    for (const bundle of this.each) {
      const exists = await bundle.checkForExistingOutput();
      if (!exists) {
        invalids.add(bundle);
      }
    }

    return invalids;
  }

  toWebpackEntries() {
    return (0, _lodash.transform)(this.each, function (entries, bundle) {
      entries[bundle.id] = bundle.entryPath;
    }, {});
  }

  getIds() {
    return (0, _lodash.map)(this.each, 'id');
  }

  toJSON() {
    return this.each;
  }
}

module.exports = UiBundleCollection;
