import React from 'react';
import { BrowserRouter } from 'react-router-dom';
import { shape } from 'prop-types';
import { mount, shallow } from 'enzyme';
import expect from 'expect.js';
import sinon from 'sinon';
import {
  testConfig,
  datasources,
  virtualIndices,
  expectedMergedConfig
} from './data';
import ConfigurationForm from '../configuration_form';
import Client from '../../../client';
import cloneDeep from 'lodash.clonedeep';

// Instantiate router context
const router = {
  history: new BrowserRouter().history,
  route: {
    location: {},
    match: {
      params: {}
    },
  },
};

const createContext = () => ({
  context: { router },
  childContextTypes: { router: shape({}) },
});

describe('ingest ConfigurationForm', () => {
  describe('form renders', () => {
    const errorNotifierMock = sinon.mock();
    const renderHeaderStub = sinon.stub(ConfigurationForm.prototype, 'renderHeader');
    let initDatasourceListStub = sinon.stub(ConfigurationForm.prototype, 'initDatasourceList');
    const clientStub = sinon.stub(new Client('', () => {}));
    let configFormMount;
    let configFormShallow;
    let routerParams;
    const getConfigFormShallow = (props) => {
      if (!configFormShallow) {
        configFormShallow = shallow(
          <ConfigurationForm
            {...routerParams}
            {...props}
            client={clientStub}
            createNotifier={() => { return { error: errorNotifierMock }; }}
          />, createContext());
      }
      return configFormShallow;
    };
    const getConfigFormMount = (props) => {
      if (!configFormMount) {
        configFormMount = mount(
          <ConfigurationForm
            {...routerParams}
            {...props}
            client={clientStub}
            createNotifier={() => { return { error: errorNotifierMock }; }}
          />, createContext());
      }
      return configFormMount;
    };
    beforeEach(() => {
      configFormMount = null;
      routerParams = {
        match: {
          params: {}
        },
        history: {
          push: () => {}
        }
      };
      errorNotifierMock.reset();
      initDatasourceListStub.reset();
      renderHeaderStub.reset();
    });

    afterEach(() => {
      getConfigFormMount().unmount();
    });

    describe('validateAndCleanseBeforeSave', () => {
      let configuration;
      beforeEach(() => {
        configFormShallow = null;
        configuration = {};
      });

      it('clears pk_field if it does not exist', () => {
        const configFormShallow = getConfigFormShallow();
        const instance = configFormShallow.instance();
        configuration.pk_field = 'random-garbage-field';
        instance.validateAndCleanseBeforeSave(configuration);
        expect(configuration.pk_field).to.equal('');
      });

      it('does not change pk_field if it exists', () => {
        const configFormShallow = getConfigFormShallow();
        const instance = configFormShallow.instance();
        const primaryKey = 'my-field';
        configuration.pk_field = primaryKey;
        instance.state.mappingData.items = [{ sanitizedFieldName: primaryKey }];
        instance.validateAndCleanseBeforeSave(configuration);
        expect(configuration.pk_field).to.be(primaryKey);
      });
    });

    describe('autoFill', () => {
      let newState;
      let oldState;
      const sourceValue = 'my-datasource';

      beforeEach(() => {
        configFormShallow = null;
        oldState = {
          isNew: true,
          id: '',
          configuration: {
            target: '',
            virtual_index: '',
            datasource: '',
            description: '',
            pk_field: ''
          }
        };
        newState = {
          configuration: {
            virtual_index: '',
            datasource: ''
          }
        };
      });

      [
        {
          source: 'virtual_index'
        },
        {
          source: 'datasource'
        }
      ].forEach(({ source }) => {
        it(`sets id upon ${source} change`, () => {
          const configFormShallow = getConfigFormShallow();
          const instance = configFormShallow.instance();
          newState.configuration[source] = sourceValue;
          expect(instance.autoFill(newState, oldState)).to.have.property('id', `${sourceValue}.reflection`);
        });

        it(`sets target index upon ${source} change`, () => {
          const configFormShallow = getConfigFormShallow();
          const instance = configFormShallow.instance();
          newState.configuration[source] = sourceValue;
          expect(instance.autoFill(newState, oldState).configuration).to.have.property('target', `${sourceValue}.reflection`);
        });

        it(`sets description upon ${source} change`, () => {
          const configFormShallow = getConfigFormShallow();
          const instance = configFormShallow.instance();
          newState.configuration[source] = sourceValue;
          expect(instance.autoFill(newState, oldState).configuration.description).to.contain(sourceValue);
        });

        it(`upon ${source} change: nothing is changed when editing existing configuration`, () => {
          const configFormShallow = getConfigFormShallow();
          const instance = configFormShallow.instance();
          newState.configuration[source] = sourceValue;
          oldState.isNew = false;
          const passedState = cloneDeep(newState);
          expect(instance.autoFill(newState, oldState)).to.eql(passedState);
        });
      });

      it(`sets pk_field upon virtual_index change`, () => {
        const configFormShallow = getConfigFormShallow();
        const instance = configFormShallow.instance();
        const primaryKey = 'xyz';
        instance.virtualIndexPrimaryKeys = { [sourceValue]: primaryKey };
        newState.configuration.virtual_index = sourceValue;
        expect(instance.autoFill(newState, oldState).configuration).to.have.property('pk_field', primaryKey);
      });

      it(`does NOT set pk_field upon datasource change`, () => {
        const configFormShallow = getConfigFormShallow();
        const instance = configFormShallow.instance();
        const primaryKey = 'xyz';
        instance.virtualIndexPrimaryKeys = { [sourceValue]: primaryKey };
        newState.configuration.datasource = sourceValue;
        expect(instance.autoFill(newState, oldState).configuration).to.not.have.property('pk_field');
      });
    });

    it('initDatasourceList called with true upon new configuration', () => {
      const configFormMount = getConfigFormMount();
      expect(initDatasourceListStub.calledOnce);
      expect(initDatasourceListStub.alwaysCalledWithExactly([true]));
      expect(errorNotifierMock.notCalled);
    });
    it('renderHeader called upon mount (By Section)', () => {
      const configFormMount = getConfigFormMount();
      expect(renderHeaderStub.calledOnce);
      expect(errorNotifierMock.notCalled);
    });
    it('STEP_INPUT rendered upon new config', () => {
      const renderStepInputSpy = sinon.spy(ConfigurationForm.prototype, 'renderStepInput');
      const configFormMount = getConfigFormMount();
      expect(renderStepInputSpy.calledOnce);
      expect(errorNotifierMock.notCalled);
      renderStepInputSpy.restore();
    });
    it('opening saved configuration works', (done) => {
      ConfigurationForm.prototype.initDatasourceList.restore();
      const initDatasourceListSpy = sinon.spy(ConfigurationForm.prototype, 'initDatasourceList');
      const renderStepInputSpy = sinon.spy(ConfigurationForm.prototype, 'renderStepInput');
      const onMountSpy = sinon.spy(ConfigurationForm.prototype, 'onMount');
      const testQueryStub = sinon.stub(ConfigurationForm.prototype, 'testQuery');
      const ID = testConfig.id;
      routerParams.match.params.id = ID;
      const fetchConfigurationStub = clientStub.fetchConfiguration;
      fetchConfigurationStub.withArgs(ID).returns(Promise.resolve(testConfig));
      const jdbcDatasources = {
        list: sinon.stub(),
        listVirtualIndices: sinon.stub()
      };
      jdbcDatasources.list.returns(Promise.resolve(datasources));
      jdbcDatasources.listVirtualIndices.returns(Promise.resolve(virtualIndices));
      const configFormMount = getConfigFormMount({ jdbcDatasources });
      //Check basic calls
      expect(initDatasourceListSpy.calledOnce);
      expect(initDatasourceListSpy.alwaysCalledWithExactly([false]));
      expect(renderStepInputSpy.calledOnce);
      expect(renderHeaderStub.calledOnce);

      //Loading saved Config
      onMountSpy.returnValues[0].then(() => {
        expect(fetchConfigurationStub.calledOnce);
        expect(fetchConfigurationStub.alwaysCalledWithExactly(ID));
        expect(jdbcDatasources.list.calledOnce);
        expect(jdbcDatasources.listVirtualIndices.calledOnce);
        expect(testQueryStub.calledOnce);
        expect(testQueryStub.firstCall.args).to.eql([expectedMergedConfig, true]);
        expect(errorNotifierMock.notCalled);

        initDatasourceListSpy.restore();
        renderStepInputSpy.restore();
        onMountSpy.restore();
        testQueryStub.restore();
        initDatasourceListStub = sinon.stub(ConfigurationForm.prototype, 'initDatasourceList');
        done();
      });
    });
  });
});
