import {
  status
} from '../jobs_table';
import {
  JobStatus
} from './constants';

class OntologyService {
  constructor(jobInfo, client, chrome) {
    this.client = client;
    this.jobId = jobInfo.jobId;
    this.jobData = jobInfo.jobData;
    this._apiBasePath = chrome.addBasePath(chrome.getInjected('apiBasePath'));
    this._neo4jAPI = `${this._apiBasePath}v1/neo4j/`;
    this.relatedJobs = jobInfo.jobs_created.reduce((acc, jobId) => {
      acc.add(jobId);
      return acc;
    }, new Set());
  }

  /**
   * Fetches all reflection jobs
   * @return {Array.<Object>}
   */
  async _fetchReflectionJobs() {
    return await this.client.fetchConfigurations();
  }

  async _createOntology() {
    const resp = await this.client.fetch(`${this._neo4jAPI}createOntology`, 'POST', { indexEntities: this.jobData, jobId: this.jobId });
    if (!resp.acknowledged && resp.error) {
      throw new Error(resp.error);
    }
    return resp;
  }

  /**
   * Decides the status of passed job
   * @param  {Object} job
   * @return {JobStatus}
   */
  _getJobStatus(job) {
    const jobLog = job._run || job._last || {};
    if (jobLog.count > 0) {
      return JobStatus.SUCCESS; //Enough for us to start with the data model
    }
    switch (jobLog.state) {
      case status.running:
        return JobStatus.WAITING;
      case status.completed:
        //Federate would've pushed some mapping
        return JobStatus.SUCCESS;
      default:
        return JobStatus.FATAL;
    }
  }

  /**
   * Returns reflection jobs relevant to current job
   * Along with their _status
   * @return {Object}
   */
  async _fetchFilteredReflectionJobs() {
    const jobs = await this._fetchReflectionJobs();
    const jobStatusArray = [];
    const filteredJobs = jobs.filter(job => {
      if (this.relatedJobs.has(job.id)) {
        const status = this._getJobStatus(job);
        job._status = status;
        jobStatusArray.push(status);
        return true;
      }
      return false;
    });
    return {
      status: jobStatusArray.length > 0 ? Math.min(...jobStatusArray) : JobStatus.NO_JOBS_FOUND,
      jobs: filteredJobs
    };
  }

  /**
   * Returns job status and reflection
   * @return {[type]} [description]
   */
  async getReflectionJobs() {
    return await this._fetchFilteredReflectionJobs();
  }

  /**
   * get a list of indexEntities for ontology mapping
   * @return {Array.<Object>}
   */
  getJobDataList() {
    return Object.keys(this.jobData).map(indexEntity => this.jobData[indexEntity]);
  }

  /**
   * @param  {Array.<Object>} jobDataList
   */
  async publishOntology(jobDataList) {
    return await this._createOntology();
  }
}

export default {
  JobStatus,
  OntologyService
};