import React from 'react';
import TransformDropdownOptions from './transform_options';
import MappingHandler from './mapping_handler';
import MappingEditor from './mapping_editor';

import {
  EuiFieldText,
  EuiFormRow,
  EuiSpacer,
  EuiButton,
  EuiButtonIcon,
  EuiFlexGroup,
  EuiFlexItem,
  EuiSuperSelect,
  EuiSelect
} from '@elastic/eui';


//TODO: if there are many fields this will need filtering and pagination
export default class TransformEditor extends React.Component {

  static getDerivedStateFromProps(props, state) {
    const derived = {};

    if (state === null || state.prevProps.transforms !== props.transforms) {
      derived.transforms = TransformEditor.mapTransforms(props.transforms);
      derived.prevProps = {
        transforms: props.transforms
      };
    }

    return derived;
  }

  static mapTransforms(transforms) {
    return transforms.map(transform => {
      return {
        input: transform.input,
        transform: transform.transform,
        output: transform.output,
        mappingOption: MappingHandler.getMappingOption(transform.mapping),
        mapping: transform.mapping
      };
    });
  }

  constructor(props) {
    super(props);
    this.dirty = false;
    this.mappingOptions = MappingHandler.getMappingOptions();
    this.state = {
      transforms: TransformEditor.mapTransforms(props.transforms),
      prevProps: {
        transforms: props.transforms
      }
    };
  }

  // TODO: not implemented
  validate() {
    this.dirty = false;
    return true;
  }

  // TODO: fix dirty handling
  componentDidUpdate() {
    if (this.dirty && this.validate() && this.props.onChange) {
      this.dirty = false;
      this.props.onChange(this.state.transforms);
    }
  }

  onAddTransform() {
    this.dirty = true;
    this.setState(state => {
      const transforms = state.transforms;
      transforms.push({
        input: '',
        output: '',
        transform: 'copy',
        mappingOption: 'default',
        mapping: '',
        valid: false
      });
      return { transforms };
    });
  }

  onRemoveTransform(index) {
    this.dirty = true;
    this.setState(state => {
      const transforms = state.transforms;
      transforms.splice(index, 1);
      return { transforms };
    });
  }

  onTransformChange(index, field, value) {
    this.setState(state => {
      this.dirty = true;
      const transforms = state.transforms;
      transforms[index][field] = value;
      if (field === 'mappingOption') {
        const mapping = MappingHandler.getMapping(value);
        transforms[index].mapping = JSON.stringify(mapping);
      }
      return { transforms };
    });
  }

  renderHeader() {
    if (this.state.transforms.length === 0) {
      return;
    }
    return (
      <tr>
        <th>Type</th>
        <th>Input fields</th>
        <th>Output field</th>
        <th>Mapping</th>
        <th />
      </tr>
    );
  }

  renderRows() {
    const elements = [];
    for (let i = 0; i < this.state.transforms.length; i++) {
      elements.push(
        <tr key={`row-${i}`}>
          <td>
            <EuiFormRow hasEmptyLabelSpace>
              <EuiSuperSelect
                options={TransformDropdownOptions}
                valueOfSelected={this.state.transforms[i].transform}
                onChange={value => this.onTransformChange(i, 'transform', value)}
                itemLayoutAlign="top"
                hasDividers
              />
            </EuiFormRow>
          </td>
          <td>
            <EuiFormRow hasEmptyLabelSpace>
              <EuiFieldText
                value={this.state.transforms[i].input}
                onChange={e => this.onTransformChange(i, 'input', e.target.value)}
              />
            </EuiFormRow>
          </td>
          <td>
            <EuiFormRow hasEmptyLabelSpace>
              <EuiFieldText
                value={this.state.transforms[i].output}
                onChange={e => this.onTransformChange(i, 'output', e.target.value)}
              />
            </EuiFormRow>
          </td>
          <td>
            <EuiFormRow hasEmptyLabelSpace>
              <EuiSelect
                options={this.mappingOptions}
                value={this.state.transforms[i].mappingOption}
                onChange={e => {this.onTransformChange(i, 'mappingOption', e.target.value);}}
              />
            </EuiFormRow>
            <MappingEditor
              mapping={this.state.transforms[i].mapping}
              mappingOption={this.state.transforms[i].mappingOption}
              onChange={mapping => this.onTransformChange(i, 'mapping', mapping)}
            />
          </td>
          <td>
            <EuiFormRow hasEmptyLabelSpace>
              <EuiButtonIcon
                color="danger"
                iconType="trash"
                aria-label="Remove"
                onClick={() => this.onRemoveTransform(i)}
              />
            </EuiFormRow>
          </td>
        </tr>
      );
    }
    return elements;
  }

  render() {
    return (
      <React.Fragment>

        <table className="transforms-table">
          <thead>
            {this.renderHeader()}
          </thead>
          <tbody>
            {this.renderRows()}
          </tbody>
        </table>

        <EuiSpacer/>

        <EuiFormRow
          fullWidth
        >
          <EuiFlexGroup gutterSize="s" alignItems="flexStart">
            <EuiFlexItem grow={false}>
              <EuiButton
                onClick={() => this.onAddTransform()}
                iconType="listAdd"
                data-test-subj="add-field"
              >
                Add Field
              </EuiButton>
            </EuiFlexItem>
          </EuiFlexGroup>
        </EuiFormRow>
      </React.Fragment>
    );
  }

}