import React, { Component } from 'react';
import PropTypes from 'prop-types';
import {
  EuiBasicTable,
  EuiAccordion,
  EuiSpacer
} from '@elastic/eui';

class ErrorTable extends Component {
  static computeItems(props, startingIndex, accumulator) {
    return (props.errors && props.errors.slice(startingIndex).reduce(function (unique, ele) {
      const { reason } = ele.index.error;
      const existingError = unique.find((ele) => ele.index.error.reason === reason);
      if (!existingError) {
        ele.index.error.errorCount = 1;
        ele.index.docIds = [ ele.index._id ];
        unique.push(ele);
      } else {
        existingError.index.error.errorCount++;
        if (existingError.index.docIds.length < 5) {
          existingError.index.docIds.push(ele.index._id);
        }
      }
      return unique;
    }, accumulator)) || [];
  }

  constructor(props) {
    super(props);
    this.totalErrorCount = (props.errors && props.errors.length) || 0;
    this.state = {
      items: ErrorTable.computeItems(props, 0, [])
    };
    this.columns = [{
      field: 'index',
      name: 'Error Type',
      render: ({ error }) => {
        return error.type;
      }
    }, {
      field: 'index',
      name: 'Caused By',
      render: ({ error }) => {
        const causedBy = error.caused_by || error;
        return `${causedBy.type}: ${causedBy.reason}`;
      }
    }, {
      field: 'index',
      name: 'Reason',
      render: ({ error }) => {
        return error.reason;
      }
    }, {
      field: 'index',
      name: 'Error Count',
      render: ({ error }) => {
        return error.errorCount;
      }
    }, {
      field: 'index',
      name: 'Document _id(s)',
      render: ({ docIds }) => {
        const columns = [{
          name: 'First 5 _id (Indexing Failed)',
          render: (id) => {
            return id;
          }
        }];
        return (
          <EuiBasicTable
            items={docIds}
            columns={columns}
          />
        );
      }
    }];
  }

  componentWillReceiveProps(nextProps) {
    const currentErrors = (this.props.errors && this.props.errors.length) || 0;
    if (!nextProps.errors) {
      this.setState({
        items: []
      });
    } else if (nextProps.newJob || nextProps.errors.length < currentErrors) {
      this.setState({
        items: ErrorTable.computeItems(nextProps, 0, [])
      });
    } else if (nextProps.errors.length > currentErrors) {
      this.setState(state => {
        return {
          items: ErrorTable.computeItems(nextProps, currentErrors, state.items)
        };
      });
    }
  }

  render() {
    if (this.state.items.length === 0) {
      return '';
    }
    return (
      <EuiAccordion
        id="errortable"
        buttonContent="Show Errors"
        initialIsOpen={this.props.initialIsOpen}
      >
        <div>
          <EuiSpacer size="m" />
          <EuiBasicTable
            items={this.state.items}
            columns={this.columns}
          />
          <EuiSpacer size="m" />
        </div>
      </EuiAccordion>
    );
  }
}

ErrorTable.propTypes = {
  errors: PropTypes.array,
  initialIsOpen: PropTypes.bool,
  newJob: PropTypes.bool
};

export default ErrorTable;