import React from 'react';
import { mount, shallow } from 'enzyme';
import { findTestSubject } from '@elastic/eui/lib/test';
import ImportControls from './importControls';
import { initConfigService } from '../../../test_utils';

describe('ImportControls', () => {
  let props;
  let componentMount;
  const handleNextStep = jest.fn();
  const previousStep = jest.fn();
  const switchSaveModal = jest.fn();
  const abortIngestion = jest.fn();
  const addErrorToast = jest.fn();
  const setImportPrompt = jest.fn();
  const DATA_TEST_SUBJ = Object.freeze({
    previousStepButton: 'previous-step',
    importButton: 'import',
    saveConfigButton: 'save-config',
    abortButton: 'abort-import',
    progressBarComponent: 'EuiProgress',
    importModalComponent: 'EuiConfirmModal',
    confirmModalCancelButton: 'confirmModalCancelButton',
    confirmModalConfirmButton: 'confirmModalConfirmButton'
  });

  const getMount = (args) => {
    if (!componentMount) {
      componentMount = mount(<ImportControls {...(args || props)} />);
    }
    return componentMount;
  };

  // findTestSubject cannot be used for shallow mounts
  const fetchDataTestSubj  = (component, value) => {
    return component.find(`[data-test-subj="${value}"]`);
  };

  const getBackButton = component => {
    return fetchDataTestSubj(component, DATA_TEST_SUBJ.previousStepButton);
  };

  const getImportButton = component => {
    return fetchDataTestSubj(component, DATA_TEST_SUBJ.importButton);
  };

  const getSaveButton = component => {
    return fetchDataTestSubj(component, DATA_TEST_SUBJ.saveConfigButton);
  };

  const getAbortButton = component => {
    return fetchDataTestSubj(component, DATA_TEST_SUBJ.abortButton);
  };

  const getImportModal = component => {
    return component.find(DATA_TEST_SUBJ.importModalComponent);
  };

  const testButtonsExist = (component, isRunning) => {
    expect(getBackButton(component).length).toBe(1);
    expect(getImportButton(component).length).toBe(1);
    expect(getSaveButton(component).length).toBe(1);

    if (isRunning) {
      expect(getAbortButton(component).length).toBe(1);
    }
  };

  const getProgressBar = component => {
    return component.find(DATA_TEST_SUBJ.progressBarComponent);
  };

  beforeAll(() => {
    initConfigService();
  });

  beforeEach(() => {
    handleNextStep.mockReset();
    previousStep.mockReset();
    switchSaveModal.mockReset();
    abortIngestion.mockReset();
    addErrorToast.mockReset();
    setImportPrompt.mockReset();

    props = {
      blockIngestion: false,
      importButton: {
        text: 'Import',
        loading: false
      },
      showImportPrompt: false,
      handleNextStep,
      previousStep,
      switchSaveModal,
      abortIngestion,
      addErrorToast,
      setImportPrompt
    };

    if (componentMount) {
      componentMount.unmount();
      componentMount = null;
    }
  });

  it('renders', () => {
    const component = getMount();
    expect(component.length).toBe(1);
  });

  it('renders progress bar', () => {
    const component = getMount();
    expect(getProgressBar(component).length).toBe(1);
  });

  describe('render buttons', () => {

    it('render buttons while not importing', () => {
      const component = shallow(<ImportControls {...props} />);
      testButtonsExist(component);
    });

    it('render buttons buttons while importing', () => {
      props.importButton.loading = true;
      const component = shallow(<ImportControls {...props} />);
      testButtonsExist(component, true);
    });
  });

  it('calls no method upon render', () => {
    const component = getMount();
    expect(handleNextStep).not.toBeCalled();
    expect(previousStep).not.toBeCalled();
    expect(switchSaveModal).not.toBeCalled();
    expect(abortIngestion).not.toBeCalled();
    expect(addErrorToast).not.toBeCalled();
    expect(setImportPrompt).not.toBeCalled();
  });

  describe('Interactions Work', () => {
    it('can go to previous step', () => {
      const component = getMount();
      findTestSubject(component, DATA_TEST_SUBJ.previousStepButton).simulate('click');
      expect(previousStep.mock.calls.length).toBe(1);
    });

    it('can abort import', () => {
      props.importButton.loading = true;
      const component = getMount(props);
      findTestSubject(component, DATA_TEST_SUBJ.abortButton).simulate('click');
      expect(abortIngestion.mock.calls.length).toBe(1);
    });

    it('can save config', () => {
      const component = getMount(props);
      findTestSubject(component, DATA_TEST_SUBJ.saveConfigButton).simulate('click');
      expect(switchSaveModal.mock.calls.length).toBe(1);
    });

    it('can trigger import', () => {
      const component = getMount(props);
      findTestSubject(component, DATA_TEST_SUBJ.importButton).simulate('click');
      expect(handleNextStep.mock.calls.length).toBe(1);
    });

    it('progress tracker passed to handleNextStep', () => {
      const component = getMount(props);
      findTestSubject(component, DATA_TEST_SUBJ.importButton).simulate('click');
      expect(handleNextStep).toBeCalledWith(component.instance().progressTracker);
    });
  });

  describe('Quick import modal', () => {
    beforeAll(() => {
      window.requestAnimationFrame = () => {};
    });

    beforeEach(() => {
      props.showImportPrompt = true;
    });

    it('renders when showImportPrompt: true', () => {
      const component = getMount(props);
      expect(getImportModal(component).length).toBe(1);
    });

    it('does not render when showImportPrompt: false', () => {
      props.showImportPrompt = false;
      const component = getMount(props);
      expect(getImportModal(component).length).toBe(0);
    });

    it('onCancel works', () => {
      const component = getMount(props);
      findTestSubject(component, DATA_TEST_SUBJ.confirmModalCancelButton).simulate('click');
      expect(setImportPrompt.mock.calls.length).toBe(1);
      expect(setImportPrompt).toBeCalledWith(false);
    });

    it('onConfirm works', () => {
      const component = getMount(props);
      const handleNextStepSpy = jest.spyOn(component.instance(), 'handleNextStep');
      findTestSubject(component, DATA_TEST_SUBJ.confirmModalConfirmButton).simulate('click');
      expect(setImportPrompt.mock.calls.length).toBe(1);
      expect(setImportPrompt).toBeCalledWith(false);
      expect(handleNextStepSpy.mock.calls.length).toBe(1);
      handleNextStepSpy.mockRestore();
    });
  });
});