import React, { Component } from 'react';
import MultiValued from './multiValued';
import AdvJson from './advJson';
import {
  EuiFlexGroup,
  EuiFlexItem,
  EuiSelect,
  EuiSpacer
} from '@elastic/eui';
import {
  jsonObjectToString,
  jsonStringToObject
} from '../../../../shared_components';
import typeOptions from './typeOptions';

const Operations = Object.freeze({
  ADD: 1,
  REMOVE: 2
});

class TypeSelector extends Component {
  constructor(props) {
    super(props);
    this.state = props.item;
  }

  componentWillReceiveProps(props) {
    const item = props.item;
    item.multiValued = item.multiValued || this.state.multiValued;
    this.setState(item);
  }

  getManipulation(operation, type) {
    switch (type) {
      case 'text':
        if (operation === Operations.ADD) {
          return {
            fielddata: true,
            analyzer: 'english'
          };
        } else if (operation === Operations.REMOVE) {
          return ['fielddata', 'analyzer'];
        }
        break;
      case 'text_keyword':
        if (operation === Operations.ADD) {
          return {
            fielddata: true,
            analyzer: 'english',
            fields: {
              keyword: {
                type: 'keyword',
                ignore_above: 768
              }
            }
          };
        } else if (operation === Operations.REMOVE) {
          return ['fielddata', 'analyzer', 'fields'];
        }
        break;
    }
  }

  manipulateAdvJson(operation, type, advJson) {
    const opts = this.getManipulation(operation, type);
    if (opts) {
      switch (operation) {
        case Operations.ADD:
          advJson = ({ ...opts, ...advJson });
          break;
        case Operations.REMOVE:
          if (advJson) {
            opts.forEach(prop => delete advJson[prop]);
            for (let i = 0; i < opts.length; i++) {
              delete advJson[opts[i]];
            }
          }
          break;
      }
    }
    return advJson;
  }

  handleTypeChange(newType) {
    const prevType = this.state.type;
    let advJson = this.state.advJson;
    advJson = jsonStringToObject(advJson);

    const nonEmptyObject = advJson && Object.keys(advJson).length;
    if (nonEmptyObject) {
      advJson = this.manipulateAdvJson(Operations.REMOVE, prevType, advJson);
    }
    advJson = this.manipulateAdvJson(Operations.ADD, newType, advJson);

    advJson = advJson && Object.keys(advJson).length ? jsonObjectToString(advJson) : '';
    this.updateAdvJson(advJson, nonEmptyObject ? null : this.state.useAdv); // Preserve toggle if set by user
  }

  onChangeMapping = e => {
    const { name, value } = e.target;
    if (name === 'type') {
      this.handleTypeChange(value);
    }
    this.setState({ [name]: value });
  }

  onToggle = e => {
    const { name, checked } = e.target;
    this.setState({ [name]: checked });
  }

  updateAdvJson = (advJson, useAdv) => {
    this.setState({
      advJson,
      useAdv: useAdv || !!advJson
    });
  }

  render() {
    let dateNotice = null;
    let multiValued = null;
    if (this.state.type === 'date') {
      dateNotice = (
        <div>
          <b>Dates must be formatted according to one of the syntaxes allowed by the
            strict_date_optional_time ISO format e.g. YYYY-MM-DD [&#39;T&#39; HH:MM:SS [offset]]
          </b>
          <br />
          <a href="https://docs.support.siren.io/latest/platform/en/siren-investigate/data-reflection/date-format.html" target="_blank">
            Click here to know about custom date formats
          </a>
          <br />
        </div>
      );
    }
    if (!this.props.hideMultiValued) {
      multiValued = (
        <MultiValued
          item={this.state}
          onChangeMapping={this.onChangeMapping}
          onToggle={this.onToggle}
        />
      );
    }
    return (
      <EuiFlexGroup>
        <EuiFlexItem style={{ minWidth: 200 }}>
          <EuiSelect
            options={typeOptions}
            value={this.state.type}
            onChange={this.onChangeMapping}
            name="type"
          />
          {dateNotice}
          <EuiSpacer size="s" />
          {multiValued}
        </EuiFlexItem>
        <EuiFlexItem>
          <AdvJson item={this.state} triggerUpdate={this.updateAdvJson} />
        </EuiFlexItem>
      </EuiFlexGroup>
    );
  }
}

export default TypeSelector;