import React, {
  Component, Fragment
} from 'react';

import {
  EuiCallOut,
  EuiFormRow,
  EuiFlexGroup,
  EuiFlexItem,
  EuiFilePicker,
  EuiTitle,
  EuiText,
  EuiSpacer,
  EuiButton,
  EuiLoadingChart,
  EuiSelect,
  EuiSuperSelect,
  EuiTextColor
} from '@elastic/eui';

import PreviewTable from './previewTable';
import FileErrors from './file_errors';

import XLSXClient from '../services/xlsx_client.js';
import SheetService from '../services/sheet_service.js';
import {
  configService,
  confirmDestructModal
} from '../../../shared_components';

class StepOne extends Component {
  constructor(props) {
    super(props);
    this.xlsxClient = XLSXClient.getInstance();
    if (props.state) {
      this.fetchConfigs();
    }
    this.state = {
      proceeding: false,
      deletionDialog: false
    };
    this.noConfigSelected = '*None-Selected*';
    this.previewSize = configService().previewSize;
    this.stateValueChange = this.stateValueChange.bind(this);
    this.onFileChange = this.onFileChange.bind(this);
    this.onFileLoad = this.onFileLoad.bind(this);
    this.onSheetChange = this.onSheetChange.bind(this);
    this.onNextClick = this.onNextClick.bind(this);
    this.useConfig = this.useConfig.bind(this);
    this.deleteConfig = this.deleteConfig.bind(this);
  }

  componentDidMount() {
    this.fetchConfigs();
  }

  fetchConfigs() {
    this.xlsxClient.getConfigList().then(resp => {
      const dropdownDisplay = function (option, description) {
        return (
          <Fragment>
            <strong>{option}</strong>
            <EuiSpacer size="xs" />
            <EuiText size="s" color="subdued">
              <p className="euiTextColor--subdued">{description}</p>
            </EuiText>
          </Fragment>
        );
      };
      const configOptions = [];
      let selectedConfig;
      let usePreviouslySelected = false;
      if (resp.hits.total > 0) {
        const hits = resp.hits.hits;
        configOptions.push({
          value: this.noConfigSelected,
          inputDisplay: 'No configuration selected.'
        });
        hits.forEach((ele) => {
          if (this.props.state.selectedConfig === ele._id) {
            usePreviouslySelected = true;
          }
          configOptions.push({
            value: ele._id,
            inputDisplay: ele._source.name,
            dropdownDisplay: dropdownDisplay(ele._source.name, ele._source.description)
          });
        });
        selectedConfig = usePreviouslySelected && this.props.state.selectedConfig || configOptions[0].value;
      }
      this.props.stateUpdate({ configOptions, selectedConfig });
    });
  }

  onFileLoad(workbook, fileErrors) {
    if (fileErrors) {
      if (Array.isArray(fileErrors)) {
        this.props.stateUpdate({ fileErrors });
      } else {
        this.props.stateUpdate({ error: fileErrors });
      }
    }
    if (!workbook) {
      this.props.stateUpdate({ loading: false });
      return;
    }
    const options = workbook.getSheetNames();
    this.props.stateUpdate({ workbook, selectItem: { options }, loading: false,
      selectedItem: options[0] });
    this.onSheetChange({ target: options[0] });
  }

  onFileChange = (fileArray) => {
    if (fileArray.length > 0) {
      //UI reset
      this.props.stateUpdate({
        filename: fileArray[0].name,
        selectItem: { options: [{ value: '', text: 'Select a sheet' }] },
        data: { loaded: false, items: [], columns: [] },
        disableButton: true,
        loading: true,
        error: ''
      });
      SheetService.readFile(fileArray, this.onFileLoad);
    }
    else {
      this.props.stateUpdate({
        selectItem: { options: [{ value: '', text: '' }] },
        data: { loaded: false, items: [], columns: [] },
        disableButton: true,
        loading: false,
        error: '',
        fileErrors: [],
        workbook: {}
      });
    }
  };

  async onSheetChange(item) {
    this.props.stateUpdate({ data: { loaded: false }, sheetname: item.target.value,
      selectedItem: item.target.value
    });

    if (item.target.value === '') {
      this.props.stateUpdate({ data: { loaded: true, items: [], columns: [] }, disableButton: true });
      return;
    }

    if (this.props.state.workbook.sheetExists(item.target.value)) {
      this.processSheetChange(item.target.value);
    }
    else {
      this.props.stateUpdate({ data: { loaded: true, items: [], columns: [] }, disableButton: true });
    }
  }

  async processSheetChange(sheetName) {
    const columns = this.props.state.workbook.getHeaderRow(sheetName);
    const items = await this.props.state.workbook.sheetToJSON(sheetName, this.previewSize);
    this.props.stateUpdate({ data: { loaded: true, items: items, columns: columns }, disableButton: false });
  }

  onValueChange(value, fieldName) {
    this.props.onFieldValueChange(fieldName, value);
  }

  stateValueChange(value, fieldname) {
    this.setState({ [fieldname]: value });
  }

  useConfig() {
    this.xlsxClient.getConfig(this.props.state.selectedConfig).then(res => {
      res._source.state.savedConfiguration = {
        id: res._id,
        name: res._source.name,
        description: res._source.description
      };
      res._source.state.showImportPrompt = true;
      this.onNextClick(res._source.state);
    });
  }

  deleteConfig() {
    this.stateValueChange(true, 'deletionDialog');
  }

  deleteConfigModal() {
    if (this.state.deletionDialog) {
      const title = 'Delete Excel Configuration';
      const body = (
        <div>
          <p>You&rsquo;re about to delete the excel configuration.</p>
          <p>Are you sure you want to do this?</p>
        </div>
      );
      const onCancel = () => {
        this.stateValueChange(false, 'deletionDialog');
      };
      const onConfirm = () => {
        this.stateValueChange(false, 'deletionDialog');
        this.xlsxClient.deleteConfig(this.props.state.selectedConfig).then(() => {
          setTimeout(() => this.fetchConfigs(), 800);
        });
      };
      return confirmDestructModal(title, body, onCancel, onConfirm);
    }
  }

  onNextClick(savedConfig = {}) {
    this.setState({ proceeding: true });
    setTimeout(() => this.props.nextStep(savedConfig));
  }

  renderControls() {
    let sheetDisabled = true;
    if (!(Object.keys(this.props.state.workbook).length === 0 && this.props.state.workbook.constructor === Object))
    {
      sheetDisabled = false;
    }
    const configDisabled = !this.props.state.selectedConfig || this.props.state.selectedConfig === this.noConfigSelected;
    return (
      <EuiFlexGroup gutterSize="l" alignItems="flexEnd" justifyContent="spaceBetween" style={{ marginLeft: '0px' }}>

        <EuiFlexItem grow={true}>
          <EuiFormRow label="Select the sheet to import">
            <EuiSelect
              options={this.props.state.selectItem.options}
              disabled={sheetDisabled}
              value={this.props.state.selectedItem}
              style={{ minWidth: '180px' }}
              onChange={this.onSheetChange}
            />
          </EuiFormRow>
        </EuiFlexItem>

        <EuiFlexItem grow={false}>
          <EuiFormRow label="Select a configuration (optional)">
            <EuiSuperSelect
              valueOfSelected={this.props.state.selectedConfig}
              options={this.props.state.configOptions}
              hasDividers
              disabled={this.props.state.configOptions.length <= 0}
              onChange={e => this.onValueChange(e, 'selectedConfig')}
            />
          </EuiFormRow>
        </EuiFlexItem>

        <EuiFlexItem grow={false}>
          <EuiFormRow>
            <EuiButton
              iconType={configDisabled ? 'arrowRight' : 'memory'}
              disabled={this.props.state.disableButton || this.state.proceeding}
              isLoading={this.state.proceeding}
              onClick={() => {
                if (configDisabled) {
                  this.onNextClick();
                } else {
                  this.useConfig();
                }
              }}
            >
              {configDisabled ? 'Next' : 'Use Config'}
            </EuiButton>
          </EuiFormRow>
        </EuiFlexItem>
        {
          configDisabled ? '' :
            (
              <EuiFlexItem grow={false}>
                <EuiFormRow>
                  <EuiButton
                    iconType="trash"
                    color="danger"
                    disabled={configDisabled || this.props.state.configOptions.length <= 0}
                    onClick={this.deleteConfig}
                  >
                      Delete Config
                  </EuiButton>
                </EuiFormRow>
              </EuiFlexItem>
            )
        }

      </EuiFlexGroup>
    );
  }

  renderModals() {
    return (
      <div>
        {this.deleteConfigModal()}
      </div>
    );
  }

  renderPreviewTable() {
    if (this.props.state.data.loaded) {
      return (
        <PreviewTable
          items={this.props.state.data.items}
          columns={this.props.state.data.columns}
        />
      );
    }
  }

  renderLoadingAnimation() {
    if (this.props.state.loading) {
      return (
        <div>
          <EuiFlexGroup justifyContent="spaceAround">
            <EuiFlexItem grow={false}>
              <EuiLoadingChart size="xl" mono/>
            </EuiFlexItem>
          </EuiFlexGroup>
          <EuiFlexGroup justifyContent="spaceAround">
            <EuiFlexItem grow={false}>
              <p>
                <EuiTextColor color="subdued">
                  Note: Page may lock up while processing large files. Excel sheets over 20MB may crash the
                  browser (Depending on your browser and machine).
                </EuiTextColor>
              </p>
            </EuiFlexItem>
          </EuiFlexGroup>
        </div>
      );
    }
  }

  renderErrorMessage() {
    if (this.props.state.error) {
      return (
        <EuiCallOut
          title="Sorry, there was an error"
          color="danger"
          iconType="cross"
        >
          <p>
            {this.props.state.error.toString()}
          </p>
        </EuiCallOut>
      );
    }
  }

  render() {
    return (
      <Fragment>
        <EuiFlexGroup gutterSize="l">
          <EuiFlexItem grow={false}>
            <EuiFormRow>
              <EuiTitle size="s">
                <h1>Import your xlsx and csv file to ElasticSearch</h1>
              </EuiTitle>
            </EuiFormRow>

            <EuiFormRow>
              <EuiFilePicker
                initialPromptText="Select or drag and drop a file"
                onChange={this.onFileChange}
              />
            </EuiFormRow>

            {this.renderControls()}

          </EuiFlexItem>

        </EuiFlexGroup>

        <EuiSpacer size="l" />

        {this.renderErrorMessage()}

        {this.renderLoadingAnimation()}

        <FileErrors errors={this.props.state.fileErrors} />

        {this.renderPreviewTable()}

        {this.renderModals()}
      </Fragment>
    );
  }
}

export default StepOne;
