import React, {
  Component, Fragment
} from 'react';

import {
  EuiForm,
  EuiFormRow,
  EuiFieldText,
  EuiSwitch,
  EuiSpacer,
  EuiFlexGroup,
  EuiFlexItem,
  EuiButton,
  EuiButtonEmpty,
  EuiGlobalToastList,
  EuiProgress,
  EuiComboBox,
  EuiTextArea,
  EuiTitle,
  EuiCheckbox,
  EuiCallOut
} from '@elastic/eui';
import Beforeunload from 'react-beforeunload';
import cloneDeep from 'lodash.clonedeep';
import MappingTable from './mapping_table';
import CustomId from './customId';
import ConditionalSwitch from './conditionalSwitch';
import ImportControls from './importControls';

import {
  createMapping,
  createKbnCustomId,
  mergeFromIndexMapping,
  retrieveMapping,
  getUpdatedMappingTableState,
  retrieveMappingFromLocalStorage,
  storeMappingToLocalStorage
} from '../services/services.js';
import { formatJSON } from '../services/sheetServices.js';
import XLSXClient from '../services/xlsx_client.js';
import {
  getDelayedTrigger,
  secondsToHms
} from '../services/helper.js';
import {
  configService,
  modalWithForm,
  PipelineDefinition
} from '../../../shared_components';

class StepTwo extends Component {

  constructor(props) {
    super(props);
    this.state = {
      progress: {
        current: 0,
        color: 'secondary'
      },
      toasts: []
    };
    this.jobStats = {};
    this.configService = configService();
    const previousMappingState = props.state.mappingInfo.length > 0;
    this.mappingInfo = previousMappingState ? props.state.mappingInfo : props.state.items;

    if (!previousMappingState && localStorage) {
      this.mappingInfo = retrieveMappingFromLocalStorage(this.mappingInfo);
    }

    this.indexNameChange = this.indexNameChange.bind(this);
    this.kbnIdChange = this.kbnIdChange.bind(this);
    this.onChangeAnonColumns = this.onChangeAnonColumns.bind(this);
    this.switchChange = this.switchChange.bind(this);
    this.onChangeMapping = this.onChangeMapping.bind(this);
    this.handleClick = this.handleClick.bind(this);
    this.handleNextStep = this.handleNextStep.bind(this);
    this.addErrorToast = this.addErrorToast.bind(this);
    this.addMappingToast = this.addMappingToast.bind(this);
    this.removeToast = this.removeToast.bind(this);
    this.onToggleIndexDeletion = this.onToggleIndexDeletion.bind(this);
    this.fetchMappingFromIndex = this.fetchMappingFromIndex.bind(this);
    this.updateMappingInfo = this.updateMappingInfo.bind(this);
    this.fieldRemovalFromMappingAction = this.fieldRemovalFromMappingAction.bind(this);
    this.clearFailedIndexing = this.clearFailedIndexing.bind(this);
    this.getUsername = this.getUsername.bind(this);
    this.onPipelineConfigChange = this.onPipelineConfigChange.bind(this);
    this.updatePipelineDef = this.updatePipelineDef.bind(this);
    this.testPipeline = this.testPipeline.bind(this);
    this.saveConfig = this.saveConfig.bind(this);
    this.preparePreviousState = this.preparePreviousState.bind(this);

    this.xlsxClient = XLSXClient.getInstance();
    this.indexNameDelayedTrigger = getDelayedTrigger();
    this.kbnIdDelayedTrigger = getDelayedTrigger();
    this.pipelineDefDelayedTrigger = getDelayedTrigger();
  }

  componentDidMount() {
    const opt = this.props.state.header.map((s) => ({
      label: s
    }));
    this.indexNameChange({ keyCode: 13,
      target: {
        value: this.props.state.indexName
      }
    });
    this.props.onFieldValueChange('options', opt);
  }

  componentWillUnmount() {
    this.abortIngestion();
  }

  onPipelineConfigChange(value, fieldName) {
    const updatePipeline = (step, state) => {
      const stepState = state[step];
      stepState.pipelineConfig[fieldName] = value;
      return {
        [step]: stepState
      };
    };
    this.props.stateUpdate(updatePipeline);
  }

  updatePipelineDef(param) {
    if (!param.trigger) {
      param.trigger = true;
      this.pipelineDefDelayedTrigger(this.updatePipelineDef, 200, param);
      return;
    }
    this.onPipelineConfigChange(param.value, 'pipeline');
  }

  async testPipeline(paramData) {
    const pipeline = this.props.state.pipelineConfig.pipeline;
    const data = paramData || this.props.state.examples[this.props.state.pipelineConfig.pipelineInput];
    return await this.xlsxClient.simulatePipeline(data, pipeline);
  }

  updateMappingInfo() {
    if (this.props.state.switchMap.value) {
      this.mappingInfo = getUpdatedMappingTableState(this.mappingInfo, this.props.state.selectedAnonOptions);
    }
  }

  async getUsername() {
    let username = null;
    if (this.props.injector.has('kibiAccessControl')) {
      const userInfo = await this.props.injector.get('kibiAccessControl').getUserInfo();
      username = userInfo ? userInfo.username : null;
    }
    return username;
  }

  async indexNameChange(e) {
    if (e.keyCode !== 13 && !(e.type === 'delayedTrigger' && !!e.target.value)) {
      this.indexNameDelayedTrigger(this.indexNameChange, 200, {
        target: {
          value: e.target.value,
        },
        type: 'delayedTrigger'
      });
      return;
    }
    this.updateMappingInfo();
    const hasInvalidCharactersRegex = /[~`!#$%\^&*+=\\[\]\\';,/{}|\\":<>\?]/g;
    const hasUpperCaseCharactersRegex = /[A-Z]/;
    if (hasInvalidCharactersRegex.test(e.target.value) || hasUpperCaseCharactersRegex.test(e.target.value)) {
      this.props.stateUpdate({ indexName: e.target.value, indexNameError: true, blockIngestion: true, indexExists: false,
        indexNameErrors: [ 'Index name must be all lowercase and not contain special characters' ] });
    }
    else {
      const indexName = e.target.value;
      const response = await this.xlsxClient.checkIndexExists(indexName);
      if (response.status === 404) {
        this.props.stateUpdate({ indexName, indexNameError: false, blockIngestion: false, indexExists: false });
      } else if (response.status === 403) {
        this.props.stateUpdate({ indexName, indexNameErrors: [response.error.reason],
          blockIngestion: true, indexNameError: true, indexExists: false });
      } else {
        this.props.stateUpdate({ indexName, indexNameError: true, indexExists: true, blockIngestion: false,
          switchMap: { value: this.props.state.deleteIndexBeforeImport || !!this.props.state.savedConfiguration.id },
          indexNameErrors: [ `Index: ${indexName} Already Exists! (Your data will be pushed into an existing Index)` ]
        });
      }
    }
  }

  kbnIdChange(e, shouldDelay) {
    if (shouldDelay) {
      this.kbnIdDelayedTrigger(this.kbnIdChange, 200, {
        target: {
          value: e.target.value,
        }
      });
      return;
    }
    this.updateMappingInfo();
    this.props.stateUpdate({
      kbnId: {
        model: e.target.value,
        preview: createKbnCustomId(e.target.value, this.props.state.examples[0])
      }
    });
  }

  onChangeAnonColumns(e) {
    this.updateMappingInfo();
    this.props.stateUpdate({ selectedAnonOptions: e });
  }

  fieldRemovalFromMappingAction(removedItem) {
    const newRemovalArray = this.props.state.selectedAnonOptions.concat({
      label: removedItem.sanitizedFieldName
    });
    this.onChangeAnonColumns(newRemovalArray);
  }

  onToggleIndexDeletion(e) {
    this.updateMappingInfo();
    this.props.stateUpdate({ deleteIndexBeforeImport: e.target.checked });
  }

  switchChange(e) {
    this.updateMappingInfo();
    this.props.stateUpdate({ switchMap: { value: e.target.checked } });
  }

  onChangeMapping(items) {
    if (!items) {
      this.addErrorToast('Items cannot be undefined!');
      console.trace();
      return;
    }
    if (!this.props.state.switchMap.value) {
      this.props.stateUpdate({ switchMap: { value: true } });
      this.mappingInfo = items;
      this.addMappingToast();
      return;
    }
    this.mappingInfo = items;
  }

  handleClick() {
    this.setState({ progress: { current: 0, color: 'secondary' } });
    this.handleNextStep();
  }

  async clearFailedIndexing() {
    if (!this.props.state.indexExists || this.props.state.deleteIndexBeforeImport) {
      this.xlsxClient.deleteIndex(this.props.state.indexName).then(response => {
        if (response.error) {
          response.error.statusCode !== 404 && this.addErrorToast(`Unable to clear ${this.props.state.indexName}`, response.error.msg);
        } else {
          this.addSuccessToast('Index deleted!', (<p>Cleared index: <b>{this.props.state.indexName}</b></p>));
        }
      });
    }
  }

  saveConfig() {
    const configuration = cloneDeep(this.preparePreviousState());
    delete configuration.pipelineConfig.pipelineResult;
    const name = this.props.state.savedConfiguration.name;
    const description = this.props.state.savedConfiguration.description;
    const onError = (error) => {
      this.addErrorToast(error.toString(), '',
        'Unable to save configuration!');
    };
    if (!this.props.state.savedConfiguration.id || this.props.state.savedConfiguration.asNew) {
      this.xlsxClient.saveConfig(configuration, name, description).then(res => {
        this.onSaveConfigChange(res._id, 'id');
        this.addSuccessToast('Config saved as new!', (<p>Switched to the new config.</p>));
      }).catch(onError);
    } else {
      this.xlsxClient.saveConfig(configuration, name, description, this.props.state.savedConfiguration.id).then(() => {
        this.addSuccessToast('Changes to config saved!');
      }).catch(onError);
    }
    this.switchSaveModal(false);
  }

  onSaveConfigChange(value, fieldName) {
    const updateSavedConfig = (step, state) => {
      const stepState = state[step];
      stepState.savedConfiguration[fieldName] = value;
      return {
        [step]: stepState
      };
    };
    this.props.stateUpdate(updateSavedConfig);
  }

  switchSaveModal = savingConfig => {
    this.updateMappingInfo();
    this.props.onFieldValueChange('savingConfig', savingConfig);
  }

  saveConfigModal() {
    if (this.props.state.savingConfig) {
      let saveAsNew;
      const title = 'Save Configuration';
      if (this.props.state.savedConfiguration.id) {
        saveAsNew = (
          <EuiFormRow>
            <EuiCheckbox
              label="Save as new?"
              checked={this.props.state.savedConfiguration.asNew}
              onChange={(e) => this.onSaveConfigChange(e.target.checked, 'asNew')}
            />
          </EuiFormRow>
        );
      }
      const form = (
        <EuiForm>
          <EuiFormRow>
            <p>If you believe you&apos;ll be ingesting more spreadsheets with exactly the same structure,
              then you may name and save this import configuration for future use.
            </p>
          </EuiFormRow>
          <EuiFormRow
            label="Name:"
          >
            <EuiFieldText
              value={this.props.state.savedConfiguration.name}
              onChange={(e) => this.onSaveConfigChange(e.target.value, 'name')}
            />
          </EuiFormRow>
          <EuiFormRow
            label="Description:"
          >
            <EuiTextArea
              rows={3}
              value={this.props.state.savedConfiguration.description}
              onChange={(e) => this.onSaveConfigChange(e.target.value, 'description')}
            />
          </EuiFormRow>
          {saveAsNew}
        </EuiForm>
      );
      const footer = (
        <div>
          <EuiButtonEmpty
            onClick={() => this.switchSaveModal(false)}
          >
            Cancel
          </EuiButtonEmpty>

          <EuiButton
            onClick={() => this.saveConfig()}
            fill
          >
            Save
          </EuiButton>
        </div>
      );
      const onClose = () => this.switchSaveModal(false);
      return modalWithForm(title, form, footer, onClose);
    }
  }

  preparePreviousState() {
    return {
      indexName: this.props.state.indexName,
      deleteIndexBeforeImport: this.props.state.deleteIndexBeforeImport,
      selectedAnonOptions: this.props.state.selectedAnonOptions,
      kbnId: {
        useKbnId: !!this.props.state.kbnId.model,
        model: this.props.state.kbnId.model,
        preview: this.props.state.kbnId.preview
      },
      switchMapValue: this.props.state.switchMap && this.props.state.switchMap.value,
      mappingInfo: this.mappingInfo,
      pipelineConfig: this.props.state.pipelineConfig
    };
  }

  indexingProgress = (status, progressTracker) => {
    this.jobStats = progressTracker(status, this.startingTime);
  };

  abortIngestion = () => {
    this.props.workbook.haltIngestion();
  };

  setImportPrompt = showImportPrompt => {
    this.props.stateUpdate({ showImportPrompt });
  }

  handleFailedIndexing = (reason) => {
    this.addErrorToast(reason);
    this.clearFailedIndexing();
    this.setState({
      progress: {
        current: 100,
        color: 'danger'
      }
    });
    this.props.stateUpdate({ uploadButton: { text: 'Import', loading: false } });
  }

  async handleNextStep(progressTracker) {
    this.updateMappingInfo();
    this.props.stateUpdate({ uploadButton: { text: 'Loading...', loading: true } });
    try {
      const indexingConfig = {
        bulkSize: this.configService.bulkPackageSize,
        indexName: this.props.state.indexName,
        username: await this.getUsername(),
        customIdModel: this.props.state.kbnId.model
      };
      if (this.props.state.selectedAnonOptions.length > 0) {
        indexingConfig.columnFilter = this.props.state.selectedAnonOptions.map((s) => (
          s.label
        ));
      }
      if (this.props.state.deleteIndexBeforeImport && this.props.state.indexExists) {
        const response = await this.xlsxClient.deleteIndex(this.props.state.indexName);
        if (response.error && response.error.statusCode !== 404) {
          this.addErrorToast(response.error.msg, 'Ingestion will still try to continue',
            'Unable to delete index');
        }
      }
      if (this.props.state.switchMap.value && (!this.props.state.indexExists || this.props.state.deleteIndexBeforeImport)) {
        console.log('Creating index', this.props.state.indexName);
        const resIndex = await this.xlsxClient.createIndex(this.props.state.indexName);
        if (resIndex.error !== undefined) {
          this.handleFailedIndexing(resIndex.error.msg);
          return;
        }
      }
      if (this.props.state.switchMap.value) {
        console.log('Applying mapping');
        const typeDropdown = document.getElementsByClassName('euiSelect');
        const properties = createMapping(this.mappingInfo, typeDropdown, this.props.state.selectedAnonOptions);
        const resMap = await this.xlsxClient.pushIndexMapping(this.props.state.indexName, properties);
        if (resMap.error !== undefined) {
          this.handleFailedIndexing(resMap.error.msg);
          return;
        }
        indexingConfig.delimiters = document.getElementsByClassName('delimiterfield');
        indexingConfig.mappingInfo = this.mappingInfo;
      }
      let pipeline = this.props.state.pipelineConfig.pipeline;
      try {
        if (pipeline && this.props.state.pipelineConfig.pipelineMethod !== 'skip') {
          pipeline = typeof pipeline === 'object' ? pipeline : JSON.parse(pipeline);
          //PUT Pipeline
          const pipelineId = `${this.props.state.indexName}-${new Date().valueOf()}`;
          await this.xlsxClient.savePipeline(pipelineId, pipeline);
          indexingConfig.pipelineId = pipelineId;
        }
      } catch (e) {
        this.handleFailedIndexing(e.message, 'Verify your Pipeline Config.');
        return;
      }

      this.startingTime = new Date();
      const indexing = this.props.workbook.ingestSheetToES(this.props.sheetname, indexingConfig, status => {
        this.indexingProgress(status, progressTracker);
      });
      indexing.then(({ errorsDuringIndexing, totalDocuments }) => {
        if (pipeline) {
          //DELETE Pipeline
          this.xlsxClient.deletePipeline(indexingConfig.pipelineId);
        }
        if (totalDocuments - errorsDuringIndexing.length > 0) {
          storeMappingToLocalStorage(this.mappingInfo, this.props.state.selectedAnonOptions); //Only if any Doc got indexed
        }
        this.props.stateUpdate({ uploadButton: { text: 'Import', loading: false } });
        this.props.nextStep({
          jobStats: this.jobStats,
          errorsDuringIndexing,
          nbDocument: totalDocuments
        });
      }, (reason) => {
        this.startingTime = null;
        if (pipeline) {
          //DELETE Pipeline
          this.xlsxClient.deletePipeline(indexingConfig.pipelineId);
        }
        this.handleFailedIndexing(reason);
      });

    } catch (error) {
      this.handleFailedIndexing(error.message, 'Please verify your advanced JSON or your fields name');
    }
  }

  async fetchMappingFromIndex() {
    this.updateMappingInfo();
    const response = await this.xlsxClient.checkIndexExists(this.props.state.indexName);
    const indexMapping = retrieveMapping(response, this.props.state.indexName);
    this.mappingInfo = mergeFromIndexMapping(this.mappingInfo, indexMapping);
    this.addSuccessToast('Mapping Fetched!', <p>Only fields with matching names are updated.</p>);
  }

  addSuccessToast(title, text) {
    const toast = {
      title,
      color: 'success',
      text,
    };
    this.concatToast(toast);
  }

  concatToast(toast) {
    this.setState((state) => {
      return {
        toasts: state.toasts.concat(toast)
      };
    });
  }

  addErrorToast = (msg, help, title) => {
    const toast = {
      title: title || 'Couldn\'t complete the import',
      color: 'danger',
      iconType: 'alert',
      text: (
        <div>
          <p>
            {msg}
          </p>
          <p>
            {help}
          </p>
        </div>
      ),
    };
    this.concatToast(toast);
  };

  addMappingToast = () => {
    const toast = {
      title: 'Mapping change detected',
      text: (
        <p>
          Custom mapping enable.
        </p>
      ),
    };
    this.concatToast(toast);
  };

  removeToast = (removedToast) => {
    this.setState((state) => {
      return {
        toasts: state.toasts.filter(toast => toast.id !== removedToast.id)
      };
    });
  };

  removeAllToasts = () => {
    this.setState({
      toasts: []
    });
  };

  renderInfoBar() {
    if (this.props.state.savedConfiguration.id) {
      return (
        <EuiCallOut
          size="s"
          color="success"
          title={`Loaded from configutation.`}
          iconType="indexMapping"
        >
          <b><p>{this.props.state.savedConfiguration.name}</p></b>
          <p>{this.props.state.savedConfiguration.description}</p>
        </EuiCallOut>
      );
    }
  }

  renderSectionHeading(heading) {
    return (
      <EuiTitle>
        <h4>
          {heading}
        </h4>
      </EuiTitle>
    );
  }

  renderModals() {
    return (
      <div>
        {this.saveConfigModal()}
      </div>
    );
  }

  renderImportControls() {
    return (
      <ImportControls
        blockIngestion={this.props.state.blockIngestion}
        handleNextStep={this.handleNextStep}
        importButton={this.props.state.uploadButton}
        previousStep={this.props.previousStep}
        switchSaveModal={this.switchSaveModal}
        abortIngestion={this.abortIngestion}
        addErrorToast={this.addErrorToast}
        progress={this.state.progress}
        showImportPrompt={this.props.state.showImportPrompt}
        setImportPrompt={this.setImportPrompt}
      />
    );
  }

  render() {
    const mappingSwitchWarning = !this.props.state.indexExists || this.props.state.deleteIndexBeforeImport ||
      !this.props.state.switchMap.value;
    const mappingSwitchError = 'Modifying Index Mapping! (Note: Mapping should NOT conflict with existing mapping!)';
    const isIndexNameInvalid = this.props.state.blockIngestion || this.props.state.indexNameError &&
      !this.props.state.deleteIndexBeforeImport;
    let fetchMapping = null;
    if (this.props.state.indexExists) {
      fetchMapping = (
        <EuiFlexItem grow={false}>
          <EuiButton onClick={this.fetchMappingFromIndex}>
            Fetch Mapping From Index
          </EuiButton>
        </EuiFlexItem>
      );
    }
    let mappingTable = null;
    if (this.props.state.switchMap && this.props.state.switchMap.value) {
      mappingTable = (<MappingTable
        items={this.mappingInfo}
        removedItems={this.props.state.selectedAnonOptions}
        examples={this.props.state.examples}
        onFieldRemoval={this.fieldRemovalFromMappingAction}
        onChangeMapping={this.onChangeMapping}
      />);
    }

    return (
      <Beforeunload onBeforeunload={()=>this.props.onBeforeUnload()}>
        <EuiForm>
          {this.renderInfoBar()}
          {this.renderSectionHeading('Transform Pipeline')}
          <PipelineDefinition
            documents={this.props.state.examples}
            configuration={this.props.state.pipelineConfig}
            updatePipelineDef={this.updatePipelineDef}
            onConfigChange={this.onPipelineConfigChange}
            testPipeline={this.testPipeline}
          />
          {this.renderModals()}
          <EuiSpacer size="s" />
          {this.renderSectionHeading('Indexing Settings')}
          <EuiFormRow
            isInvalid={isIndexNameInvalid}
            label="Index name"
            fullWidth={true}
            error={this.props.state.indexNameErrors}
          >
            <EuiFlexGroup alignItems="center">
              <EuiFlexItem grow={false}>
                <EuiFieldText
                  isInvalid={isIndexNameInvalid}
                  id="indexName"
                  defaultValue={this.props.state.indexName}
                  onKeyUp={this.indexNameChange}
                  onChange={this.indexNameChange}
                />
              </EuiFlexItem>
              {fetchMapping}
              <EuiFlexItem>
                <EuiFormRow>
                  <ConditionalSwitch
                    onToggle={this.onToggleIndexDeletion}
                    condition={this.props.state.indexExists}
                    checked={this.props.state.deleteIndexBeforeImport}
                    label="Delete Index Before Importing?"
                  />
                </EuiFormRow>
              </EuiFlexItem>
            </EuiFlexGroup>
          </EuiFormRow>
          <CustomId
            kbnIdModel={this.props.state.kbnId.model}
            kbnIdChange={this.kbnIdChange}
            useKbnId={this.props.state.kbnId.useKbnId || !!this.props.state.kbnId.model}
            kbnIdPreview={this.props.state.kbnId.preview}
          />
          <EuiSpacer size="s" />
          <EuiFormRow label="Remove columns">
            <EuiComboBox
              options={this.props.state.options}
              selectedOptions={this.props.state.selectedAnonOptions}
              onChange={this.onChangeAnonColumns}
            />
          </EuiFormRow>

          <EuiSpacer size="s" />
          {this.renderSectionHeading('Mapping')}
          <EuiFormRow fullWidth={true} isInvalid={!mappingSwitchWarning}>
            <div>
              <EuiSpacer size="m" />
              <EuiFormRow
                helpText="Define Index Mapping."
                isInvalid={!mappingSwitchWarning}
                error={mappingSwitchError}
              >
                <EuiSwitch label="Use Custom Mapping?" checked={this.props.state.switchMap.value} onChange={this.switchChange}/>
              </EuiFormRow>
              {mappingTable}
            </div>
          </EuiFormRow>

          {this.renderImportControls()}

          <EuiGlobalToastList
            toasts={this.state.toasts}
            dismissToast={this.removeToast}
            toastLifeTimeMs={6000}
          />
        </EuiForm>
      </Beforeunload>
    );
  }
}

export default StepTwo;
