import PapaParseDao from '../pp_dao';
import sinon from 'sinon';
import expect from 'expect.js';
import { initConfigService } from '../../../../test_utils';
import { sanitizeFieldName } from '../../services/services';

describe('ingest pp_dao', () => {
  const testIntantiateConfig = {
    worker: false
  };
  const configService = initConfigService();
  const smallCsvHeader = ['A', 'B', 'C'];
  const smallCsvRow = ['X', 'Y', 'Z'];
  const smallCsvFileName = 'sample';

  it('small csv parses', (done) => {
    const smallCsvOneRow = new File([`${smallCsvHeader}\n${smallCsvRow}`], `${smallCsvFileName}.csv`);
    const papaParseDao = new PapaParseDao(smallCsvOneRow);
    papaParseDao.instantiate((errors) => {
      expect(errors.length).to.be(0);
      expect(papaParseDao.getHeaderRow(smallCsvFileName, false)).to.eql(smallCsvHeader);
      const sheetJson = papaParseDao.sheetToJSON(smallCsvFileName);
      expect(sheetJson.length).to.be(1);
      const expectedTuple = {
        A: 'X',
        B: 'Y',
        C: 'Z'
      };
      expect(sheetJson[0]).to.eql(expectedTuple);
      done();
    }, testIntantiateConfig);
  });

  describe('formatting', () => {
    const header = ['row one', 'row two', 'row three'];
    const csvWithSpacedHeaders = new File([`${header}\n${smallCsvRow}`], `${smallCsvFileName}.csv`);
    const result = {
      meta: {
        fields: header
      },
      errors: [],
      data: [
        {
          'row one': 'X',
          'row two': 'Y',
          'row three': 'Z'
        }
      ]
    };
    const expectedRow = {
      row_one: 'X',
      row_two: 'Y',
      row_three: 'Z'
    };
    let papaParseDao;

    beforeEach(() => {
      papaParseDao = new PapaParseDao(csvWithSpacedHeaders);
      papaParseDao._instantiationDone(() => {}, result);
    });

    it('header is sanitized', () => {
      expect(papaParseDao.getHeaderRow(smallCsvFileName, false)).to.eql(header.map((ele) => sanitizeFieldName(ele)));
    });

    it('sheetJson is formatted', () => {
      expect(papaParseDao.sheetToJSON(smallCsvFileName)).to.eql([expectedRow]);
    });
  });


  describe('streaming', () => {
    let fileData = `${smallCsvHeader}\n`;
    const tupleCount = 50;
    for (let i = 0; i < tupleCount; i++) {
      fileData += `${smallCsvRow}\n`;
    }
    const csvFile = new File([fileData], `${smallCsvFileName}.csv`);
    let papaParseDao;

    beforeEach(() => {
      papaParseDao = new PapaParseDao(csvFile);
      papaParseDao.streamFile = true;
    });

    it('_instantiationDone called only once', (done) => {
      const _instantiationDoneSpy = sinon.spy(papaParseDao, '_instantiationDone');
      papaParseDao.instantiate((errors) => {
        expect(errors.length).to.be(0);
        expect(_instantiationDoneSpy.calledOnce);
        done();
      }, testIntantiateConfig);
    });

    it('_setMetaFields called with correct fileData', (done) => {
      const _setMetaFieldsSpy = sinon.spy(papaParseDao, '_setMetaFields');
      papaParseDao.instantiate((errors) => {
        expect(errors.length).to.be(0);
        expect(_setMetaFieldsSpy.calledOnce);
        expect(_setMetaFieldsSpy.firstCall.args[0].fields).to.eql(smallCsvHeader);
        done();
      }, testIntantiateConfig);
    });

    it('_previewStepper called previewSize times', (done) => {
      const _previewStepperSpy = sinon.spy(papaParseDao, '_previewStepper');
      papaParseDao.instantiate((errors) => {
        expect(errors.length).to.be(0);
        expect(_previewStepperSpy.callCount).to.be(papaParseDao.previewSize);
        done();
      }, testIntantiateConfig);
    });

    it('file only parsed till previewSize', (done) => {
      papaParseDao.instantiate((errors) => {
        expect(errors.length).to.be(0);
        const sheetJson = papaParseDao.sheetToJSON(smallCsvFileName);
        expect(sheetJson.length).to.be(papaParseDao.previewSize);
        const expectedSheetJson = [];
        const expectedTuple = {
          A: 'X',
          B: 'Y',
          C: 'Z'
        };
        for (let i = 0; i < papaParseDao.previewSize; i++) {
          expectedSheetJson.push(expectedTuple);
        }
        expect(sheetJson).to.eql(expectedSheetJson);
        done();
      }, testIntantiateConfig);
    });

    it('correct header', (done) => {
      papaParseDao.instantiate((errors) => {
        expect(errors.length).to.be(0);
        expect(papaParseDao.getHeaderRow(smallCsvFileName, false)).to.eql(smallCsvHeader);
        done();
      }, testIntantiateConfig);
    });
  });
});