import SpreadsheetDao from '../spreadsheet_dao';
import sinon from 'sinon';
import expect from 'expect.js';
import XLSXClient from '../../services/xlsx_client.js';
import { initConfigService } from '../../../../test_utils';
import { testDoc } from './data';

describe('ingest spreadsheet_dao', () => {
  const configService = initConfigService();
  const CLUSTER_BLOCK_EXCEPTION = 'cluster_block_exception';
  const dummyFile = new File(['A,B,C\nX,Y,Z'], 'sample.csv');
  const spreadsheetDao = new SpreadsheetDao(dummyFile);
  const xlsxClient = XLSXClient.getInstance();
  const indexingConfig = {
    indexName: 'test-index',
    pipelineId: 'test-pipeline'
  };
  const networkErrorResponse = {
    error: {
      message: 'Network failure!'
    }
  };

  const progressCallbackSpy = sinon.spy();
  const bulkIndexStub = sinon.stub(xlsxClient, 'bulkIndex');

  it('constructor initializes file data', () => {
    expect(spreadsheetDao.file).to.equal(dummyFile);
    expect(spreadsheetDao.fileName).to.equal('sample');
    expect(spreadsheetDao.fileExtension).to.equal('csv');
    expect(spreadsheetDao.largeFile).to.equal(configService.largeFile);
    expect(spreadsheetDao.maxRowsToParse).to.equal(configService.maxRowsToParse);
    expect(spreadsheetDao.simultaneousBulkRequests).to.equal(configService.simultaneousBulkRequests);
  });

  describe('_ingestToES', () => {
    const bulkRequestObj = [
      'Dummy bulkRequestBatch', 'Dummy bulkRequestBatch', 'Dummy bulkRequestBatch',
      'Dummy bulkRequestBatch', 'Dummy bulkRequestBatch', 'Dummy bulkRequestBatch',
      'Dummy bulkRequestBatch', 'Dummy bulkRequestBatch', 'Dummy bulkRequestBatch'
    ];
    beforeEach(() => {
      progressCallbackSpy.resetHistory();
      bulkIndexStub.reset();
    });
    it('_ingestBatchToES is only called simultaneousBulkRequests times upon failure in first batch', async () => {

      bulkIndexStub.returns(Promise.resolve(networkErrorResponse));
      try {
        await spreadsheetDao._ingestToES([], bulkRequestObj, indexingConfig, progressCallbackSpy);
        expect().fail('Expected _ingestToES to throw an Error');
      } catch (reason) {
        expect(reason).to.equal(networkErrorResponse.error.message);
      }
      expect(bulkIndexStub.callCount).to.be(configService.simultaneousBulkRequests);
    });
  });

  describe('_getFilteredSheetJson', () => {
    it('filters documents based on column filter', () => {
      let sheetJson = [{
        ...testDoc,
        ...{
          unwantedField1: 'unwanted value 1',
          unwantedField2: 'unwanted value 2',
        }
      }];
      const columnFilter = [
        'unwantedField1',
        'unwantedField2'
      ];
      sheetJson = spreadsheetDao._getFilteredSheetJson(sheetJson, columnFilter);
      expect(sheetJson.length).to.be(1);
      expect(sheetJson[0]).to.eql(testDoc);
      sheetJson.forEach((tuple) =>
        columnFilter.forEach((property) =>
          expect(tuple).to.not.have.property(property)
        )
      );
    });
  });

  describe('_ingestBatchToES', () => {
    const bulkRequestBatch = 'Dummy bulkRequestBatch';
    const resolveSpy = sinon.spy();
    const rejectSpy = sinon.spy();
    let indexingResult;
    beforeEach(() => {
      indexingResult = {
        errorsDuringIndexing: []
      };
      progressCallbackSpy.resetHistory();
      resolveSpy.resetHistory();
      rejectSpy.resetHistory();
      bulkIndexStub.reset();
    });
    it('partially resolves ingestion in case of CLUSTER_BLOCK_EXCEPTION', async () => {
      const response = {
        errors: true,
        items: [
          {
            index: {
              error: {
                type: CLUSTER_BLOCK_EXCEPTION
              },
              status: 403,
            }
          }
        ]
      };
      bulkIndexStub.returns(Promise.resolve(response));
      await spreadsheetDao._ingestBatchToES(indexingConfig, bulkRequestBatch, indexingResult, progressCallbackSpy, resolveSpy, rejectSpy);
      expect(bulkIndexStub.calledOnce);
      expect(indexingResult.errorsDuringIndexing).to.have.length(1);
      expect(resolveSpy.calledOnce);
      expect(resolveSpy.firstCall.args).to.eql([indexingResult]);
      expect(rejectSpy.notCalled);
      expect(progressCallbackSpy.notCalled);
    });

    it('rejects ingestion in case of network error', async () => {
      bulkIndexStub.returns(Promise.resolve(networkErrorResponse));
      await spreadsheetDao._ingestBatchToES(indexingConfig, bulkRequestBatch, indexingResult, progressCallbackSpy, resolveSpy, rejectSpy);
      expect(bulkIndexStub.calledOnce);
      expect(rejectSpy.calledOnce);
      expect(rejectSpy.firstCall.args[0]).to.equal(networkErrorResponse.error.message);
      expect(resolveSpy.notCalled);
      expect(progressCallbackSpy.calledOnce);
      expect(progressCallbackSpy.firstCall.args[0].networkError).to.be(true);
    });

    it('bulkIndex called with correct parameters', async () => {
      const response = {};
      bulkIndexStub.returns(Promise.resolve(response));
      await spreadsheetDao._ingestBatchToES(indexingConfig, bulkRequestBatch, indexingResult, progressCallbackSpy, resolveSpy, rejectSpy);
      expect(bulkIndexStub.calledOnce);
      expect(bulkIndexStub.firstCall.args).to.eql([indexingConfig.indexName, bulkRequestBatch, indexingConfig.pipelineId]);
      expect(rejectSpy.notCalled);
      expect(resolveSpy.notCalled);
      expect(progressCallbackSpy.notCalled);
    });
  });
});