import SpreadsheetDao from './spreadsheet_dao';
import XLSX from 'xlsx';
import {
  getHeaderRow,
  getHeaderWithType,
  formatJSON
} from '../services/sheetServices.js';

export default class XlsxDao extends SpreadsheetDao {
  constructor(file) {
    super(file);
    this.dateFormat = 'YYYY-MM-DD"T"hh:mm:ss';
    this.isWorkerSupported = typeof Worker !== 'undefined';
    this.readAsBinaryString = typeof FileReader !== 'undefined' && FileReader.prototype
      && FileReader.prototype.readAsBinaryString;
  }

  instantiate(done) {
    const reader = new FileReader();
    reader.onload = this._onFileLoad.bind(this, done);
    if (this.readAsBinaryString) {
      reader.readAsBinaryString(this.file);
    } else {
      reader.readAsArrayBuffer(this.file);
    }
  }

  async _onFileLoad(done, event) {
    try {
      const result = event.target.result;
      const readType = this.readAsBinaryString ? 'binary' : 'base64';
      if (this.isWorkerSupported) {
        const worker = new Worker(this.xlsxClient.getWorkerPath());
        worker.onmessage = this._onWorkerMessage.bind(this, done);
        worker.postMessage({
          sheet: result,
          readType,
          status: 'success',
          dateFormat: this.dateFormat,
          xlsxDist: this.xlsxClient.getXlsxDistPath()
        });
      } else {
        this.workbook = await XLSX.read(result, { type: readType, cellDates: true,
          dateNF: this.dateFormat });
        done();
      }
    } catch (error) {
      done(error);
    }
  }

  _onWorkerMessage(done, event) {
    try {
      switch (event.data.status) {
        case 'ready':
          break;
        case 'e': throw new Error('Error @Worker: ' + event.data.res);
          break;
        case 'success':
          this.workbook = JSON.parse(event.data.res);
          done();
          break;
      }
    } catch (e) {
      done(e);
    }
  }

  getSheetNames() {
    return this.workbook.SheetNames.map((s) => ({
      value: s,
      text: s
    }));
  }

  sheetExists(sheetName) {
    return this.workbook.Sheets[sheetName]['!ref'] !== undefined;
  }

  getHeaderRow(sheetName, mapped = true) {
    if (mapped) {
      return getHeaderRow(this.workbook.Sheets[sheetName]).map((s) => ({
        field: s,
        name: s,
        truncateText: true
      }));
    }
    return getHeaderRow(this.workbook.Sheets[sheetName]);
  }

  getHeaderWithType(sheetName) {
    return getHeaderWithType(this.workbook.Sheets[sheetName], this.maxRowsToParse);
  }

  sheetToJSON(sheetName, previewSize) {
    const range = XLSX.utils.decode_range(this.workbook.Sheets[sheetName]['!ref']);
    if (previewSize && range.e.r > previewSize) {
      range.e.r = previewSize;
    }
    return formatJSON(
      XLSX.utils.sheet_to_json(this.workbook.Sheets[sheetName],
        { range, raw: true, dateNF: this.dateFormat })
    );
  }

  _ingestSheetToES(sheetName, indexingConfig, progressCallback) {
    const worksheet = this.workbook.Sheets[sheetName];
    try {
      const filteredSheetJson = this._getFilteredSheetJson(this.sheetToJSON(sheetName), indexingConfig.columnFilter);
      const bulkRequestObj = this._createBulkRequest(filteredSheetJson, indexingConfig);
      return this._ingestToES(filteredSheetJson, bulkRequestObj, indexingConfig, progressCallback);
    } catch (error) {
      return Promise.reject(error);
    }
  }
};