import {
  createBulk,
  createMapping,
  createKbnCustomId,
  //generateExample,
  //storeMappingState,
  sanitizeFieldName,
  retrieveMapping,
  mergeFromIndexMapping,
  //getUpdatedMappingTableState,
  //retrieveMappingFromLocalStorage,
  storeMappingToLocalStorage
} from '../services';

import {
  testDoc,
  fields,
  mappingInfo,
  retrieveMappingTestData,
  mergeFromIndexMappingTestData,
  createBulkTestData,
  createMappingTestData
} from './data';
import _ from 'lodash';
import expect from 'expect.js';
import sinon from 'sinon';

describe('ingest services', () => {

  describe('createKbnCustomId', () => {
    it('single field', () => {
      for (let i = 0; i < fields.length; i++) {
        const template = `{${fields[i]}}`;
        const expectedResponse = testDoc[sanitizeFieldName(fields[i])] || fields[i];
        expect(createKbnCustomId(template, testDoc)).to.equal(String(expectedResponse));
      }
    });
    it('multiple fields', () => {
      let template = '';
      let expectedResponse = '';
      for (let i = 0; i < fields.length; i++) {
        template += `{${fields[i]}}`;
        expectedResponse += testDoc[sanitizeFieldName(fields[i])] || fields[i];
      }
      expect(createKbnCustomId(template, testDoc)).to.equal(String(expectedResponse));
    });
  });

  describe('retrieveMapping', () => {
    it('works', () => {
      retrieveMappingTestData.forEach((test) => {
        expect(retrieveMapping(test.responseFromES, test.indexName)).to.eql(test.expectedReturn);
      });
    });
  });

  it('sanitizeFieldName works', () => {
    const inputArray = [
      '.field-name',
      'My Field'
    ];
    const expectedReturn = [
      '.field-name',
      'My_Field'
    ];
    inputArray.forEach((ele, i) => expect(sanitizeFieldName(ele)).to.equal(expectedReturn[i]));
  });

  describe('storeMappingToLocalStorage', () => {
    const removedFields = [{
      label: mappingInfo[0].sanitizedFieldName
    }];
    beforeEach(() => {
      localStorage.clear();
    });
    it('stores mappingInfo', () => {
      storeMappingToLocalStorage(mappingInfo);
      const localMappingInfo = JSON.parse(localStorage.getItem('mappingInfo'));
      mappingInfo.forEach((ele) => {
        const { sanitizedFieldName, ...expectedFieldMapping } = ele;
        expect(localMappingInfo[sanitizedFieldName]).to.eql(expectedFieldMapping);
      });
    });
    it('skips removed fields', () => {
      storeMappingToLocalStorage(mappingInfo, removedFields);
      const localMappingInfo = JSON.parse(localStorage.getItem('mappingInfo'));
      mappingInfo.slice(1).forEach((ele) => {
        const { sanitizedFieldName, ...expectedFieldMapping } = ele;
        expect(localMappingInfo[sanitizedFieldName]).to.eql(expectedFieldMapping);
      });
      removedFields.forEach((ele) => {
        expect(!localMappingInfo[ele.label]).to.equal(true);
      });
    });
    it('does not clear existing (unrelated/removed) field mappings', () => {
      const { sanitizedFieldName, ...expectedFieldMapping } = mappingInfo[0];
      localStorage.setItem('mappingInfo', JSON.stringify({
        [sanitizedFieldName]: expectedFieldMapping
      }));
      storeMappingToLocalStorage(mappingInfo, removedFields);
      const localMappingInfo = JSON.parse(localStorage.getItem('mappingInfo'));
      removedFields.forEach((ele) => {
        expect(localMappingInfo[ele.label]).to.eql(expectedFieldMapping);
      });
    });
  });

  it('mergeFromIndexMapping', () => {
    mergeFromIndexMappingTestData.forEach((test) => {
      expect(mergeFromIndexMapping(test.mappingInfo, test.indexMappingJson)).to.eql(test.expectedReturn);
    });
  });

  describe('createBulk', () => {
    let clock;
    before(() => clock = sinon.useFakeTimers(1548959400000));
    after(() => clock.restore());
    function getClone(data) {
      return _.cloneDeep(data);
    };
    function testRequestDimensions(bulkRequest, testData) {
      expect(bulkRequest.length).to.be(testData.expected.bulkRequestLength);
      bulkRequest.forEach((bulkPackage, packageIndex) => {
        expect(bulkPackage.length).to.be(testData.expected.bulkPackageLength[packageIndex]);
      });
    };
    it('@timestamp is added to every document', () => {
      createBulkTestData.forEach((testData) => {
        const bulkRequest = createBulk(getClone(testData.json), false, testData.bulkSize);
        testRequestDimensions(bulkRequest, testData);
        bulkRequest.forEach((bulkPackage) => bulkPackage.forEach((ele, index) => {
          if (index % 2 !== 0) {
            expect(ele['@timestamp'].constructor.name).to.equal('Date');
          }
        }));
      });
    });
    it('_sirenImportUser is added to every document', () => {
      createBulkTestData.forEach((testData) => {
        const bulkRequest = createBulk(getClone(testData.json), false, testData.bulkSize, testData.username);
        testRequestDimensions(bulkRequest, testData);
        bulkRequest.forEach((bulkPackage) => bulkPackage.forEach((ele, index) => {
          if (index % 2 !== 0) {
            expect(ele._sirenImportUser).to.equal(testData.username);
          }
        }));
      });
    });
    it('kbnCustomId is used to generate _id(s)', () => {
      createBulkTestData.forEach((testData) => {
        const bulkRequest = createBulk(getClone(testData.json), testData.kbnCustomId, testData.bulkSize);
        testRequestDimensions(bulkRequest, testData);
        let docIndex = 0;
        bulkRequest.forEach((bulkPackage) => bulkPackage.forEach((ele, index) => {
          if (index % 2 === 0) {
            expect(ele.index._id).to.equal(testData.expected.customIds[docIndex++]);
          }
        }));
      });
    });
    it('delimiters work - commaSeparated', () => {
      createBulkTestData.forEach((testData) => {
        const bulkRequest = createBulk(getClone(testData.json), false, testData.bulkSize, '', true, testData.mappingItems.splitByComma);
        testRequestDimensions(bulkRequest, testData);
        let docIndex = 0;
        bulkRequest.forEach((bulkPackage) => bulkPackage.forEach((ele, index) => {
          if (index % 2 !== 0) {
            expect(ele.multiValueField).to.eql(testData.expected.splitFieldByComma[docIndex++]);
          }
        }));
      });
    });
    it('delimiters work - json parsing', () => {
      createBulkTestData.forEach((testData) => {
        const bulkRequest = createBulk(getClone(testData.json), false, testData.bulkSize, '', true, testData.mappingItems.jsonParsing);
        testRequestDimensions(bulkRequest, testData);
        let docIndex = 0;
        bulkRequest.forEach((bulkPackage) => bulkPackage.forEach((ele, index) => {
          if (index % 2 !== 0) {
            expect(ele.jsonField).to.eql(testData.expected.jsonParsedField[docIndex++]);
          }
        }));
      });
    });
    it('everything works with all custom settings', () => {
      createBulkTestData.forEach((testData) => {
        const bulkRequest = createBulk(getClone(testData.json), testData.kbnCustomId, testData.bulkSize,
          testData.username, true, testData.mappingItems.all);
        testRequestDimensions(bulkRequest, testData);
        expect(bulkRequest).to.eql(testData.expected.everythingWorks);
      });
    });
    it('everything works with NO custom settings', () => {
      createBulkTestData.forEach((testData) => {
        const bulkRequest = createBulk(getClone(testData.json), false, testData.bulkSize);
        testRequestDimensions(bulkRequest, testData);
        expect(bulkRequest).to.eql(testData.expected.withoutCustomSettings);
      });
    });
  });

  describe('createMapping', () => {
    createMappingTestData.forEach((testData, index) => {
      const createdMapping = createMapping(testData.items, testData.euiSelect, testData.removedFields);
      if (testData.removedFields && testData.removedFields.length > 0) {
        it(`testData[${index}]: removedFields do NOT exist in mapping`, () => {
          testData.removedFields.forEach(({ label }) => {
            expect(createdMapping.properties).to.not.have.property(label);
          });
        });
      }
      it(`testData[${index}]: correct mapping is created`, () => {
        expect(createdMapping).to.eql(testData.expected);
      });
    });
  });

});