import _ from 'lodash';
import { getESType } from '../../../shared_components';

const sanitizeCache = {};

function sanitizeFieldName(fieldName) {
  if (sanitizeCache[fieldName]) {
    return sanitizeCache[fieldName];
  }
  sanitizeCache[fieldName] = fieldName.replace(/\s+/g, '_');
  return sanitizeCache[fieldName];
}

function retrieveMapping(response, indexName) {
  const indexMapping = _.cloneDeep(indexName ? _.get(response, [indexName, 'mappings', 'doc', 'properties'], {}) : response);
  let fieldMappingJson = {};
  for (const key in indexMapping) {
    if (indexMapping.hasOwnProperty(key)) {
      if (indexMapping[key].properties) {
        const subFieldMappingJson = retrieveMapping(indexMapping[key].properties);
        for (const prop in subFieldMappingJson) {
          if (subFieldMappingJson.hasOwnProperty(prop)) {
            const tempEle = {};
            tempEle[`${key}.${prop}`] = subFieldMappingJson[prop];
            fieldMappingJson = ({ ...fieldMappingJson, ...tempEle });
          }
        }
      } else {
        const tempEle = {};
        tempEle[key] = {
          type: indexMapping[key].type
        };
        delete indexMapping[key].type;
        let morePropsExist = false;
        for (const prop in indexMapping[key]) {
          if (indexMapping[key].hasOwnProperty(prop)) {
            morePropsExist = true;
            break;
          }
        }
        if (morePropsExist) {
          tempEle[key].useAdv = true;
          tempEle[key].advJson = indexMapping[key];
        }
        fieldMappingJson = ({ ...fieldMappingJson, ...tempEle });
      }
    }
  }
  return fieldMappingJson;
}

function retrieveMappingFromLocalStorage(mappingInfo) {
  if (localStorage) {
    return mergeFromIndexMapping(mappingInfo, JSON.parse(localStorage.getItem('mappingInfo')) || {});
  }
  return mappingInfo;
}

function storeMappingToLocalStorage(mappingInfo, removedFields = []) {
  if (localStorage) {
    try {
      const mappingJson = {};
      const removedFieldsJson = {};
      removedFields.forEach(ele => {
        removedFieldsJson[ele.label] = true;
      });
      for (let i = 0; i < mappingInfo.length; i++) {
        const {
          sanitizedFieldName,
          name,
          type,
          multiValued,
          delimiter,
          advJson,
          useAdv
        } = mappingInfo[i];
        if (removedFieldsJson[sanitizedFieldName]) {
          continue;
        }
        mappingJson[sanitizedFieldName] = {
          name,
          type,
          multiValued,
          delimiter,
          advJson,
          useAdv
        };
      }
      const existingMapping = JSON.parse(localStorage.getItem('mappingInfo'));
      localStorage.setItem('mappingInfo', JSON.stringify(
        ({ ...existingMapping, ...mappingJson })
      ));
    } catch (e) {
      console.log(e);
    }
  }
}

function mergeFromIndexMapping(mappingInfo, indexMappingJson) {
  const existingMapping = _.cloneDeep(mappingInfo);
  for (let i = 0; i < existingMapping.length; i++) {
    const fieldName = existingMapping[i].sanitizedFieldName;
    if (indexMappingJson[fieldName]) {
      existingMapping[i] = ({ ...existingMapping[i], ...indexMappingJson[fieldName] });
    }
  }
  return existingMapping;
}

function createBulk(json, kbnCustomId, BULKSIZE, username, useDelimiters, mappingItems) {
  BULKSIZE = BULKSIZE * 2; //Accounting for action data
  const bulkRequest = [];
  let bulkPackage = [];

  if (useDelimiters) {
    const processField = (ele, fieldName, splitBy) => {
      let fieldValue = ele[fieldName];
      try {
        if (splitBy === 'JSON') {
          fieldValue = fieldValue && JSON.parse(fieldValue);
        } else {
          fieldValue = fieldValue && fieldValue.toString().split(splitBy).map(item => item.trim());
        }
      } catch (err) {
        //DO Nothing, the value of field remains unprocessed
        //Probable cause:- This field wasn't JSON.
      }
      ele[fieldName] = fieldValue;
      return ele;
    };
    for (let k = 0; k < mappingItems.length; k++) {
      if (mappingItems[k].multiValued) {
        const splitBy = mappingItems[k].delimiter;
        if (splitBy && splitBy !== '') {
          const fieldName = mappingItems[k].sanitizedFieldName;
          json = json.map((ele) => processField(ele, fieldName, splitBy));
        }
      }
    }
  }
  const timestamp = new Date();
  for (let i = 0; i < json.length; i++) {
    json[i]['@timestamp'] = timestamp;
    json[i]._sirenImportUser = username;
    if (kbnCustomId) {
      bulkPackage.push({ index: { _id: createKbnCustomId(kbnCustomId, json[i]) } });
    }
    else {
      bulkPackage.push({ index: {} });
    }
    Object.keys(json[i]).forEach(k => json[i][k] === '' && delete json[i][k]); //Clear Empty Strings
    bulkPackage.push(json[i]);

    if (bulkPackage.length >= BULKSIZE) {
      bulkRequest.push(bulkPackage);
      bulkPackage = [];
      if (json[i + 1] === undefined)
      {return bulkRequest;}
    }
  }
  bulkRequest.push(bulkPackage);

  return bulkRequest;
}


function createMapping(items, euiSelect, removedFields = []) {
  const types = [];
  const mappingParameters = [];
  const removedFieldsJson = {};
  removedFields.forEach(ele => {
    removedFieldsJson[ele.label] = true;
  });
  let mappingTableIndex = 0;
  const properties = {};
  for (let i = 0; i < items.length; i++) {
    if (!items[i].type) {
      items[i].type = euiSelect[i].value;
    }
    if (removedFieldsJson[items[i].sanitizedFieldName]) {
      continue;
    }
    types.push(items[i].type);
    if (items[i].useAdv) {
      mappingParameters.push(items[i].advJson);
    } else {
      mappingParameters.push('{}');
    }

    properties[items[i].sanitizedFieldName] = { 'type': getESType(types[mappingTableIndex]) };

    if (mappingParameters[mappingTableIndex]) {
      properties[items[i].sanitizedFieldName] = ({ ...properties[items[i].sanitizedFieldName],
        ...JSON.parse(mappingParameters[mappingTableIndex]) });
    }
    mappingTableIndex++;
  }

  return { 'properties': properties };
}

function createKbnCustomId(template, obj) {
  const getFromBetween = {
    results: [],
    string: '',
    getFromBetween: function (sub1, sub2) {
      if (this.string.indexOf(sub1) < 0 || this.string.indexOf(sub2) < 0) return false;
      const SP = this.string.indexOf(sub1) + sub1.length;
      const string1 = this.string.substr(0, SP);
      const string2 = this.string.substr(SP);
      const TP = string1.length + string2.indexOf(sub2);
      return this.string.substring(SP, TP);
    },
    removeFromBetween: function (sub1, sub2) {
      if (this.string.indexOf(sub1) < 0 || this.string.indexOf(sub2) < 0) return false;
      const removal = sub1 + this.getFromBetween(sub1, sub2) + sub2;
      this.string = this.string.replace(removal, '');
    },
    getAllResults: function (sub1, sub2) {
      // first check to see if we do have both substrings
      if (this.string.indexOf(sub1) < 0 || this.string.indexOf(sub2) < 0) return;

      // find one result
      const result = this.getFromBetween(sub1, sub2);
      // push it to the results array
      this.results.push(result);
      // remove the most recently found one from the string
      this.removeFromBetween(sub1, sub2);

      // if there's more substrings
      if (this.string.indexOf(sub1) > -1 && this.string.indexOf(sub2) > -1) {
        this.getAllResults(sub1, sub2);
      }
      else return;
    },
    get: function (string, sub1, sub2) {
      this.results = [];
      this.string = string;
      this.getAllResults(sub1, sub2);
      return this.results;
    }
  };
  const keys = getFromBetween.get(template, '{', '}');

  keys.forEach(function (key) {
    const sanitizedKey = sanitizeFieldName(key);
    if (obj[sanitizedKey] !== undefined)
    {template = template.replace(`{${key}}`, obj[sanitizedKey]);}
    else
    {template = template.replace(`{${key}}`, key);}
  });

  return template;
}

function getUpdatedMappingTableState(mappingInfo, removedItems = []) {
  const mappingInfoClone = _.cloneDeep(mappingInfo);
  const useAdvArray = document.querySelectorAll('[name="useAdv"]');
  const advJsonArray = document.querySelectorAll('[name="advJson"]');
  let advJsonCounter = 0;
  const typeArray = document.querySelectorAll('[name="type"]');
  const multiValuedArray = document.querySelectorAll('[name="multiValued"]');
  const delimiterArray = document.querySelectorAll('[name="delimiter"]');
  let delimiterCounter = 0;
  const removedItemsJson = {};
  removedItems.forEach(ele => {
    removedItemsJson[ele.label] = true;
  });
  let mappingTableIndex = 0;
  for (let i = 0; i < mappingInfoClone.length; i++) {
    const currentItem = mappingInfoClone[i];
    if (removedItemsJson[currentItem.sanitizedFieldName]) {
      continue;
    }
    currentItem.type = typeArray[mappingTableIndex].value;
    currentItem.multiValued = multiValuedArray[mappingTableIndex] ? multiValuedArray[mappingTableIndex].checked : false;
    if (currentItem.multiValued) {
      currentItem.delimiter = delimiterArray[delimiterCounter++].value;
    } else {
      currentItem.delimiter = null;
    }
    currentItem.tokenize = null;
    currentItem.useAdv = useAdvArray[mappingTableIndex].checked;
    if (currentItem.useAdv) {
      currentItem.advJson = advJsonArray[advJsonCounter++].value;
    } else {
      currentItem.advJson = null;
    }
    mappingTableIndex++;
  }
  return mappingInfoClone;
}

export {
  createBulk,
  createMapping,
  createKbnCustomId,
  sanitizeFieldName,
  retrieveMapping,
  mergeFromIndexMapping,
  getUpdatedMappingTableState,
  retrieveMappingFromLocalStorage,
  storeMappingToLocalStorage
};
