/* eslint-disable import/no-unresolved */
import chrome from 'ui/chrome';

class XLSXClient {

  constructor(httpClient) {
    this.BASE_PATH = chrome.getBasePath();
    this.HOST_PATH = `${this.BASE_PATH}${chrome.getInjected('xlsxHostPath')}`;
    this.API_PATH = `${this.BASE_PATH}${chrome.getInjected('xlsxApiPath')}`;
    this.httpClient = httpClient;
  }

  getWorkerPath() {
    return `${this.HOST_PATH}worker.js`;
  }

  getXlsxDistPath() {
    return `${this.HOST_PATH}xlsx.full.min.js`;
  }

  getPapaParseDistPath() {
    return `${this.HOST_PATH}papaparse.min.js`;
  }

  async call(url, method, data = null) {
    const response = await this.httpClient({
      method,
      url,
      data
    });
    return response.data;
  }

  async getConfigList() {
    return await this.call(`${this.API_PATH}list/configs`, 'GET');
  }

  async getConfig(configId) {
    return await this.call(`${this.API_PATH}configs/${configId}`, 'GET');
  }

  async saveConfig(configuration, name, description, configId) {
    let api;
    if (!configId) {
      api = `${this.API_PATH}config`;
    } else {
      api = `${this.API_PATH}configs/${configId}`;
    }
    return await this.call(api, 'POST', {
      state: configuration,
      name,
      description
    });
  }

  async deleteConfig(configId) {
    return await this.call(`${this.API_PATH}configs/${configId}`, 'DELETE');
  }

  getJSON(data) {
    return typeof data === 'object' ? data : JSON.parse(data);
  }

  async simulatePipeline(data, pipeline) {
    if (data && typeof data !== 'object') {
      try {
        data = JSON.parse(data);
      } catch (e) {
        return Promise.resolve(e.toString());
      }
    }
    let response;
    try {
      pipeline = this.getJSON(pipeline);
      const docs = [
        {
          _index: 'index',
          _type: '_doc',
          _id: 'id',
          _source: data
        }
      ];
      response = await this.call(`${this.BASE_PATH}/elasticsearch/_ingest/pipeline/_simulate`,
        'POST',
        {
          pipeline,
          docs
        }
      );
    } catch (e) {
      return e.data ? e.data.error.reason : e.toString();
    }
    return response.docs[0].doc._source;
  }

  async checkIndexExists(indexName) {
    return await this.call(`${this.API_PATH}${indexName}/_exists`, 'GET');
  }

  async createIndex(indexName) {
    return await this.call(`${this.API_PATH}${indexName}`, 'POST');
  }

  async deleteIndex(indexName) {
    return await this.call(`${this.API_PATH}${indexName}`, 'DELETE');
  }

  async pushIndexMapping(indexName, properties) {
    return await this.call(`${this.API_PATH}${indexName}/_mapping/doc`, 'POST', properties);
  }

  async savePipeline(pipelineId, pipelineDef) {
    return await this.call(`${this.API_PATH}pipeline/${pipelineId}`, 'POST', this.getJSON(pipelineDef));
  }

  async deletePipeline(pipelineId) {
    return await this.call(`${this.API_PATH}delete-pipeline/${pipelineId}`, 'DELETE');
  }

  async bulkIndex(indexName, payload, pipelineId) {
    let pipelineAPI = '';
    if (pipelineId) {
      pipelineAPI = `/${pipelineId}`;
    }
    return await this.call(`${this.API_PATH}${indexName}/doc/_bulk${pipelineAPI}`, 'POST', payload);
  }

}

let xlsxClient;
export default {
  getInstance: () => {
    if (!xlsxClient) {
      throw new Error('XLSXClient not Instantiated!');
    }
    return xlsxClient;
  },
  instantiateClient: httpClient => {
    xlsxClient = new XLSXClient(httpClient);
  }
};