import { uiModules } from 'ui/modules';
import chrome from 'ui/chrome';
import { setESIndexName } from '../sections/xlsx/services/sheetServices';

uiModules.get('ingest/core/services').service('ingestionService', function ($http) {

  /**
   * Guess if the index is a reflection of which datasource
   * @param  {String} indexName
   * @param  {Array.<String>} sanitizedDatasourceList
   * @Nullable
   * @return {String}
   */
  function _guessIndexDatasource(indexName, sanitizedDatasourceList) {
    let max = -1;
    let guessedDatasource = null;
    sanitizedDatasourceList.forEach(ds => {
      if (ds.length > max && (indexName.startsWith('.' + ds) || indexName.startsWith(ds))) {
        max = ds.length;
        guessedDatasource = ds;
      }
    });
    return guessedDatasource;
  }

  function _calculateMostMatched(datasources) {
    let max = -1;
    let mostMatched = '';
    Object.keys(datasources).forEach(key => {
      const arr = datasources[key];
      if (arr.length > max) {
        max = arr.length;
        mostMatched = key;
      }
    });
    return mostMatched;
  }

  class IngestionService {
    constructor(basePath, httpClient) {
      this.API_ROOT = `${basePath}/connector_elasticsearch`;
      this.DATASOURCE_ROOT = `${this.API_ROOT}/_siren/connector/datasource`;
      this.httpClient = httpClient;
    }

    async fetch(url, method, data) {
      if (!method) {
        method = 'GET';
      }
      const response = await this.httpClient({
        method,
        url,
        data
      });
      return response.data;
    }


    /**
     * Compute nested fields.
     * Note: Since nested fields do not receive any type, they are not
     * included in fields array
     * @param  {Object} options.result
     * @param  {Array.<String>} options.fields
     * @return {Array.<String>} nestedFields
     */
    _getNestedFields({ result, fields }) {
      const fieldSet = new Set(fields.map(({ field }) => field));
      const nestedFields = [];
      Object.keys(result).forEach(field => {
        if (!fieldSet.has(field)) {
          nestedFields.push(field);
        }
      });
      return nestedFields;
    }

    /**
     * Fire a sample query against a datasource
     * @param  {String} datasourceId
     * @param  {String} query
     * @param  {Number} rowLimit; default = 2000
     * @param  {Number} maxTextSize; default = 10000
     * @return {Object}
     */
    async fetchSample(datasourceId, query, rowLimit = 2000, maxTextSize = 10000) {
      let response;
      try {
        response =  await this.fetch(`${this.DATASOURCE_ROOT}/${datasourceId}/_sample`,
          'POST', {
            query,
            row_limit: rowLimit,
            max_text_size: maxTextSize
          });
      } catch (e) {
        if (e.status === 404) {
          throw new Error(`Datasource (${datasourceId}) Not Found`);
        }
        throw new Error(this._getReasonForError(e.data.error));
      }
      if (!response.found) {
        return {
          fields: [],
          results: []
        };
      }
      const { types, results } = response;
      const fields = [];
      for (const field of Object.keys(types)) {
        if (types.hasOwnProperty(field)) {
          fields.push({
            field,
            type: types[field]
          });
        }
      }
      const computedResult = {
        results,
        fields
      };
      // Only considering first row of results, as in JDBC: the column count remains constant for the entire ResultSet
      if (results.length > 0 && fields.length < Object.keys(results[0]).length) {
        computedResult.nestedFields = this._getNestedFields({ fields, result: results[0] });
      }
      return computedResult;
    }

    /**
     * Guess datasources of selectedIndices
     * @param  {Set.<String>} selectedIndices
     * @param  {Array.<Object>} datasources
     * @return {Object} Guess result:
     * {
     *   mostMatched: index,
     *   datasources: {
     *     datasourceName: [...indices]
     *   },
     *   unableToResolve: [...indices]
     * }
     */
    guessIndexDatasources(selectedIndices, datasources) {
      const datasourceSanitizedMap = datasources.reduce((acc, e) => ({ [setESIndexName(e._id)]: e._id }), {});
      const datasourceSanitizedList = Object.keys(datasourceSanitizedMap);
      const result = {
        datasources: {},
        unableToResolve: []
      };
      selectedIndices.forEach(index => {
        const guess = _guessIndexDatasource(index, datasourceSanitizedList);
        if (guess) {
          const key = datasourceSanitizedMap[guess];
          if (result.datasources[key]) {
            result.datasources[key].push(index);
          } else {
            result.datasources[key] = [index];
          }
        } else {
          result.unableToResolve.push(index);
        }
      });
      result.mostMatched = _calculateMostMatched(result.datasources);
      return result;
    }

    _getReasonForError(e) {
      let currentException = e;
      let reason = '';
      while (currentException.caused_by) {
        reason += `${currentException.caused_by.reason} | `;
        currentException = currentException.caused_by;
      }
      return reason.slice(0, -3) || e.reason;
    }

  }

  const ingestionService = new IngestionService(chrome.getBasePath(), $http);

  return ingestionService;
});