import PropTypes from 'prop-types';
import React, { Component } from 'react';

import {
  EuiCode,
  EuiCodeBlock,
  EuiCodeEditor,
  EuiFormRow,
  EuiPanel,
  EuiSpacer,
  EuiSelect,
  EuiSuperSelect,
  EuiTabbedContent,
  EuiFlexGroup,
  EuiFlexItem,
  EuiButton
} from '@elastic/eui';

const pipelineMethodList = [
  {
    value: 'skip',
    inputDisplay: 'Skip transform'
  },
  {
    value: 'elasticsearch',
    inputDisplay: 'Elasticsearch ingestion pipeline method'
  }
];

const codeEditorOptions = {
  fontSize: '12px',
  enableBasicAutocompletion: true,
  //Riddled with bugs as of Eui 4.3.0
  //enableSnippets: true,
  //enableLiveAutocompletion: true,
};

function skipPipeline(pipelineMethod) {
  return pipelineMethod === pipelineMethodList[0].value;
}

class DefineTransform extends Component {

  constructor(props) {
    super(props);
    this.state = {
      pipelineValue: props.documents[props.config.pipelineInput || 0],
      pipelineInput: props.config.pipelineInput,
      testingPipeline: false
    };
    this.testPipeline = this.testPipeline.bind(this);
  }

  componentWillReceiveProps(props) {
    if (this.state.pipelineInput !== props.config.pipelineInput) {
      this.setState({
        pipelineValue: props.documents[props.config.pipelineInput],
        pipelineInput: props.config.pipelineInput
      });
    }
  }

  formatText(value) {
    if (value && typeof value === 'object') {
      value = JSON.stringify(value, null, 2);
    } else {
      try {
        value = JSON.parse(value);
        value = JSON.stringify(value, null, 2);
      } catch (e) {
        //Do Nothing
      }
    }
    return value;
  }

  testPipeline() {
    this.setState({ testingPipeline: true });
    this.props.testPipeline(this.state.pipelineValue).then(res => {
      this.props.onConfigChange(res, 'pipelineResult');
      this.setState({ testingPipeline: false });
    });
  }

  render() {
    const { pipelineMethod, pipeline, pipelineInput, pipelineResult } = this.props.config;
    const {
      documents,
      updatePipelineDef,
      onConfigChange
    } = this.props;
    if (skipPipeline(pipelineMethod)) {
      return <div />;
    }
    const options = documents.map((ele, index) => {
      return {
        value: index,
        text: `Record ${index + 1}`
      };
    });
    const pipelineTabs = [
      {
        id: 'pipeline',
        name: 'Elasticsearch pipeline definition',
        content: (
          <EuiCodeEditor
            mode="sql"
            theme="github"
            width="100%"
            height="306px"
            value={pipeline}
            onChange={value => updatePipelineDef({ value })}
            setOptions={codeEditorOptions}
          />
        )
      },
      {
        id: 'help',
        name: 'Help',
        content: (
          <EuiPanel paddingSize="l">
            <p>A pipeline is a definition of a series of processors that are to be executed in the same order as they are declared.</p>
            <p>A pipeline consists of two main fields: a <EuiCode>description</EuiCode> and a list of <EuiCode>processors</EuiCode>:</p>
            <EuiSpacer size="s"/>
            <EuiCodeBlock>{
              JSON.stringify({
                description: '...',
                processors: [ ]
              }, null, 2)
            }
            </EuiCodeBlock>
            <EuiSpacer size="s"/>
            <p>The <EuiCode>description</EuiCode> is a special field to store a helpful description of what the pipeline does.</p>
            <p>The <EuiCode>processors</EuiCode> parameter defines a list of processors to be executed in order.</p>
            <p>Try our&nbsp;
              <a
                target="_blank"
                href="https://docs.support.siren.io/latest/platform/en/siren-investigate/data-reflection/datasource-reflection-pipelines.html"
              >
                Sample Pipelines
              </a>
            </p>
          </EuiPanel>
        )
      }
    ];
    const defineTransform = (
      <div>
        <EuiTabbedContent
          tabs={pipelineTabs}
        />
        <EuiSpacer size="m"/>
        <EuiFormRow
          label="Select Sample Record"
          fullWidth
        >
          <EuiFlexGroup>
            <EuiFlexItem grow={false}>
              <EuiSelect
                options={options}
                value={pipelineInput}
                disabled={this.state.testingPipeline}
                onChange={(e) => onConfigChange(e.target.value, 'pipelineInput')}
              />
            </EuiFlexItem>
            <EuiFlexItem
              grow={false}
            >
              <EuiButton
                color="primary"
                fill
                iconType="play"
                isLoading={this.state.testingPipeline}
                onClick={this.testPipeline}
              >
                Test
              </EuiButton>
            </EuiFlexItem>
          </EuiFlexGroup>
        </EuiFormRow>
        <EuiFlexGroup>
          <EuiFlexItem>
            <EuiFormRow
              label="Sample Input"
              fullWidth
            >
              <EuiCodeEditor
                mode="sql"
                theme="github"
                width="100%"
                height="220px"
                onChange={(pipelineValue) => this.setState({ pipelineValue })}
                value={this.formatText(this.state.pipelineValue)}
                setOptions={codeEditorOptions}
              />
            </EuiFormRow>
          </EuiFlexItem>
          <EuiFlexItem>
            <EuiFormRow
              label="Test Output"
              fullWidth
            >
              <EuiCodeEditor
                mode="sql"
                theme="github"
                width="100%"
                height="220px"
                value={JSON.stringify(pipelineResult, null, 2)}
                setOptions={codeEditorOptions}
              />
            </EuiFormRow>
          </EuiFlexItem>
        </EuiFlexGroup>
      </div>
    );
    return defineTransform;
  }
}

function PipelineDefinition(props) {
  return (
    <div>
      <EuiFormRow
        fullWidth
      >
        <EuiFlexGroup>
          <EuiFlexItem
            style={{ flex: '2', paddingTop: '.8em' }}
          >
            <b>Method</b>
          </EuiFlexItem>
          <EuiFlexItem
            style={{ flex: '6' }}
          >
            <EuiSuperSelect
              valueOfSelected={props.configuration.pipelineMethod}
              options={pipelineMethodList}
              onChange={e => props.onConfigChange(e, 'pipelineMethod')}
            />
          </EuiFlexItem>
        </EuiFlexGroup>
      </EuiFormRow>
      <EuiSpacer size="m"/>
      <DefineTransform
        documents={props.documents}
        config={props.configuration}
        updatePipelineDef={props.updatePipelineDef}
        onConfigChange={props.onConfigChange}
        testPipeline={props.testPipeline}
      />
    </div>
  );
}

export {
  DefineTransform,
  PipelineDefinition
};

PipelineDefinition.propTypes = {
  documents: PropTypes.array.isRequired,
  configuration: PropTypes.object.isRequired,
  updatePipelineDef: PropTypes.func.isRequired,
  onConfigChange: PropTypes.func.isRequired,
  testPipeline: PropTypes.func.isRequired
};

DefineTransform.propTypes = {
  documents: PropTypes.array.isRequired,
  config: PropTypes.object.isRequired,
  updatePipelineDef: PropTypes.func.isRequired,
  onConfigChange: PropTypes.func.isRequired,
  testPipeline: PropTypes.func.isRequired
};