import { pendingJobsCache } from './pending_jobs';

const DEFAULT_HEADERS = {
  'Content-Type': 'application/json',
  'kbn-xsrf': 'anything',
};
class DataModelGenerator {
  constructor(indexEntities, server, jobId, user) {
    this._indexEntities = indexEntities;
    this._server = server;
    this._queryEngine = server.plugins.query_engine.getQueryEngine();
    this.currentJob = null;
    this._savedObjectsClient = server.savedObjectsClientFactory({
      callCluster: server.plugins.elasticsearch.getCluster('admin').callWithInternalUser
    });
    this._jobId = jobId;
    this._datasourceId = pendingJobsCache.getDatasourceId(jobId);
    this._user = user;
  }

  get indexEntities() {
    return this._indexEntities;
  }

  get server() {
    return this._server;
  }

  get queryEngine() {
    return this._queryEngine;
  }

  get savedObjectsClient() {
    return this._savedObjectsClient;
  }

  get datasourceId() {
    return this._datasourceId;
  }

  async _getFields(indexEntity) {
    const url = `/api/index_patterns/_fields_for_wildcard?pattern=${indexEntity.es}` +
      `&meta_fields=${encodeURIComponent('["_source","_id","_type","_index","_score"]')}`;
    const response = await this.server.inject({
      url,
      credentials: this._user
    });
    return response.result.fields;
  }

  _createIndexPatternPayload(indexPattern, fields) {
    return {
      attributes: {
        title: indexPattern,
        fields: JSON.stringify(fields),
        excludeIndices: true
      }
    };
  }

  _createSavedSearchPayload(indexEntity) {
    return {
      attributes: {
        title: indexEntity.savedSearch,
        hits: 0,
        columns: ['_source'],
        sort: [],
        version: 1,
        kibanaSavedObjectMeta: {
          searchSourceJSON: JSON.stringify({
            index: `index-pattern:${indexEntity.es}`,
            highlightAll: true,
            version: true
          })
        },
        siren: {
          ui: {
            color: indexEntity.color,
            icon: indexEntity.icon
          }
        }
      }
    };
  }

  async _commitIndexPattern(indexEntity, fields) {
    const url = `/api/saved_objects/index-pattern/${indexEntity.es}`;
    return await this.server.inject({
      headers: DEFAULT_HEADERS,
      method: 'PUT',
      url,
      credentials: this._user,
      payload: JSON.stringify(this._createIndexPatternPayload(indexEntity.es, fields))
    });
  }

  async _createIndexPattern(indexEntity) {
    const fields = await this._getFields(indexEntity);
    return await this._commitIndexPattern(indexEntity, fields);
  }

  /**
   * Mutates indexEntity with the committed searchId
   * @param  {IndexEntity} indexEntity
   */
  async _commitSavedSearch(indexEntity) {
    const url = `/api/saved_objects/search/search:${this.datasourceId}:${indexEntity.savedSearch}`;
    const response = await this.server.inject({
      method: 'PUT',
      url,
      credentials: this._user,
      headers: DEFAULT_HEADERS,
      payload: JSON.stringify(this._createSavedSearchPayload(indexEntity))
    });
    indexEntity.searchId = response.result.id;
  }

  async _commitDataModel(indexEntity) {
    const resp = await this._createIndexPattern(indexEntity);
    const respSS = await this._commitSavedSearch(indexEntity); //enriches indexEntity with searchId
    return respSS; //TODO: Check for errors in responses
  }

  async _buildDataModel() {
    const promises = [];
    Object.keys(this.indexEntities).forEach(key => {
      promises.push(this._commitDataModel(this.indexEntities[key]));
    });
    await Promise.all(promises);
  }

  /**
   * Create relation id
   * @param  {IndexEntity}  domainEntity
   * @param  {IndexEntity}  rangeEntity
   * @param  {IndexEntity}  otherEntity
   * @param  {Boolean} isInverse
   * @return {String}
   */
  _getRelationId(domainEntity, rangeEntity, otherEntity, isInverse) {
    return `${this.datasourceId}:${otherEntity.nodeName}-${domainEntity.nodeName}` +
      `-${rangeEntity.nodeName}${isInverse ? '-inverse' : ''}`;
  }

  /**
   * Creates a title for the relation
   * @param  {String} domainNodeName
   * @param  {String} rangeNodeName
   * @param  {String} relation
   * @return {String}
   */
  _getRelationTitle(domainNodeName, rangeNodeName, relation) {
    return `${this.datasourceId}: ${domainNodeName} -${relation}-> ${rangeNodeName}`;
  }

  /**
   * @param  {Object(IndexEntity)} relationEntity Relation indexEntity
   * @param  {Object(IndexEntity)} start          Relation start indexEntity
   * @param  {Object(IndexEntity)} end            Relation end indexEntity
   * @return {Array.<Object>} A 2 element array: [startRelation, endRelation]
   */
  _createRelationPayload(relationEntity, start, end) {
    const type = 'relation';
    const relationStartId = this._getRelationId(start, relationEntity, end, false);
    const relationStartInvId = this._getRelationId(start, relationEntity, end, true);
    let relationEndId;
    let relationEndInvId;
    if (start.nodeName === end.nodeName) {
      relationEndId = this._getRelationId({ nodeName: `${end.nodeName}-selfRelation` }, relationEntity, start, false);
      relationEndInvId = this._getRelationId({ nodeName: `${end.nodeName}-selfRelation` }, relationEntity, start, true);
    } else {
      relationEndId = this._getRelationId(end, relationEntity, start, false);
      relationEndInvId = this._getRelationId(end, relationEntity, start, true);
    }
    const relationStart = {
      id: relationStartId,
      type,
      attributes: {
        directLabel: relationEntity.label,
        inverseLabel: relationEntity.inverseLabel,
        domainId: start.searchId,
        domainField: 'node_id',
        id: relationStartId,
        inverseOf: relationStartInvId,
        rangeField: 'start_node_id',
        rangeId: relationEntity.searchId,
        title: this._getRelationTitle(start.nodeName, relationEntity.nodeName, relationEntity.label)
      }
    };

    const relationEnd = {
      id: relationEndId,
      type,
      attributes: {
        directLabel: relationEntity.inverseLabel,
        inverseLabel: relationEntity.label,
        domainId: end.searchId,
        domainField: 'node_id',
        id: relationEndId,
        inverseOf: relationEndInvId,
        rangeField: 'end_node_id',
        rangeId: relationEntity.searchId,
        title: this._getRelationTitle(end.nodeName, relationEntity.nodeName, relationEntity.inverseLabel)
      }
    };
    return [relationStart, relationEnd];
  }

  async _commitRelations(payload) {
    const opts = {
      overwrite: true,
      format: 'v6'
    };
    return await this.savedObjectsClient.bulkCreate(payload, opts);
  }

  /**
   * Checks if the passed entity is a relation
   * @param  {[type]}  entity
   * @return {Boolean}
   */
  _isRelationEntity(entity) {
    return entity.relations && entity.relations.length > 0;
  }

  /**
   * Filters out duplicate relations (Based on title)
   * @param  {Array.<Object>} relationsPayload
   * @return {Array.<Object>} Filtered, unique payload array
   */
  _filterDuplicateRelations(relationsPayload) {
    const addedRelations = new Set();
    return relationsPayload.filter(({ attributes: { title } }) => {
      if (!addedRelations.has(title)) {
        addedRelations.add(title);
        return true;
      }
      return false;
    });
  }

  _createRelationshipEntities() {
    let relationsPayload = [];
    Object.keys(this.indexEntities).forEach(entity => {
      // Optimize half step relations
      if (this._isRelationEntity(this.indexEntities[entity])) {
        this.indexEntities[entity].relations.forEach(relation => {
          relationsPayload = relationsPayload.concat(this._createRelationPayload(this.indexEntities[entity],
            this.indexEntities[relation.startNode],
            this.indexEntities[relation.endNode])
          );
        });
      }
    });
    return this._filterDuplicateRelations(relationsPayload);
  }

  async _buildOntology() {
    const payload = this._createRelationshipEntities();
    return await this._commitRelations(payload);

  }

  _initJobData(promise) {
    this.currentJob = {
      promise
    };
  }

  async generateDataModel() {
    if (!this.currentJob) {
      const dmResp = await this._buildDataModel();
      const relResp = await this._buildOntology();
      //TODO: Check for errors in responses
      pendingJobsCache.clearJob(this._user && this._user.username, this._jobId);
      return {
        acknowledged: true
      };
    } else {
      return new Error('Existing job not finished yet!');
    }
  }
}

export {
  DataModelGenerator
};