import { pendingJobsCache } from './pending_jobs';
import crypto from 'crypto';

const TARGET_INDEX_SUFFIX = 'neo4j-index';
const DEFAULT_INGESTION_STRATEGY = 'REPLACE';
const DATA_NODE_DEFAULT_TRANSFORMS = [
  {
    input: [{ source: 'node_id' }],
    output: 'node_id',
    mapping: {
      type: 'keyword'
    }
  },
  {
    input: [{ source: 'node_labels' }],
    output: 'node_labels',
    mapping: {
      type: 'keyword'
    }
  }
];
const RELATION_NODE_DEFAULT_TRANSFORMS = [
  {
    input: [{ source: 'node_id' }],
    output: 'node_id',
    mapping: {
      type: 'keyword'
    }
  },
  {
    input: [{ source: 'start_node_id' }],
    output: 'start_node_id',
    mapping: {
      type: 'keyword'
    }
  },
  {
    input: [{ source: 'end_node_id' }],
    output: 'end_node_id',
    mapping: {
      type: 'keyword'
    }
  },
  {
    input: [{ source: 'relation_type' }],
    output: 'relation_type',
    mapping: {
      type: 'keyword'
    }
  }
];

/**
 * Mainly, a one time use ingestion job creator object
 */
class IngestionJobCreator {
  constructor(jobsArray, datasourceId, neo4jDocumentField, dataCluster, request) {
    this._neo4jDocumentField = neo4jDocumentField;
    this._pipeline = Object.freeze({
      description: 'Brings nested Neo4J document to top level.',
      processors: [
        {
          script: {
            lang: 'painless',
            source:
              `for (String key : ctx.${neo4jDocumentField}.keySet() ) {` +
                `ctx[key] = ctx.${neo4jDocumentField}[key];` +
              '}' +
              `ctx.remove("${neo4jDocumentField}");`
          }
        }
      ]
    });

    this._ingestionJobs = jobsArray;
    this._datasourceId = datasourceId;
    this._dataCluster = dataCluster;
    this._request = request;
    this._authUser = request.auth.credentials;
    this.currentJob = null;
  }

  /**
   * Trigger ingestion job creation
   * Warning: The previous job must have finished before starting a new job
   * on a single {@code IngestionJobCreator} object
   * @return {Promise} A promise that returns job result is successful
   */
  createIngestionJobs() {
    if (!this.currentJob) {
      return new Promise((resolve, reject) => {
        this._initJobData({ resolve, reject });
        this.ingestionJobs.forEach(ingestionJob =>
          this._createIngestionJobForEntity(ingestionJob)
        );
        this._commitJobs();
      });
    } else {
      return Promise.reject(new Error('Existing job not finished yet!'));
    }
  }

  get datasourceId() {
    return this._datasourceId;
  }

  get ingestionJobs() {
    return this._ingestionJobs;
  }

  get authUser() {
    return this._authUser;
  }

  /**
   * Returns Elasticsearch pipeline for the Neo4J reflection jobs
   * @return {Object}
   */
  get pipeline() {
    return this._pipeline;
  }

  _generateJobId(seed) {
    return crypto
      .createHash('md5')
      .update(seed)
      .digest('hex');
  }

  /**
   * Checks if the passed entity is a relation
   * @param  {Object}  entity
   * @return {Boolean}
   */
  _isRelationEntity(entity) {
    return entity.relations && entity.relations.length > 0;
  }

  /**
   * @param  {String} query
   * @param  {String} targetIndex
   * @param  {String} entitiyName
   * @param  {Array.<Object>} transforms
   * @return {Object} Ingestion job configuration
   */
  _generateIngestionConfig(query, targetIndex, entitiyName, transforms) {
    return {
      datasource: this.datasourceId,
      description: `Datasource: ${this.datasourceId}; Node Name:${entitiyName}. Created by Neo4j Wizard.`,
      query,
      transforms,
      pipeline: this.pipeline,
      enable_scheduler: false,
      target: targetIndex,
      strategy: DEFAULT_INGESTION_STRATEGY,
      pk_field: 'node_id'
    };
  }

  _revertCommittedJobs() {
    const promises = [];
    this.currentJob.ingestionJobs.forEach(job => {
      promises.push(
        this._dataCluster.callWithRequest(this._request, 'transport.request', {
          path: `_siren/connector/ingestion/${job.id}`,
          method: 'DELETE'
        })
      );
    });
    return Promise.all(promises);
  }

  _commitIngestionJobs() {
    const promises = [];
    this.currentJob.ingestionJobs.forEach(job => {
      promises.push(
        this._dataCluster.callWithRequest(this._request, 'transport.request', {
          path: `_siren/connector/ingestion/${job.id}`,
          method: 'PUT',
          body: job.body
        })
      );
    });
    return Promise.all(promises);
  }

  /**
   * Adds the ingestion job to be committed later
   * @param  {Object} config; job config
   * @param  {String} id
   */
  _createIngestionJob(config, id) {
    const body = {
      ingest: config
    };
    this.currentJob.ingestionJobs.push({
      id,
      body
    });
  }


  /**
   * Builds and adds jobData for a nodeJob to currentJob
   * @param  {String} options.nodeName
   * @param  {String} options.targetIndex
   * @param  {String} options.jobName
   */
  _buildJobDataForNodeJob({ nodeName, targetIndex, jobName }) {
    this.currentJob.jobData[nodeName] = {
      es: targetIndex,
      savedSearch: nodeName,
      jobId: jobName,
      label: nodeName,
      color: '#00e1ff',
      icon: 'fab fa-neos',
      nodeName
    };
  }

  /**
   * Creates an ingestion for config for the passed entity and builds jobData for currentJob
   * @param  {Object} job
   */
  _createIngestionJobForEntity(job) {
    const { targetIndex, nodeName, cypherQuery, jobName } = job;
    let transforms;
    if (this._isRelationEntity(job)) {
      this._buildJobDataForRelationJob(job);
      transforms = RELATION_NODE_DEFAULT_TRANSFORMS;
    } else {
      this._buildJobDataForNodeJob(job);
      transforms = DATA_NODE_DEFAULT_TRANSFORMS;
    }
    this._createIngestionJob(this._generateIngestionConfig(cypherQuery, targetIndex, nodeName, transforms), jobName);
  }

  /**
   * Builds and adds jobData for a relationJob to currentJob
   * @param  {String} options.nodeName
   * @param  {String} options.targetIndex
   * @param  {String} options.jobName
   * @param  {String} options.relation
   */
  _buildJobDataForRelationJob({ nodeName, targetIndex, jobName, relations }) {
    this.currentJob.jobData[nodeName] = {
      es: targetIndex,
      savedSearch: nodeName,
      jobId: jobName,
      label: nodeName,
      inverseLabel: `inverse of: ${nodeName}`,
      color: '#211df0',
      icon: 'fas fa-link',
      nodeName,
      relations
    };
  }

  _initJobData(promise) {
    this.currentJob = {
      jobData: {},
      ingestionJobs: [],
      promise
    };
  }

  _clearJobData() {
    this.currentJob = null;
  }

  async _commitJobs() {
    try {
      await this._commitIngestionJobs();
      const jobId = this._generateJobId(`${new Date().getTime()}${this.datasourceId}`);
      pendingJobsCache.registerJob(this.authUser && this.authUser.username, jobId, this.currentJob.jobData,
        this.datasourceId);
      this.currentJob.promise.resolve(
        {
          acknowledged: true,
          jobs_created: this.currentJob.ingestionJobs.map(ele => ele.id),
          jobId,
          jobData: this.currentJob.jobData
        }
      );
      this._clearJobData();
    } catch (e) {
      this._handleJobFailure(e);
    }
  }

  _handleJobFailure = error => {
    this.currentJob.promise.reject(error);
    const clearData = () => this._clearJobData();
    this._revertCommittedJobs().then(clearData).catch(clearData);
    //TODO: Review: Should I revert? Don't want to delete existing configs
  }
}

export {
  IngestionJobCreator
};