/**
 * An in-memory cache of all the pending jobs, non persistant
 */
class PendingJobs {
  constructor() {
    this.jobsCache = {};
    this.userJobsCache = {};
    this.datasourceCache = {};
  }

  /**
   * Register an ongoing job
   * @param  {String} username
   * @param  {String} jobId
   * @param  {Object} jobData
   */
  registerJob(username, jobId, jobData, datasourceId) {
    this.datasourceCache[jobId] = datasourceId;
    if (!username) {
      this.jobsCache[jobId] = jobData;
    } else {
      if (!this.userJobsCache[username]) {
        this.userJobsCache[username] = {};
        this.userJobsCache[username][jobId] = jobData;
      } else {
        this.userJobsCache[username][jobId] = jobData;
      }
    }
  }

  clearJob(username, jobId) {
    delete this.datasourceCache[jobId];
    if (!username) {
      delete this.jobsCache[jobId];
    } else {
      delete this.userJobsCache[username][jobId];
    }
  }

  /**
   * @param  {String} jobId
   * @Nullable
   * @return {String}
   */
  getDatasourceId(jobId) {
    return this.datasourceCache[jobId];
  }

  /**
   * @param  {String} username
   * @param  {String} jobId
   * @Nullable
   * @return {jobData}
   */
  getJob(username, jobId) {
    if (!username) {
      return this.jobsCache[jobId];
    } else if (this.jobsCache[username]) {
      return this.userJobsCache[username][jobId] || this.jobsCache[jobId];
    }
  }

  getJobs(username) {
    // Only one of those cache will ever be filled as either you have security or
    // not but anyways.
    return {
      ...this.jobsCache,
      ...this.userJobsCache[username]
    };
  }
}

const pendingJobsCache = new PendingJobs();

export {
  pendingJobsCache
};