import { transform, forEach } from 'lodash';
import Migration from 'kibiutils/lib/migrations/migration';

/**
 * Investigate Core - Migration 16.
 *
 * Looks for visualisation with _source.visState.type == 'kibi_multi_chart_vis'
 *
 * - change the value to 'multi_chart_vis'
 */

export default class Migration1 extends Migration {

  constructor(configuration) {
    super(configuration);

    this._client = configuration.client;
    this._index = configuration.config.get('kibana.index');
    this._logger = configuration.logger;
    this._type = 'visualization';
  }

  static get description() {
    return 'Upgrade Multichart visualisation type from kibi_multi_chart_vis to multi_chart_vis';
  }

  _isUpgradeable(visState) {
    let result = false;
    if (visState.kibiSettings && visState.kibiSettings.settings) {
      visState.kibiSettings.settings.some(setting => {
        if (setting.activeMode === 'kibi_bubble_diagram_vis' || setting.modes.kibi_bubble_diagram_vis) {
          result = true;
        }
        return result;
      });
    }
    return visState.type === 'kibi_multi_chart_vis' || result;

  }

  async count() {
    const objects = await this.scrollSearch(this._index, this._type);
    return objects.reduce((count, obj) => {
      const visState = JSON.parse(obj._source.visState);
      if (this._isUpgradeable(visState)) {
        return count + 1;
      }
      return count;
    }, 0);
  }

  async upgrade() {

    let keyPath = '';
    const updateVisState = (result, value, key) => {
      if (keyPath === '' && key === 'type') {
        result[key] = 'multi_chart_vis';
      } else if (keyPath === '' && key === 'kibiSettings') {
        keyPath = key;
        result[key] = transform(value, updateVisState);
        keyPath = '';
      } else if (keyPath + '.' + key === 'kibiSettings.settings') {
        const oldKeyPath = keyPath;
        keyPath = keyPath + '.' + key;
        result[key] = value.map(setting => {
          return transform(setting, updateVisState);
        });
        keyPath = oldKeyPath;
      } else if (keyPath + '.' + key === 'kibiSettings.settings.activeMode' && value === 'kibi_bubble_diagram_vis') {
        result[key] = 'bubble_diagram_vis';
      } else if (keyPath + '.' + key === 'kibiSettings.settings.modes') {
        const oldKeyPath = keyPath;
        keyPath = keyPath + '.' + key;
        const modeResult = {};
        forEach(value, (mode, subKey) => {
          if (subKey === 'kibi_bubble_diagram_vis') {
            subKey = 'bubble_diagram_vis';
          }
          modeResult[subKey] = transform(mode, updateVisState);
        });
        result[key] = modeResult;
        keyPath = oldKeyPath;
      } else if (keyPath + '.' + key === 'kibiSettings.settings.modes.kibi_bubble_diagram_vis' && key === 'kibi_bubble_diagram_vis') {
        const oldKeyPath = keyPath;
        keyPath = keyPath + '.' + key;
        result[key] = forEach(value, mode => {
          return transform(mode, updateVisState);
        });
        keyPath = oldKeyPath;
        key = 'bubble_diagram_vis';
      } else if (keyPath + '.' + key === 'kibiSettings.settings.modes.type' && value === 'kibi_bubble_diagram_vis') {
        result[key] = 'bubble_diagram_vis';
      } else {
        result[key] = value;
      }
    };

    const objects = await this.scrollSearch(this._index, this._type);
    if (objects.length === 0) {
      return 0;
    }
    let body = '';
    let count = 0;
    for (const obj of objects) {
      const visState = JSON.parse(obj._source.visState);
      if (this._isUpgradeable(visState)) {
        const newVisState = transform(visState, updateVisState);
        obj._source.visState = JSON.stringify(newVisState);
        body += JSON.stringify({
          update: {
            _index: obj._index,
            _type: obj._type,
            _id: obj._id
          }
        }) + '\n' + JSON.stringify({ doc: obj._source }) + '\n';
        count++;
      }
    }

    if (count > 0) {
      await this._client.bulk({
        refresh: true,
        body: body
      });
    }
    return count;
  }

}

