import ngMock from 'ng_mock';
import expect from 'expect.js';
import sinon from 'sinon';
import Promise from 'bluebird';

describe('smart default configuration', function () {
  let fieldStatsAnalysis;
  let sdcQuery;
  const ip = {};

  beforeEach(ngMock.module('kibana'));
  beforeEach(ngMock.module('kibana/multi_chart_vis'));
  beforeEach(ngMock.inject(function ($injector) {
    fieldStatsAnalysis = $injector.get('multiChartSDC');
    sdcQuery = $injector.get('multiChartSDCQuery');
  }));

  describe('basics', function () {
    it('not support type must be rejected', function (done) {
      const field = {
        name: 'dummyname',
        type: 'dummytype',
        aggregatable: true
      };

      fieldStatsAnalysis(ip, field).then(resp => {
        expect(resp).to.have.property('vis', 'N/A');
        done();
      }).catch(done);
    });

    it('failure on es connection must return error', function () {
      const field = {
        name: 'city',
        type: 'string',
        aggregatable: true
      };

      return expect(fieldStatsAnalysis(ip, field)).to.be.rejected;
    });

  });

  describe('correct size on an aggregation request', function () {
    const testCase = {
      a: (sizeValue) => {
        return {
          took: 2,
          timed_out: false,
          _shards: {
            total: 5,
            successful: 5,
            failed: 0
          },
          hits: {
            total: 160106,
            max_score: 0,
            hits: []
          },
          aggregations: {
            1: {
              value: sizeValue
            },
            2: {
              buckets: []
            }
          }
        };
      }
    };
    const stringField = {
      name: 'dummyname',
      type: 'string',
      aggregatable: true
    };
    const numericField = {
      name: 'dummyname',
      type: 'number',
      aggregatable: true
    };

    afterEach(() => {
      sdcQuery.searchAgg.restore();
    });

    it('should update size if size is equal 0 on the string field', function (done) {

      const SIZE_VALUE = 0;
      const EXPECTED_SIZE_VALUE = 1;

      sinon.stub(sdcQuery, 'searchAgg').callsFake(() => {
        return Promise.resolve(testCase.a(SIZE_VALUE));
      });

      fieldStatsAnalysis(ip, stringField).then(resp => {
        expect(resp.aggs[1].params.size).to.be(EXPECTED_SIZE_VALUE);
        done();
      }).catch(done);
    });

    it('should update size if size is equal 0 on the numeric field', function (done) {

      const INTERVAL_VALUE = 0;
      const EXPECTED_INTERVAL_VALUE = 1;

      sinon.stub(sdcQuery, 'searchAgg').callsFake(() => {
        return Promise.resolve(testCase.a(INTERVAL_VALUE));
      });

      fieldStatsAnalysis(ip, numericField).then(resp => {
        expect(resp.aggs[1].params.interval).to.be(EXPECTED_INTERVAL_VALUE);
        done();
      }).catch(done);
    });

    it('should update size if size is NaN on the string field', function (done) {

      const SIZE_VALUE = NaN;
      const EXPECTED_SIZE_VALUE = 1;

      sinon.stub(sdcQuery, 'searchAgg').callsFake(() => {
        return Promise.resolve(testCase.a(SIZE_VALUE));
      });

      fieldStatsAnalysis(ip, stringField).then(resp => {
        expect(resp.aggs[1].params.size).to.be(EXPECTED_SIZE_VALUE);
        done();
      }).catch(done);
    });

    it('should update size if size is NaN on the numeric field', function (done) {

      const INTERVAL_VALUE = NaN;
      const EXPECTED_INTERVAL_VALUE = 1;

      sinon.stub(sdcQuery, 'searchAgg').callsFake(() => {
        return Promise.resolve(testCase.a(INTERVAL_VALUE));
      });

      fieldStatsAnalysis(ip, numericField).then(resp => {
        expect(resp.aggs[1].params.interval).to.be(EXPECTED_INTERVAL_VALUE);
        done();
      }).catch(done);
    });

    it('should update size if size is undefined on the string field', function (done) {

      const SIZE_VALUE = undefined;
      const EXPECTED_SIZE_VALUE = 1;

      sinon.stub(sdcQuery, 'searchAgg').callsFake(() => {
        return Promise.resolve(testCase.a(SIZE_VALUE));
      });

      fieldStatsAnalysis(ip, stringField).then(resp => {
        expect(resp.aggs[1].params.size).to.be(EXPECTED_SIZE_VALUE);
        done();
      }).catch(done);
    });

    it('should update size if size is undefined on the numeric field', function (done) {

      const INTERVAL_VALUE = undefined;
      const EXPECTED_INTERVAL_VALUE = 1;

      sinon.stub(sdcQuery, 'searchAgg').callsFake(() => {
        return Promise.resolve(testCase.a(INTERVAL_VALUE));
      });

      fieldStatsAnalysis(ip, numericField).then(resp => {
        expect(resp.aggs[1].params.interval).to.be(EXPECTED_INTERVAL_VALUE);
        done();
      }).catch(done);
    });

    it('should NOT update size if size is integer and NOT equal 0 on the string field', function (done) {

      const SIZE_VALUE = 1;
      const EXPECTED_SIZE_VALUE = 1;

      sinon.stub(sdcQuery, 'searchAgg').callsFake(() => {
        return Promise.resolve(testCase.a(SIZE_VALUE));
      });

      fieldStatsAnalysis(ip, stringField).then(resp => {
        expect(resp.aggs[1].params.size).to.be(EXPECTED_SIZE_VALUE);
        done();
      }).catch(done);
    });

    it('should NOT update size if size is integer and NOT equal 0 on the numeric field', function (done) {

      const INTERVAL_VALUE = 1;
      const EXPECTED_INTERVAL_VALUE = 1;

      sinon.stub(sdcQuery, 'searchAgg').callsFake(() => {
        return Promise.resolve(testCase.a(INTERVAL_VALUE));
      });

      fieldStatsAnalysis(ip, numericField).then(resp => {
        expect(resp.aggs[1].params.interval).to.be(EXPECTED_INTERVAL_VALUE);
        done();
      }).catch(done);
    });

  });
});
