import 'plugins/multi_chart_vis/directives/multi_chart_vis_focus_me';
import 'plugins/multi_chart_vis/sdc/field_stats_analysis';
import multiChartVisConfigTemplate from 'plugins/multi_chart_vis/multi_chart_vis_config.html';
import { VisAggConfigsProvider } from 'ui/vis/agg_configs';
import { uiModules } from 'ui/modules';
import StopWatch from 'plugins/multi_chart_vis/sdc/stop_watch';
import promiseLimit from 'plugins/multi_chart_vis/sdc/promise_limit';
import compareFields from 'plugins/multi_chart_vis/sdc/sdc_compare';
import namesLogic from 'plugins/multi_chart_vis/sdc/sdc_names';
import _ from 'lodash';

const module = uiModules.get('kibana/multi_chart_vis');

module.directive('multiChartVisConfig', function (Private, $rootScope, multiChartSDC, createNotifier, confirmModal, $timeout) {
  const AggConfigs = Private(VisAggConfigsProvider);

  return {
    restrict: 'E',
    template: multiChartVisConfigTemplate,
    scope: {
      vis: '=',
      savedVis: '=',
      saveVis: '&'
    },
    link: function ($scope, $el) {
      $scope.editing = -1;
      const notify = createNotifier({
        location: 'Smart default configuration'
      });

      $scope.visStateManager = function () {
        if (!$scope.savedVis.vis._editableVis.visStateManager) {
          return {
            state: {}
          };
        } else {
          return $scope.savedVis.vis._editableVis.visStateManager;
        }
      };

      $scope.$on('multi_chart:newSetting', function () {
        const generatedName = namesLogic.generateCopyName($scope.visStateManager());
        const mode = $scope.visStateManager().activeSetting.activeMode;
        $scope.visStateManager().addSetting(generatedName, mode, $scope.vis.aggs.map(function (agg) {
          return agg.toJSON();
        }));
        const newSetting = _.find($scope.visStateManager().settings, setting => setting.name === generatedName);
        $scope.emitNameChange(newSetting);
      });

      $scope.edit = (idx) => {
        $scope.editing = idx;
        $scope.oldName = $scope.visStateManager().state.settings[idx].name;
        $scope.visStateManager().state.settings[idx].newName = $scope.oldName;
      };

      $scope.setAsActive = (idx, setting) => {
        $scope.emitNameChange(setting);
        // $timeout was applied to help us separate emitNameChange() correctly and save configuration
        $timeout(() => {
          $scope.saveVis();
        }, 100);
      };

      $scope.basedOnMissedField = (setting) => {
        const fieldToCheck = _.find(setting.aggs, agg => agg.params.field);
        if (fieldToCheck) {
          return $scope.visStateManager().missedFields.indexOf(fieldToCheck.params.field) !== -1;
        } else {
          return false;
        };  
      };

      $scope.rename = (setting) => {
        if (namesLogic.namesConflict(setting.newName, $scope.visStateManager()) && setting.name !== setting.newName) {
          const confirmModalOptions = {
            confirmButtonText: 'Save current name',
            onConfirm: () => {
              $scope.editing = -1;
            }
          };
          confirmModal(
            'A configuration with that name already exists. Please choose another name.',
            confirmModalOptions
          );
          return;
        }
        const isSelected = $scope.visStateManager().activeSetting.name === setting.name;
        const isSavedActiveSetting = $scope.visStateManager().savedActiveSetting === setting.name;
        if (!$scope.visStateManager().renameSetting($scope.editing, setting.newName)) {
          $scope.cancelEdit();
        }
        if (isSelected) {
          $rootScope.$broadcast('multi_chart:change_setting', setting.newName);
        }
        if (isSavedActiveSetting) {
          $scope.visStateManager().savedActiveSetting = setting.newName;
        }
        delete setting.newName;
        $scope.editing = -1;
      };

      $scope.trapEditKey = ($event, setting) => {
        if ($event.keyCode === 27) {
          $scope.cancelEdit();
          return;
        }
        if ($event.keyCode === 13) {
          $scope.rename(setting);
        }
      };

      $scope.cancelEdit = () => {
        $scope.visStateManager().state.settings[$scope.editing].name = $scope.oldName;
        $scope.editing = -1;
      };

      $scope.delete = function (idx) {
        if ($scope.visStateManager().state.settings[idx].name === $scope.visStateManager().savedActiveSetting) {
          $scope.visStateManager().savedActiveSetting = null;
        };
        $scope.visStateManager().deleteSetting(idx);
        $scope.emitNameChange($scope.visStateManager().activeSetting);
      };

      $scope.emitNameChange = function (setting) {
        if ($scope.basedOnMissedField(setting)) {
          $scope.vis.dirty = true;
        };
        $rootScope.$broadcast('multi_chart:change_setting', setting.name);
      };

      $scope.noActionsOnBar = false;

      function doGetSDC() {
        $scope.processState = [];
        const metaFields = $scope.vis.indexPattern.metaFields;
        const visStateByName = $scope.vis.indexPattern.fields.byName
        Object.keys(visStateByName).forEach((field) => {
          if(
            visStateByName[field].visualizable 
            && metaFields.indexOf(visStateByName[field].name) === -1 
            && !visStateByName[field].name.endsWith('.geohash')
          ) {
            $scope.processState.push(field);
          };
        });
        $scope.noActionsOnBar = true;
        const ip = $scope.savedVis.vis.indexPattern;
        const index = ip.title;
        const BATCH_SIZE = 2;
        $scope.gettingSDC = true;
        const perf = new StopWatch('sdc', 'total');
        perf.start();
        const updateFields = compareFields($scope.savedVis.vis);
        updateFields.indexesToDelete.forEach((id) => {
          $scope.visStateManager().deleteSetting(id);
        });
        const limit = promiseLimit(BATCH_SIZE);
        if (updateFields.fieldsToAdd.length === 0) $scope.noActionsOnBar = false;
        Promise.all(updateFields.fieldsToAdd.map(field => {
          return limit(() => {
            return multiChartSDC(index, field).then(resp => {
              if (resp.vis !== 'N/A') {
                const ac = new AggConfigs($scope.vis, resp.aggs);
                let name = namesLogic.namesConflict(resp.field, $scope.visStateManager()) ? resp.field + '.field' : resp.field;
                if (namesLogic.namesConflict(name, $scope.visStateManager())) {
                  name = namesLogic.generateCopyName($scope.visStateManager(), name);
                };
                $scope.visStateManager().addSetting(name, resp.vis, ac.raw.map(anAgg => {
                  return anAgg.toJSON();
                }));
                $scope.visStateManager().addMode(resp.field, resp.vis, !resp.params ? {} : resp.params);
              } else {
                updateFields.addedCount--;
              };
            });
          });
        })).then(() => {
          perf.stop();
          $scope.gettingSDC = false;
          let settingName = $scope.visStateManager().state.settings[0].name;
          $rootScope.$broadcast('multi_chart:change_setting', settingName);
          if (updateFields.addedCount === 0 && updateFields.removedCount === 0) {
            notify.info('All configurations are up to date');
          } else {
            let addedCase = updateFields.addedCount !== 1 ? ' configurations were added, ' : ' configuration was added, ';
            let removedCase = updateFields.removedCount !== 1 ? ' configurations were removed' : ' configuration was removed';
            notify.info(updateFields.addedCount + addedCase + updateFields.removedCount + removedCase);
          };
          $scope.noActionsOnBar = false;
        }).catch(reason => {
          notify.error(reason.message);
          $scope.gettingSDC = false;
          $scope.noActionsOnBar = false;
        });
      }

      $scope.getSDC = function () {
        const configurationsNo = $scope.visStateManager().state.settings.length;

        if (configurationsNo > 1) {
          const confirmModalOptions = {
            confirmButtonText: 'Yes',
            cancelButtonText: 'No',
            onConfirm: doGetSDC
          };
          confirmModal(
            'This will create automatic configurations for each field that currently does not have one. '
            + 'This might take some time on very large indexes.'
            + '\nThis will also remove configurations for fields that have changed or have been removed.'
            + '\nContinue?',
            confirmModalOptions
          );
        } else {
          doGetSDC();
        }

      };
    }
  };
});
