import './sdc_query';
import _ from 'lodash';
import { uiModules } from 'ui/modules';
const module = uiModules.get('kibana/multi_chart_vis');
import StopWatch from './stop_watch';

module.service('multiChartSDC', function (multiChartSDCQuery) {
  return function (index, field) {

    const DIST_ANALYSIS_ELEMENT_COUNT = 70;
    const TERM_ELEMENT_COUNT = 35;
    const TERM_ELEMENT_COUNT_4_TABLE = 100;
    const SAMPLE_SIZE = 10000;
    const PRECISION_THRESHOLD = 100;

    function uniqueCount() {
      let request = {
        '1': {
          cardinality: {
            precision_threshold: PRECISION_THRESHOLD
          }
        }
      };
      if (field.scripted) {
        _.merge(request['1'].cardinality, { script: { inline: field.script, lang: field.lang }});
      } else {
        _.merge(request['1'].cardinality, { field: field.name });
      }
      const perf = new StopWatch(field.name, 'cardinality');
      perf.start();
      return multiChartSDCQuery.searchAgg(index, request).then(resp => {
        perf.stop();
        return {
          total: resp.hits.total,
          unique: resp.aggregations['1'].value
        };
      });
    };

    function evalDistribution(size) {
      let request = {
        '2': {
          terms: {
            size: size,
            order: {
              _count: 'desc'
            }
          }
        }
      };
      if (field.scripted) {
        _.merge(request['2'].terms, { script: { inline: field.script, lang: field.lang }});
      } else {
        _.merge(request['2'].terms, { field: field.name });
      }
      const perf = new StopWatch(field.name, 'terms');
      perf.start();
      return multiChartSDCQuery.searchAgg(index, request).then(resp => {
        perf.stop();
        let bkts = resp.aggregations['2'].buckets;
        if (bkts.length === 0) {
          return 0;
        }
        const buckets = [];
        bkts.forEach((bucket) => {
          buckets.push(bucket.doc_count);
        });
        buckets.sort((a, b) => { return a - b; });
        const low = Math.ceil(buckets.length * 0.025);
        const high = Math.floor(buckets.length - low);
        if (low > high) {
          return 0;
        }
        let lowsum = 0;
        for (let i = 0; i < low; i++) {
          lowsum += buckets[i];
        }
        let highsum = 0;
        for (let i = high; i < buckets.length; i++) {
          highsum += buckets[i];
        }
        return lowsum / highsum;
      });

    }

    function createResponse(type, agg, sizeObj) {
      let result = {
        vis: type,
        field: field.displayName,
        aggs: [{
          type: 'count',
          schema: 'metric'
        }, {
          type: agg,
          schema: 'segment',
          params: {
            field: field.name
          }
        }]
      };
      _.merge(result.aggs[1].params, sizeObj);
      return result;
    }

    function minOne(size) {
      // size 0 or NaN is not valid value for an aggregation request
      if (size === 0 || isNaN(size)) {
        return 1;
      };
      return size;
    };

    function analyzeString() {
      return new Promise((resolve, reject) => {
        if (!field.aggregatable) {
          return resolve({
            vis: 'table',
            aggs: [{
              type: 'count',
              schema: 'metric'
            }]
          });
        }
        uniqueCount().then(resp => {
          const sizeValue = minOne(resp.unique);
          if (sizeValue < 10) {
            evalDistribution(sizeValue).then(result => {
              if (result < 0.005) {
                // data has outliers
                return resolve(createResponse('table', 'terms', {size: sizeValue}));
              } else {
                return resolve(createResponse('pie', 'terms', {size: sizeValue}));
              }
            }).catch(reason => {
              reject(reason);
            });
          } else if (resp.unique < 100) {
            if (field.type === 'text') {
              return resolve(createResponse('tagcloud', 'terms', {size: resp.unique}));
            } else {
              return resolve(createResponse('pie', 'terms', {size: resp.unique}));
            }
          } else {
            const idtest = resp.unique / resp.total;
            if (idtest < 0.02) {
              // fewer unique values
              evalDistribution(DIST_ANALYSIS_ELEMENT_COUNT).then(result => {
                if (result < 0.005) {
                  // data has outliers
                  if (field.type === 'text') {
                    return resolve(createResponse('tagcloud', 'terms', {size: TERM_ELEMENT_COUNT}));
                  } else {
                    if (result < 0.002) {
                      return resolve(createResponse('table', 'terms', {size: TERM_ELEMENT_COUNT_4_TABLE}));
                    } else {
                      return resolve(createResponse('histogram', 'terms', {size: TERM_ELEMENT_COUNT}));
                    }
                  }
                }
                if (field.type === 'text') {
                  return resolve(createResponse('tagcloud', 'terms', {size: TERM_ELEMENT_COUNT}));
                } else {
                  return resolve(createResponse('histogram', 'terms', {size: TERM_ELEMENT_COUNT}));
                }
              }).catch(reason => {
                reject(reason);
              });
            } else {
              // mostly unique values
              evalDistribution(DIST_ANALYSIS_ELEMENT_COUNT).then(result => {
                if (result < 0.005) {
                  // data has outliers
                  if (field.type === 'text') {
                    return resolve(createResponse('tagcloud', 'terms', {size: TERM_ELEMENT_COUNT}));
                  } else {
                    return resolve(createResponse('table', 'terms', {size: TERM_ELEMENT_COUNT_4_TABLE}));
                  }
                }
                if (field.type === 'text') {
                  return resolve(createResponse('tagcloud', 'terms', {size: TERM_ELEMENT_COUNT}));
                } else if (result > 0.1) {
                  // data could be mostly IDs
                  return resolve(createResponse('table', 'terms', {size: TERM_ELEMENT_COUNT}));
                } else {
                  return resolve(createResponse('histogram', 'terms', {size: TERM_ELEMENT_COUNT}));
                }
              }).catch(reason => {
                reject(reason);
              });
            }
          }
        }).catch(reason => {
          reject(reason);
        });
      });
    }

    function quartiles(array) {
      const q1 = Math.ceil(array.length * 0.025);
      const q2 = Math.floor(q1 * 2);
      const q3 = Math.floor(q1 * 3);
      const q4 = Math.floor(array.length - q1);
      return {q1, q2, q3, q4};
    }

    function evalOrdinalVariability() {
      let perf = new StopWatch(field.name, 'randomscore');
      perf.start();
      return multiChartSDCQuery.searchRandom(index, field, SAMPLE_SIZE).then(resp => {
        perf.stop();
        const data = [];
        resp.hits.hits.forEach((hit) => {
          data.push(hit._source[field.name]);
        });
        data.sort((a, b) => { return a - b; });
        let values = [];
        for (let i = 0; i < data.length - 1; i++) {
          if (data[i] !== data[i + 1]) {
            values.push(data[i]);
          }
        }
        values.sort((a, b) => { return a - b;});
        const qV = quartiles(values);

        perf = new StopWatch(field.name, 'minmax');
        perf.start();
        return multiChartSDCQuery.searchAgg(index, {
          1: {
            min: {
              field: field.name
            }
          },
          2: {
            max: {
              field: field.name
            }
          }
        }).then(resp => {
          perf.stop();
          return {
            dups: Math.max(1, values.length),
            q1: values[qV.q1],
            q2: values[qV.q2],
            q3: values[qV.q3],
            q4: values[qV.q4],
            min: resp.aggregations['1'].value,
            max: resp.aggregations['2'].value
          };
        });
      });
    }

    function analyzeNumber() {
      return new Promise((resolve, reject) => {
        if (!field.aggregatable) {
          return resolve({
            vis: 'table',
            aggs: [{
              type: 'count',
              schema: 'metric'
            }]
          });
        }
        uniqueCount().then(resp => {
          const sizeValue = minOne(resp.unique);
          if (sizeValue < 10) {
            evalDistribution(sizeValue).then(result => {
              if (result < 0.005) {
                // data has outliers
                return resolve(createResponse('table', 'histogram', {interval: 1}));
              } else {
                return resolve(createResponse('pie', 'histogram', {interval: 1}));
              }
            }).catch(reason => {
              reject(reason);
            });
          } else {
            const idtest = resp.unique / resp.total;
            if (idtest < 0.02) {
              // fewer unique values
              if (resp.unique < 100) {
                return resolve(createResponse('histogram', 'histogram', {interval: 1}));
              } else {
                evalOrdinalVariability().then(variability => {
                  let bestInterval = Math.max(variability.q1 - variability.min, variability.q3 - variability.q2);
                  bestInterval = minOne(Math.max(bestInterval, variability.max - variability.q4));
                  return resolve(createResponse('histogram', 'histogram', {interval: bestInterval}));
                }).catch(reason => {
                  reject(reason);
                });
              }
            } else {
              evalDistribution(DIST_ANALYSIS_ELEMENT_COUNT).then(result => {
                evalOrdinalVariability().then(variability => {
                  if (result < 0.008) {
                    // data has outliers
                    return resolve(createResponse('histogram', 'range', {
                      ranges: [{
                        from: variability.min,
                        to: variability.q1
                      }, {
                        from: variability.q1,
                        to: variability.q2
                      }, {
                        from: variability.q2,
                        to: variability.q3
                      }, {
                        from: variability.q3,
                        to: variability.q4
                      }, {
                        from: variability.q4,
                        to: variability.max
                      }]
                    }));
                  }
                  if (result > 0.1) {
                    return resolve(createResponse('table', 'terms', {size: TERM_ELEMENT_COUNT}));
                  } else {
                    return resolve(createResponse('histogram', 'range', {
                      ranges: [{
                        from: variability.min,
                        to: variability.q1
                      }, {
                        from: variability.q1,
                        to: variability.q2
                      }, {
                        from: variability.q2,
                        to: variability.q3
                      }, {
                        from: variability.q3,
                        to: variability.q4
                      }, {
                        from: variability.q4,
                        to: variability.max
                      }]
                    }));
                  }
                }).catch(reason => {
                  reject(reason);
                });
              }).catch(reason => {
                reject(reason);
              });
            }
          }
        }).catch(reason => {
          reject(reason);
        });
      });
    }

    let output;

    if (field.type === 'number') {
      output = analyzeNumber();
    } else if (field.type === 'string' || field.type === 'text') {
      output = analyzeString();
    } else if (field.type === 'date') {
      output = new Promise(resolve => {
        resolve(createResponse('line', 'date_histogram', {
          interval: 'M',
          min_doc_count: 1,
          customInterval: '2h',
          extended_bounds: {}
        }));
      });
    } else if (field.type === 'boolean') {
      output = new Promise(resolve => {
        resolve(createResponse('pie', 'terms', {size: 2}));
      });
    } else {
      output = new Promise(resolve => {
        resolve({
          vis: 'N/A'
        });
      });
    }

    return output;
  };
});
