import _ from 'lodash';
import { IndexedArray } from 'ui/indexed_array';

/*
  State management for multichart

  uiState:
  ~~~~~~~
  {
    activeSetting: 'default',
    activeMode: 'line',
    settings: {
      default: {
        table: {
          vis: {
            params: { ... }
          }
        },
        ...
      },
      ...
    },
    options: {
      typeSelector: true,
      configSelector: true,
      circleButtons: false
    }
  }

  visState:
  ~~~~~~~~
  title: 'multichart',
  type: 'multi_chart_vis',
  params: {
    activeSetting: 'default',
    settings: [
      0: {
        name: 'default',
        activeMode: 'line',
        aggs: [...],
        modes: {
          line: {
            type: 'line',
            params: {...}
          },
          ...
        }
      },
      1: {
        name: 'Top 5 companies',
        activeMode: 'pie',
        aggs: [...],
        modes: { ... }
      },
      ...
    ]
  },
  aggs: [{ id: 1, enabled: true, type: count, schema: metric, params: {} }],
  listeners: {}

*/

export default class VisStateManager {
  constructor() {
  }

  get activeSetting() {
    return this.settings.all.byName[this.store.activeSetting];
  }

  set activeSetting(value) {
    if (this.settings.all.byName[value]) {
      this.store.activeSetting = value;
    }
  }

  get activeMode() {
    return this.activeSetting.modes[this.activeSetting.activeMode];
  }

  set activeMode(value) {
    if (!this.activeSetting.modes[value]) {
      this.activeSetting.modes[value] = {
        type: value,
        params: {}
      };
    }
    this.activeSetting.activeMode = value;
  }

  addSetting(name, mode, aggs) {
    const newSetting = {
      name: name,
      activeMode: mode,
      modes: {}
    };
    if (aggs) {
      newSetting.aggs = aggs;
    }
    newSetting.modes[mode] = {
      type: mode,
      params: {}
    };
    this.settings.push(newSetting);
    this._setupIndices();
  }

  deleteSetting(index) {
    const configNames = this.settings.map(item => item.name);
    const stateConfigNames = this.settings.map(item => item.name);
    configNames.sort(function (a, b) {
      const aLowerCase = a.toLowerCase();
      const bLowerCase = b.toLowerCase();
      if (aLowerCase > bLowerCase) {
        return 1
      };
      if (aLowerCase < bLowerCase) {
        return -1
      };
      return 0;
    });
    const ngIndex = configNames.indexOf(this.settings[index].name);
    if (this.settings[index].name === this.store.activeSetting) {
      if (ngIndex === 0 && this.settings.length > 1) {
        this.store.activeSetting = this.settings[stateConfigNames.indexOf(configNames[1])].name;
      } else {
        this.store.activeSetting = this.settings[stateConfigNames.indexOf(configNames[0])].name;
      } 
    }
    this.settings.splice(index, 1);
    this._setupIndices();
  }

  renameSetting(index, name) {
    if (name.length === 0 || !name.trim()) {
      return false;
    }
    if (this.settings[index].name === this.store.activeSetting) {
      this.store.activeSetting = name;
    }
    this.settings[index].name = name;
    this._setupIndices();
    return true;
  }

  addMode(name, modeName, params) {
    let modeParams = params ? _.clone(params) : {};
    if (this.settings.all.byName[name].modes[modeName]) {
      _.assign(this.settings.all.byName[name].modes[modeName].params, modeParams);
    } else {
      let newMode = {
        type: modeName,
        params: modeParams
      };
      this.settings.all.byName[name].modes[modeName] = newMode;
    }
  }

  _initStorage() {
    const settings = [{
      name: 'default',
      activeMode: 'line',
      modes: {}
    }];
    const modes = {};
    modes.line = {
      type: 'line',
      params: {}
    };
    settings[0].modes.line = modes.line;
    const store = {
      activeSetting: 'default',
      settings: settings
    };
    return store;
  }

  get state() {
    if (!this.store) {
      this.clearState();
    }
    return _.clone(this.store);
  }

  set state(aStore) {
    this.store = _.clone(!aStore ? this._initStorage() : aStore);
    this.settings = this.store.settings;
    this.savedActiveSetting = this.store.activeSetting;
    this._setupIndices();
  }

  clearState() {
    this.state = false;
  }

  _setupIndices() {
    this.settings.all = new IndexedArray({
      index: ['name'],
      immutable: true,
      initialSet: this.settings
    });
  }

};
