
import { assert, lowerBound, upperBound } from './utils';

/*eslint import/namespace: 0*/
import * as d3 from './d3';
import * as d3Selection from 'd3-selection';
import _ from 'lodash';


// Constants

export const defaultMinColDistance = 150;
export const defaultMinRefSizeX = 400;

export const padLeft = 60;
export const padRight = padLeft;


// Local Functions

function translateStep(xLen, len) {
  return len > 1 ? xLen / (len - 1) : 0;
}

function numericScale(col, colRange) {
  switch (col.scaleType) {
    case 'linear':
      return d3.scaleLinear()
        .domain(colRange)
        .range([0, 1])
        .clamp(true);

    case 'log':
      return d3.scaleLog()
        .domain([Math.max(colRange[0], 1), colRange[1]])
        .range([0, 1])
        .clamp(true);

    case 'sqrt':
      return d3.scaleSqrt()
        .domain([Math.max(colRange[0], 0), colRange[1]])
        .range([0, 1])
        .clamp(true);
  }
}

function makeRangedToText(toText) {
  return function (colType, colRange, sample) {
    colType = colType.slice(0, -6);                   // Remove _range suffix

    const from = (sample.from !== null)
      ? toText(colType, colRange.values, sample.from)
      : '*';

    const to = (sample.to !== null)
      ? toText(colType, colRange.values, sample.to)
      : '*';

    return `[${from}, ${to})`;
  };
}

const rangeToText = makeRangedToText(sampleToText);
const rangeToShortText = makeRangedToText(sampleToShortText);


// Exports

// Geometry

export function rectToSize(rect) {
  return [ rect[2] - rect[0], rect[3] - rect[1] ];
}

export function rectToBox(rect) {
  return [ rect[0], rect[1], rect[2] - rect[0], rect[3] - rect[1] ];
}

export function boxToRect(box) {
  return [ box[0], box[1], box[0] + box[2], box[1] + box[3] ];
}

export function rectContains(rect, pos) {
  return rect[0] <= pos[0] && pos[0] <= rect[2]
    && rect[1] <= pos[1] && pos[1] <= rect[3];
}

export function movedInsideRect(rect, pos) {
  return [
    Math.min(Math.max(pos[0], rect[0]), rect[2]),
    Math.min(Math.max(pos[1], rect[1]), rect[3])
  ];
}

export function translator(xLen, len) {
  let step = translateStep(xLen, len);

  return function (d, c) {
    return `translate(${c * step} 0)`;
  };
}

export function closestPoint(lines, lineToX, lineToY, x, y) {
  let min = Infinity;

  const line = _(lines).reduce(function (memo, line) {
    const dx = lineToX(line) - x;
    const dy = lineToY(line) - y;

    const val = dx * dx + dy * dy;
    return (val <= min) ? (min = val, line) : memo;
  }, null);

  return line;
}

export function sampleToText(colType, colRange, sample) {
  if (sample === null) { return '<no value>'; }

  switch (colType) {
    case 'number':
      return colRange[1] <= -1e5 || 1e5 <= colRange[1]
        ? d3.format(',')(_.round(sample, 2))
        : '' + _.round(sample, 2);

    case 'integer':
      return colRange[1] <= -1e5 || 1e5 <= colRange[1]
        ? d3.format(',')(_.round(sample))
        : '' + _.round(sample);

    case 'date':
      return sample.toLocaleString();

    case 'number_range':
    case 'integer_range':
    case 'date_range':
      return rangeToText(colType, colRange, sample);

    default:
      return sample;
  }
}

export function sampleToShortText(column, colRange, sample) {
  if (sample === null) { return ''; }

  const { type } = column;

  switch (type) {
    case 'string':
      return sample.slice(0, column.shortFormat || 20);

    case 'date':
      return sample.toLocaleDateString();

    case 'date_range':
      return rangeToShortText(type, colRange, sample);

    default:
      return sampleToText(type, colRange, sample);
  }
}

export function lowerSampleInValueRange(colType, scale, v0, v1) {
  if (scale.invert) { return scale.invert(v0); }

  const domain = scale.domain();
  const lo = (v0 <= 0)
    ? 0
    : lowerBound(domain, v0, (s, dom) => scale(dom[s]));

  return lo < domain.length && scale(domain[lo]) <= v1 ? domain[lo] : '';
}

export function upperSampleInValueRange(colType, scale, v0, v1) {
  if (scale.invert) { return scale.invert(v1); }

  const domain = scale.domain();
  const hi = (v1 >= 1)
    ? domain.length - 1
    : upperBound(domain, v1, (s, dom) => scale(dom[s])) - 1;

  return hi >= 0 && v0 <= scale(domain[hi]) ? domain[hi] : '';
}


// Interaction

export function rawMousePos(ctx, event) {
  // Rationale: mouse hover events don't have an event in d3.event,
  // so we have to supply it manually.

  // See https://brianschiller.com/blog/2017/09/28/d3-event-issues
  // I tried customEvent, too - but couldn't make it work; it should
  // probably implemented that way however. If this breaks in the
  // future, look into it.

  const currEvent = d3Selection.event;

  d3Selection.event = event;
  const result = d3.mouse(d3.select(ctx.svg).node());
  d3Selection.event = currEvent;

  return result;
}

export function mousePos(ctx, event) {
  const rawPos = rawMousePos(ctx, event);
  return rawPos && movedInsideRect(ctx.contentRect, rawPos);
}

export function preventClick(selection) {
  selection.on('click.parallelprevent', function blockEvent() {
    d3.event.preventDefault();
    d3.event.stopPropagation();
  }, true);   // capture

  _.defer(function removePreventClick() { selection.on('click.parallelprevent', null); });  // eslint-disable-line memoryleaks
}

export function filterDefineEnd(ctx) {
  preventClick(d3.select(ctx.svg));

  ctx.filterDrag = null;

  ctx.view.colFilters = _.transform(ctx.view.colFilters, function (memo, filter, key) {
    if (filter.type !== 'range' || filter.v0 < filter.v1) { memo[key] = filter; }
  }, {});

  ctx.viewChanges.colFilters = true;

  refresh(ctx);
}

export function resetHover(ctx) {
  if (!ctx.view.hover.line) { return; }

  ctx.view.hover = {};
  ctx.viewChanges.hover = true;

  refresh(ctx);
}


// Rendering

export function materialize(parent, appendName, className, data, key) {
  let selector = appendName;
  if (className) { selector += '.' + className.replace(' ', '.'); }

  const all = parent.selectAll(selector).data(data, key);
  all.exit().remove();

  const enter = all.enter().append(appendName)
    .attr('class', className);

  return { all: enter.merge(all), enter };
}

export function columnTicks(column, scale, columnSize, ticksDistance) {
  const domain = scale.domain();

  ticksDistance = Math.max(20, ticksDistance || 0);
  let ticksCount = Math.max(Math.floor(columnSize / ticksDistance), 3);

  switch (column.type) {
    case 'string':
    case 'ip':
    case 'number_range':
    case 'integer_range':
    case 'date_range':
      ticksCount = Math.min(ticksCount, domain.length - 1);
      return _.map(d3.ticks(0, domain.length - 1, ticksCount),
        s => domain[s]);

    case 'integer':
      ticksCount = Math.min(ticksCount, Math.abs(domain[1] - domain[0]) + 1);
      // continue
  }

  if (column.scaleType !== 'log') {
    return scale.ticks(ticksCount);
  } else {
    const magnitudesCount = 3;

    // See d3-scale docs, log scale ticks are magnitude-based...
    const ticks = scale.ticks(magnitudesCount);
    const tickFormat = scale.tickFormat(ticksCount);

    return ticks.map(tickFormat);
  }
}

export function refresh(ctx) {
  if (_.isEmpty(ctx.viewChanges)) { return; }

  ctx.onViewChanges(ctx.viewChanges);
  ctx.viewChanges = {};
}


// State

export function scalesFromColumns(columns, colRanges) {
  return _.map(columns, (col, c) => {
    const colRange = colRanges[c];

    switch (col.type) {
      case 'number':
      case 'integer':
        return numericScale(col, colRange);

      case 'date':
        return d3.scaleTime()
          .domain(colRange)
          .range([0, 1]);

      case 'string':
      case 'ip':
        return d3.scalePoint()
          .domain(colRange)
          .range([0, 1]);

      case 'number_range':
      case 'integer_range':
      case 'date_range':
        const rawScale = numericScale(col, colRange.values);
        const scale = _.flow(_.iteratee('value'), rawScale);

        _.assign(scale, _.pick(rawScale, [ 'range', 'ticks', 'tickFormat' ]));
        scale.domain = () => colRange.samples;

        return scale;

      default:
        assert(false, `Unrecognized column type '${col.type}'`);
    }
  });
}

export function scaleFromColors(colors, colorRange) {
  colors = colors || {};

  const lowCountColor = colors.low || 'black';
  const highCountColor = colors.high || 'black';


  return d3
    .scaleSequential(d3.interpolateRgb(lowCountColor, highCountColor))
    .domain(colorRange);
}

export function buildStripSizes(ctx, padding) {
  let result = {};

  const minColDistance = ctx.minColDistance || defaultMinColDistance;
  const minRefSizeX = ctx.minRefSizeX || defaultMinRefSizeX;

  result.refSizeX = Math.max((ctx.columns.length - 1) * minColDistance, minRefSizeX);
  result.refSizeY = Math.max(ctx.sizeY - padding.top - padding.bottom, 1);
  result.xStep = translateStep(result.refSizeX, ctx.columns.length);

  let surplusPaddingX = Math.max(
    (ctx.sizeX - result.refSizeX - padding.left - padding.right) / 2,
    0);

  result.paddingLeft = padding.left + surplusPaddingX;
  result.paddingRight = padding.right + surplusPaddingX;

  result.sizeX = result.refSizeX + result.paddingLeft + result.paddingRight;

  result.viewBox = [
    -result.paddingLeft, -padding.top,
    result.sizeX, ctx.sizeY
  ];

  result.viewRect = [
    -result.paddingLeft, -padding.top,
    result.sizeX - result.paddingLeft, ctx.sizeY - padding.top
  ];

  result.contentRect = [ 0, 0, result.refSizeX, result.refSizeY ];

  return result;
}

export function buildHighlights(ctx) {
  const { view, colors } = ctx;

  return _(ctx.highlights)
    .map((hl, h) => _.defaults({
      get lines() { return ctx.filtered.highlights[h]; }
    }, hl))
    .concat([{
      id: 'hover',
      get lines() {
        const { line } = view.hover;
        return line ? [ line ] : [];
      },
      colors,
    }])
    .value();
}
