
import _ from 'lodash';


function fieldType(field) {
  if (!field) { return ''; }

  let type;

  switch (field.esType) {
    case 'byte': case 'short': case 'integer': case 'long':
      type = 'integer';
      break;

    default:
      type = field.type;
  }

  return type;
}

function columnFromAgg(agg) {
  const aggTypeName = agg.type && agg.type.name;
  const aggParams = agg.params || {};
  const isAggregator = (agg.schema.group === 'buckets');

  const { field, customLabel } = aggParams;

  const fieldName = field ? field.displayName : '';
  const fieldLabel = `${fieldName}[${aggTypeName}]`;

  let type;

  switch (aggTypeName) {
    case 'count':
    case 'cardinality':
      type = 'integer';
      break;

    case 'range':
    case 'date_range':
      type = fieldType(field) + '_range';
      break;

    default:
      type = fieldType(field);
  }

  const defaultSort = _([
    'number', 'integer', 'date',
    'number_range', 'integer_range', 'date_range'
  ]).indexBy().mapValues(_.constant('asc')).value();

  // Show param will have 3 values: true, false, undefined.
  //
  // The 'undefined' value is used to identify non-user-explicit reasons to
  // hide the column, such as the column being not enabled or the agg being
  // partially defined.
  const show = agg.type && (agg.enabled || undefined);

  return {
    isAggregator,
    id: agg.id,
    label: customLabel || fieldLabel,
    type,
    aggType: aggTypeName,
    scaleType: 'linear',
    sort: defaultSort[type] || 'none',
    show
  };
}

function getColumnValue(column, value) {
  switch (column.type) {
    case 'date':
    case 'date_range':
      // Dates are typically extracted from ES as numbers - cast them to Dates
      value = new Date(+value);
      break;

    default:
  }

  return value;
}

function getMetricValue(agg, column, bucket) {
  return getColumnValue(column, agg.getValue(bucket));
}

function getAggregatorValue(agg, column, bucket) {
  switch (agg.type.name) {
    case 'range':
    case 'date_range':
      // Note that ranges can be unbounded (no from, no to or *neither*)
      const hasFrom = bucket.hasOwnProperty('from');
      const hasTo = bucket.hasOwnProperty('to');

      let value;

      if (hasFrom && hasTo) { value = (bucket.from + bucket.to) / 2; }
      else if (hasFrom) { value = bucket.from; }
      else if (hasTo) { value = bucket.to; }
      else { value = 0; }

      const from = hasFrom ? getColumnValue(column, bucket.from) : null;
      const to = hasTo ? getColumnValue(column, bucket.to) : null;

      return { from, to, value };

    default:
      return getColumnValue(column, bucket.key);
  }
}

function addRecords(columns, aggs, records, aggregations) {
  const columnsById = _.indexBy(columns, 'id');
  const bucketAggs = _.filter(aggs.bySchemaGroup.buckets, 'enabled');

  function addRecordsRecurse(bucket, level, values) {
    if (level === bucketAggs.length) {
      if (!bucket.doc_count) { return; }

      const record = _(columns)
        .map(col => col.isAggregator
          ? values[col.id]
          : getMetricValue(aggs.byId[col.id], col, bucket))
        .concat([ bucket.doc_count || 0 ])
        .value();

      records.push(record);
      return;
    }

    const agg = bucketAggs[level++];
    const aggId = agg.id;
    const column = columnsById[aggId];
    const childBuckets = bucket[aggId].buckets;

    _.forEach(childBuckets, childBucket => {
      values[aggId] = getAggregatorValue(agg, column, childBucket);
      addRecordsRecurse(childBucket, level, values);
    });
  }

  return addRecordsRecurse(aggregations, 0, {});
}


// Exports

export function columnsFromAggs(
  aggs, oldEditableColumns, chartColumns,
  opts = { sortByChart: true }
) {
  let columns = _.map(aggs, columnFromAgg);

  const oldColumnsById = _.indexBy(oldEditableColumns, 'id');
  const chartDataById = _.reduce(chartColumns, function (memo, col, pos) {
    memo[col.id] = { col, pos };
    return memo;
  }, {});

  // Apply existing columns data
  _.forEach(columns, col => {
    const chartData = chartDataById[col.id];
    if (chartData) {
      col.sort = chartData.col.sort;
      col.show = col.show && (chartData.col.show !== false);    // undefined => true
    }

    const oldColumn = oldColumnsById[col.id];
    if (oldColumn) { col.scaleType = oldColumn.scaleType; }
  });

  if (opts.sortByChart) {
    // Sort columns by chart position
    columns = _.sortBy(columns, col => {
      const chartData = chartDataById[col.id];
      return chartData ? chartData.pos : Number.MAX_SAFE_INTEGER;
    });
  }

  return columns;
}

export function recordsFromResponse(columns, aggs, resp) {
  if (!aggs || !resp) { return []; }

  const { aggregations } = resp;
  if (!aggregations) { return []; }

  const records = [];
  addRecords(columns, aggs, records, aggregations);

  return records;
}

