
import * as cc from './chartcommons';
import { assert, moveIndex, arrayMoveValue, paddedIp } from './utils';

import _ from 'lodash';


function idToIndexFromColumns(columns) {
  return _.reduce(columns, function (memo, col, c) {
    memo[col.id] = c;
    return memo;
  }, {});
}

function computeColumnRanges(columns, records, colors) {
  const sortIteratees = {
    string: _.identity,
    ip: paddedIp
  };

  return _.map(columns, function colToRange(col, c) {
    let values = _(records).map(c).filter(val => val !== null);

    switch (col.type) {
      case 'string':
      case 'ip':
        if (col.sort !== 'none') {
          return values.sortByOrder(sortIteratees[col.type], col.sort).uniq(true).value();
        }

        if (colors.order === colors.defaultOrdinalColumnsOrder) {
          values = values.reverse();
        }

        return values.uniq().reverse().value();

      case 'number_range':
      case 'integer_range':
      case 'date_range':
        const sort = (col.sort === 'desc') ? 'desc' : 'asc';
        const samples = values.sortByOrder('value', sort).value();
        const lo = samples[0];
        const hi = samples[samples.length - 1];

        return {
          values: [ lo && lo.value, hi && hi.value ],
          samples
        };

      default:
        const min = (col.min !== undefined) ? col.min : values.min();
        const max = (col.max !== undefined) ? col.max : values.max();

        return (col.sort === 'desc') ? [ max, min ] : [ min, max ];
    }
  });
}


// Color Functions

function colorRangeFromSortedRecords(records, colorIndex) {
  records = [ records[0], records[records.length - 1] ];

  const values = _(records).compact().map(colorIndex);

  let min = values.min();
  let max = values.max();

  if (min === max) { min -= 1; max += 1; }

  return [ min, max ];
}


// Columns Visibility Functions

function visibleRecords(visibleToAll, records, rd) {
  if (!rd.onlyWholeLines) { return records; }

  return _.filter(records, rec =>
    _.every(visibleToAll, v => rec[v] !== null));
}

function visibleColumns(columns) {
  return _.filter(columns, 'show');
}

function visibleColArray(columns, array) {
  return _.filter(array, (unused, c) => columns[c].show);
}

function visibleScales(columns, scales) {
  return {
    columns: visibleColArray(columns, scales.columns),
    color: scales.color
  };
}

function visibleColumnsMap(columns) {
  let fc = 0;
  return _.map(columns, (col, c) => col.show ? fc++ : -1);
}

function allColumnsMap(columns) {
  return _(columns.length).range()
    .filter(c => columns[c].show)
    .value();
}

function visibleColumnsData(rd) {
  const { columns, records, colRanges, scales } = rd.all;

  let result = {
    columns: visibleColumns(columns),
    colRanges: visibleColArray(columns, colRanges),
    scales: visibleScales(columns, scales),
    allToVisible: visibleColumnsMap(columns),
    visibleToAll: allColumnsMap(columns)
  };

  result.idToIndex = idToIndexFromColumns(result.columns);
  result.records = visibleRecords(result.visibleToAll, records, rd);

  return result;
}


// Record Filtering Functions

function isLineFilteredIn(rd, colFilters, line) {
  const { columns, scales, idToIndex } = rd.all;
  const { allToVisible } = rd;

  return _.every(colFilters, function (filter, colId) {
    if (allToVisible[colId] < 0 && !filter.force) { return true; }

    const c = idToIndex[colId];

    const value = line[c];
    if (value === null) { return false; }

    const scaleValue = scales.columns[c](value);
    return filter.v0 <= scaleValue && scaleValue <= filter.v1;
  });
}

function filteredData(rd, colFilters) {
  function filterFn(record) { return isLineFilteredIn(rd, colFilters, record); }

  return {
    records: _.partition(rd.records, filterFn),
    highlights: _.map(rd.highlights, hl => _.filter(hl.lines, filterFn))
  };
}


// Exports

export function defaultRecordColor(line) {
  return this.scales.color(line[this.colors.index]);
}


export function valueFilter(ctx, colId, sample) {
  const c = ctx.all.idToIndex[colId];
  const v = ctx.all.scales.columns[c](sample);

  return {
    type: 'value',
    v0: v,
    v1: v,
    s0: null,
    s1: sample,
    highlighted: false
  };
}

export function rangeFilter(ctx, colId, v0, v1) {
  const c = ctx.all.idToIndex[colId];
  const col = ctx.all.columns[c];
  const scale = ctx.all.scales.columns[c];

  v0 = Math.min(Math.max(v0, 0), 1);
  v1 = Math.min(Math.max(v1, 0), 1);

  return {
    type: 'range',
    v0,
    v1,
    s0: cc.lowerSampleInValueRange(col.type, scale, v0, v1),
    s1: cc.upperSampleInValueRange(col.type, scale, v0, v1)
  };
}

export default class RenderData {
  constructor(data) {
    data = _.defaults({}, data, {
      columns: [],
      records: [],
      view: {},
      highlights: []
    });

    const colors = _.defaults({}, data.colors, {
      low: 'red',
      high: 'forestgreen',
      highlight: 'dodgerblue',
      tip: 'dodgerblue',
      defaultOrdinalColumnsOrder: 'asc',
      order: 'asc',
      record: defaultRecordColor.bind(this)
    });

    const view = _.defaults(data.view, {
      // View components are the only ones mutable by charts
      hover: {},
      colFilters: {}
    });

    const { columns } = data;
    let { records } = data;


    const idToIndex = idToIndexFromColumns(columns);
    const colorIndex = columns.length;

    // The two-step sort-reverse is to keep it in sync with computeColumnRanges,
    // so that changing stacking order doesn't affect default ordinal columns ordering
    records = _.sortByOrder(records, colorIndex, colors.defaultOrdinalColumnsOrder);
    if (colors.order !== colors.defaultOrdinalColumnsOrder) { records.reverse(); }

    const colRanges = computeColumnRanges(columns, records, colors);
    const colorRange = colorRangeFromSortedRecords(records, colorIndex);

    const scales = {
      columns: cc.scalesFromColumns(columns, colRanges),
      color: cc.scaleFromColors(colors, colorRange),
    };

    _.assign(this, data, {
      all: { columns, records, colRanges, scales, idToIndex },
      filtered: {},

      colorRange,
      colors: _.assign({}, colors, { index: colorIndex }),

      view
    });

    this.applyVisibility();
    this.applyColumnFilters();
  }

  applyColumnFilters() {
    _.assign(this.filtered, filteredData(this, this.view.colFilters));
  }

  applyVisibility() {
    _.assign(this, visibleColumnsData(this));
  }

  moveColumn(srcIdx, dstIdx) {
    srcIdx = this.visibleToAll[srcIdx];
    dstIdx = this.visibleToAll[dstIdx];

    const { columns, colRanges, scales } = this.all;

    this.colors.index = moveIndex(this.colors.index, srcIdx, dstIdx);

    arrayMoveValue(columns, srcIdx, dstIdx);
    arrayMoveValue(colRanges, srcIdx, dstIdx);
    arrayMoveValue(scales.columns, srcIdx, dstIdx);

    _.forEach(this.all.records, rec => arrayMoveValue(rec, srcIdx, dstIdx));

    this.all.idToIndex = idToIndexFromColumns(columns);

    this.applyVisibility();
  }

  sortColumn(colIdx) {
    const column = this.columns[colIdx];
    let next;

    switch (column.type) {
      case 'string':
      case 'ip':
        next = { asc: 'desc', desc: 'none', none: 'asc' };
        break;

      default:
        next = { asc: 'desc', desc: 'asc' };
        break;
    }

    column.sort = next[column.sort] || 'asc';

    this.all.colRanges = computeColumnRanges(this.all.columns, this.records, this.colors);
    this.all.scales.columns = cc.scalesFromColumns(this.all.columns, this.all.colRanges);

    delete this.view.colFilters[column.id];

    this.applyVisibility();
    this.applyColumnFilters();
  }

  setColumnVisible(colIdx, visible) {
    const column = this.columns[colIdx];
    column.show = visible;

    this.applyVisibility();
  }

  toggleColumnVisible(colIdx) {
    const column = this.columns[colIdx];
    column.show = !column.show;

    this.applyVisibility();
  }
};

