
/*eslint import/namespace: 0*/
import * as cc from './chartcommons';
import { rangeFilter } from './renderdata';
import { assert } from './utils';
import * as d3 from './d3';
import _ from 'lodash';

import '../styles/scatterplot.less';


// Consts

const fontSize = 14;

const pointsHoverRadius = 6;

const filterRangeSize = 16;
const filterRangeSizeHalf = filterRangeSize / 2;
const filterRangeHandlesSize = 5;
const filterValueRadius = 5;
const filterTextPad = 4;
const filterTextCrossPad = filterRangeSizeHalf + 4;

const padLeft = 10;
const padRight = 40;
const padTop = 20;
const padBottom = 10;

const axisTickSize = 4;
const axisLabelPad = 10;
const axisLabelSize = fontSize + axisLabelPad;

const closePad = 10;
const tooltipPad = 30;


// Local Functions

// Geometry

function valueToX(ctx, val) {
  return ctx.chartRect[0] + val * ctx.chartSize[0];
}

function valueToY(ctx, val) {
  return ctx.chartRect[3] - val * ctx.chartSize[1];
}

function xToValue(ctx, x) {
  return (x - ctx.chartRect[0]) / ctx.chartSize[0];
}

function yToValue(ctx, y) {
  return (ctx.chartRect[3] - y) / ctx.chartSize[1];
}

const valueToPix = { x: valueToX, y: valueToY };
const pixToValue = { x: xToValue, y: yToValue };
const axisToCoord = { x: 0, y: 1 };

function hasValues(ctx) {
  const xc = ctx.scatterplotAxes.x;
  const yc = ctx.scatterplotAxes.y;

  const xl = ctx.visibleToAll[xc];
  const yl = ctx.visibleToAll[yc];

  return line => (line[xl] !== null) && (line[yl] !== null);
}

function cx(ctx) {
  const c = ctx.scatterplotAxes.x;
  const l = ctx.visibleToAll[c];

  return line => ctx.xScale(line[l]);
}

function cy(ctx) {
  const c = ctx.scatterplotAxes.y;
  const l = ctx.visibleToAll[c];

  return line => ctx.yScale(line[l]);
}

function color(ctx) {
  return line => ctx.scales.color(line[line.length - 1]);
}

function text(ctx, line) {
  return ac => {
    const colType = ctx.all.columns[ac].type;
    const colRange = ctx.all.colRanges[ac];

    return cc.sampleToText(colType, colRange, line[ac]);
  };
}

function closestLine(ctx, posX, posY) {
  const px = cx(ctx);
  const py = cy(ctx);
  const filter = hasValues(ctx);

  return cc.closestPoint(
    _(ctx.filtered.records[0]).filter(filter), px, py, posX, posY);
}


// Interaction

function updateHover(ctx) {
  const mPos = cc.rawMousePos(ctx, d3.event);
  if (!cc.rectContains(ctx.hoverRect, mPos)) { return cc.resetHover(ctx); }

  const { hover } = ctx.view;

  const line = closestLine(ctx, mPos[0], mPos[1]);
  if (line === hover.line) { return; }

  hover.line = line;
  hover.event = d3.event;

  ctx.viewChanges.hover = true;

  cc.refresh(ctx);
}


function filterDefineFilter(ctx) {
  return !d3.event.button && cc.rectContains(ctx.filterDefineRect, d3.mouse(ctx.svg));
}

function filterDefineStart(ctx) {
  const mPos = [ d3.event.x, d3.event.y ];
  const values = [ xToValue(ctx, mPos[0]), yToValue(ctx, mPos[1]) ];

  const xc = ctx.scatterplotAxes.x;
  const yc = ctx.scatterplotAxes.y;

  const xColName = ctx.columns[xc].id;
  const yColName = ctx.columns[yc].id;

  const xFilter = ctx.view.colFilters[xColName];
  const yFilter = ctx.view.colFilters[yColName];

  ctx.filterDrag = {
    x: { v: values[0], p: mPos[0], filter: xFilter },
    y: { v: values[1], p: mPos[1], filter: yFilter }
  };

  cc.resetHover(ctx);
}

function filterDefineAxisMove(ctx, axis) {
  const p = d3.event[axis];
  const v = pixToValue[axis](ctx, p);
  const vPress = ctx.filterDrag[axis].v;

  const c = ctx.scatterplotAxes[axis];
  const colName = ctx.columns[c].id;

  if (vPress >= 0) {
    const v0 = Math.min(v, vPress);
    const v1 = Math.max(v, vPress);

    ctx.view.colFilters[colName] = rangeFilter(ctx, colName, v0, v1);
  } else {
    const filter = ctx.filterDrag[axis].filter;
    if (filter) { ctx.view.colFilters[colName] = filter; }
    else { delete ctx.view.colFilters[colName]; }
  }

  ctx.viewChanges.colFilters = true;
}

function filterDefineMove(ctx) {
  filterDefineAxisMove(ctx, 'x');
  filterDefineAxisMove(ctx, 'y');

  cc.refresh(ctx);
}

function rangeFilterDragStart(ctx, axis) {
  const colName = ctx.columns[ctx.scatterplotAxes[axis]].id;
  const filter = ctx.view.colFilters[colName];

  ctx.filterDrag = {
    v0: filter.v0,
    v1: filter.v1,
    vPress: pixToValue[axis](ctx, d3.event[axis])
  };
}

function rangeFilterDragMove(ctx, axis) {
  const { v0, v1, vPress } = ctx.filterDrag;
  const drag = pixToValue[axis](ctx, d3.event[axis]) - vPress;

  const c = ctx.scatterplotAxes[axis];
  const colName = ctx.columns[c].id;

  ctx.view.colFilters[colName] = rangeFilter(ctx, colName, v0 + drag, v1 + drag);

  ctx.viewChanges.colFilters = true;
  cc.refresh(ctx);
}

function rangeFilterMinDragMove(ctx, axis) {
  const { v0, v1, vPress } = ctx.filterDrag;
  const drag = pixToValue[axis](ctx, d3.event[axis]) - vPress;

  const c = ctx.scatterplotAxes[axis];
  const colName = ctx.columns[c].id;

  ctx.view.colFilters[colName] = rangeFilter(
    ctx, colName, Math.min(v0 + drag, v1), v1);

  ctx.viewChanges.colFilters = true;
  cc.refresh(ctx);
}

function rangeFilterMaxDragMove(ctx, axis) {
  const { v0, v1, vPress } = ctx.filterDrag;
  const drag = pixToValue[axis](ctx, d3.event[axis]) - vPress;

  const c = ctx.scatterplotAxes[axis];
  const colName = ctx.columns[c].id;

  ctx.view.colFilters[colName] = rangeFilter(
    ctx, colName, v0, Math.max(v0, v1 + drag));

  ctx.viewChanges.colFilters = true;
  cc.refresh(ctx);
}


function initMouse(ctx) {
  ctx.filterDefineEnd = () => cc.filterDefineEnd(ctx);

  ctx.xRangeFilterDragStart = () => rangeFilterDragStart(ctx, 'x');
  ctx.xRangeFilterDragMove = () => rangeFilterDragMove(ctx, 'x');
  ctx.xRangeFilterMinDragMove = () => rangeFilterMinDragMove(ctx, 'x');
  ctx.xRangeFilterMaxDragMove = () => rangeFilterMaxDragMove(ctx, 'x');

  ctx.yRangeFilterDragStart = () => rangeFilterDragStart(ctx, 'y');
  ctx.yRangeFilterDragMove = () => rangeFilterDragMove(ctx, 'y');
  ctx.yRangeFilterMinDragMove = () => rangeFilterMinDragMove(ctx, 'y');
  ctx.yRangeFilterMaxDragMove = () => rangeFilterMaxDragMove(ctx, 'y');

  d3.select(ctx.svg)
    .on('mousemove.prl-bsp', () => updateHover(ctx)) // eslint-disable-line memoryleaks
    .call(d3.drag().container(ctx.svg)
      .filter(() => filterDefineFilter(ctx)) // eslint-disable-line memoryleaks
      .on('start.prl-bsp', () => filterDefineStart(ctx)) // eslint-disable-line memoryleaks
      .on('drag.prl-bsp', () => filterDefineMove(ctx)) // eslint-disable-line memoryleaks
      .on('end.prl-bsp', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks
}


// Rendering

function renderPoints(ctx, group) {
  const { pointsRadius } = ctx;

  const filteredLines = ctx.filtered.records;
  const filteredOutLines = ctx.showFilteredOutRecords ? filteredLines[1] : [];

  const x = cx(ctx);
  const y = cy(ctx);
  const filter = hasValues(ctx);


  const outGroup = cc.materialize(group, 'g', 'prl-bsp-filtered-out', [0]).all;

  let sel = cc.materialize(outGroup, 'circle', '', filteredOutLines.filter(filter));
  sel.enter
    .attr('r', pointsRadius);
  sel.all
    .attr('cx', x)
    .attr('cy', y);


  const inGroup = cc.materialize(group, 'g', 'prl-bsp-filtered-in', [0]).all;

  sel = cc.materialize(inGroup, 'circle', '', filteredLines[0].filter(filter));
  sel.enter
    .attr('r', pointsRadius);
  sel.all
    .attr('cx', x)
    .attr('cy', y)
    .attr('fill', ctx.colors.record);
}

function renderContentGroup(ctx, svg) {
  const contentGroup = cc.materialize(svg, 'g', 'prl-bsp-content-group', [0]).all;

  renderPoints(ctx, contentGroup);
}


function renderFilterBounds(ctx, group) {
  const { filterDefineBoxTL, filterDefineBoxBR } = ctx;

  let sel = cc.materialize(group, 'rect', 'prl-bsp-filter-bounds-tl', [0]);
  sel.enter
    .attr('x', filterDefineBoxTL[0]).attr('y', filterDefineBoxTL[1]);
  sel.all
    .attr('width', filterDefineBoxTL[2]).attr('height', filterDefineBoxTL[3]);

  sel = cc.materialize(group, 'rect', 'prl-bsp-filter-bounds-br', [0]);
  sel.enter
    .attr('x', filterDefineBoxBR[0]).attr('y', filterDefineBoxBR[1]);
  sel.all
    .attr('width', filterDefineBoxBR[2]).attr('height', filterDefineBoxBR[3]);
}

function axisTickHidden(ctx, c, axis) {
  const column = ctx.columns[c];

  const filter = ctx.view.colFilters[column.id];
  if (!filter) { return false; }

  const pixScale = ctx.pixScales[axisToCoord[axis]];

  let pixLo = valueToPix[axis](ctx, filter.v0);
  let pixHi = valueToPix[axis](ctx, filter.v1);

  if (pixHi < pixLo) { [ pixLo, pixHi ] = [ pixHi, pixLo ]; }

  return function (s) {
    const pix = pixScale(s);
    return pix <= pixLo || pixHi <= pix;
  };
}

function renderAxis(ctx, group) {
  const { xAxis, yAxis, xScale, yScale, xTicks, yTicks } = ctx;

  const xc = ctx.scatterplotAxes.x;
  const yc = ctx.scatterplotAxes.y;
  const xColumn = ctx.columns[xc];
  const yColumn = ctx.columns[yc];
  const xColRange = ctx.colRanges[xc];
  const yColRange = ctx.colRanges[yc];


  const axisGroup = cc.materialize(group, 'g', 'prl-bsp-axis-group', [0]).all;

  // X Axis

  const xAxisGroup = cc.materialize(axisGroup, 'g', 'prl-bsp-xaxis-group', [0]).all;
  const xTickHidden = axisTickHidden(ctx, xc, 'x');

  cc.materialize(xAxisGroup, 'line', 'prl-bsp-axis', [0])
    .all
    .attr('x1', xAxis).attr('x2', ctx.chartRect[2])
    .attr('y1', yAxis).attr('y2', yAxis);

  cc.materialize(xAxisGroup, 'line', 'prl-bsp-tick-line', xTicks)
    .all
    .attr('x1', xScale).attr('x2', xScale)
    .attr('y1', yAxis).attr('y2', yAxis + axisTickSize);

  let sel = cc.materialize(xAxisGroup, 'text', 'prl-bsp-tick-text', xTicks);
  sel.enter
    .attr('dy', '0.25em');
  sel.all
    .classed('prl-bsp-tick-hidden', xTickHidden)
    .attr('x', xScale)
    .attr('y', yAxis + axisTickSize + 2)
    .attr('transform', t => `rotate(-90 ${xScale(t)},${yAxis + axisTickSize + 2})`)
    .text(s => cc.sampleToShortText(xColumn, xColRange, s));

  // Y Axis

  const yAxisGroup = cc.materialize(axisGroup, 'g', 'prl-bsp-yaxis-group', [0]).all;
  const yTickHidden = axisTickHidden(ctx, yc, 'y');

  sel = cc.materialize(yAxisGroup, 'line', 'prl-bsp-axis', [0]);
  sel.enter
    .attr('y2', ctx.chartRect[1]);
  sel.all
    .attr('x1', xAxis).attr('x2', xAxis)
    .attr('y1', yAxis);

  cc.materialize(yAxisGroup, 'line', 'prl-bsp-tick-line', yTicks)
    .all
    .attr('x1', xAxis).attr('x2', xAxis - axisTickSize)
    .attr('y1', yScale).attr('y2', yScale);

  sel = cc.materialize(yAxisGroup, 'text', 'prl-bsp-tick-text', yTicks);
  sel.enter
    .attr('dy', '0.25em');
  sel.all
    .classed('prl-bsp-tick-hidden', yTickHidden)
    .attr('x', xAxis - axisTickSize - 2)
    .attr('y', yScale)
    .text(s => cc.sampleToShortText(yColumn, yColRange, s));
}

function renderLabels(ctx, group) {
  const labelsGroup = cc.materialize(group, 'g', 'prl-bsp-labels-group', [0]).all;

  let x = ctx.xMidChart;
  let y = ctx.contentRect[3];

  let sel = cc.materialize(labelsGroup, 'text', 'prl-bsp-axis-x-label', [0]);
  sel.enter
    .attr('font-size', fontSize);
  sel.all
    .attr('x', x).attr('y', y)
    .text(ctx.columns[ctx.scatterplotAxes.x].label);

  x = ctx.contentRect[0] + fontSize;
  y = ctx.yMidChart;

  sel = cc.materialize(labelsGroup, 'text', 'prl-bsp-axis-y-label', [0]);
  sel.enter
    .attr('font-size', fontSize);
  sel.all
    .attr('x', x).attr('y', y)
    .attr('transform', `rotate(-90 ${x},${y})`)
    .text(ctx.columns[ctx.scatterplotAxes.y].label);
}

function tooltipHtml(ctx, line, bottomUpLayout) {
  const { visibleToAll } = ctx;
  const { columns } = ctx.all;

  const txt = text(ctx, line);

  function pinnedText(c) { return columns[c].pinnedText; }

  function toRow(c) {
    return `<tr><td>${columns[c].label}</td><td>${txt(c)}</td></tr>`;
  }

  const xc = ctx.scatterplotAxes.x;
  const yc = ctx.scatterplotAxes.y;

  const pinnedCs = _(0).range(columns.length)
    .filter(pinnedText)
    .sortBy(pinnedText)
    .concat([xc, yc])
    .uniq()
    .value();

  const regularCs = _(0).range(ctx.columns.length)  // ctx.columns === visible columns
    .map(c => visibleToAll[c])
    .difference(pinnedCs)
    .value();

  let upperRows = _.map(pinnedCs, toRow);
  let lowerRows = _.map(regularCs, toRow);

  if (bottomUpLayout) {
    [ upperRows, lowerRows ] = [ lowerRows, upperRows ];
    upperRows.reverse();
  }

  const tRows = _(upperRows)
    .concat(['<tr><td colspan="2"><hr></td></tr>'])
    .concat(lowerRows)
    .join('');

  return `<table><tbody>${tRows}</tbody></table>`;
}

function renderHighlight(ctx, group, highlight) {
  const { pointsHoverRadius } = ctx;
  const { lines, colors } = highlight;

  const x = cx(ctx);
  const y = cy(ctx);

  const color = colors.highlight || 'black';


  // Projections to axis

  cc.materialize(group, 'line', 'prl-bsp-hover-proj-h', lines)
    .all
    .attr('x1', ctx.xAxis)
    .attr('x2', x)
    .attr('y1', y).attr('y2', y)
    .attr('stroke', color);

  cc.materialize(group, 'line', 'prl-bsp-hover-proj-v', lines)
    .all
    .attr('x1', x).attr('x2', x)
    .attr('y1', ctx.yAxis).attr('y2', y)
    .attr('stroke', color);

  // Hovered point

  cc.materialize(group, 'circle', 'prl-bsp-hover', lines)
    .all
    .attr('cx', x).attr('cy', y)
    .attr('r', pointsHoverRadius)
    .attr('stroke', color);
}

function renderHighlights(ctx, group) {
  const { highlights } = ctx;

  cc.materialize(group, 'g', 'prl-bsp-hover-group', highlights, hl => hl.id)
    .all
    .each(function (hl) {
      const highlight = d3.select(this);
      renderHighlight(ctx, highlight, hl);
    })
    .order();
}

function renderTooltip(ctx, group) {
  const hoverLine = ctx.view.hover.line;
  const lines = (hoverLine && hasValues(ctx)(hoverLine)) ? [hoverLine] : [];


  const x = cx(ctx);
  const y = cy(ctx);

  const body = d3.select('body');

  const svgRect = ctx.svg.getBoundingClientRect();
  const svgL = window.pageXOffset + svgRect.left;
  const svgT = window.pageYOffset + svgRect.top;
  const svgR = body.node().clientWidth - window.pageXOffset - svgRect.right;
  const svgB = body.node().clientHeight - window.pageYOffset - svgRect.bottom;

  const chartPageL = svgL - ctx.viewRect[0];
  const chartPageT = svgT - ctx.viewRect[1];
  const chartPageR = svgR + svgRect.width + ctx.viewRect[0];
  const chartPageB = svgB + svgRect.height + ctx.viewRect[1];


  cc.materialize(body, 'div', 'prl-bsp-tooltip', lines).all
    .each(function (line) {
      const toolTip = d3.select(this);

      const xPos = x(line);
      const yPos = y(line);

      const xRelPos = xToValue(ctx, xPos);
      const yRelPos = yToValue(ctx, yPos);

      const positions = {
        left: undefined, right: undefined,
        top: undefined, bottom: undefined
      };

      if (xRelPos < .5) {
        positions.left = `${chartPageL + xPos + tooltipPad}px`;
      } else {
        positions.right = `${chartPageR - xPos + tooltipPad}px`;
      }

      if (yRelPos < .5) {
        positions.bottom = `${chartPageB - yPos + tooltipPad}px`;
      } else {
        positions.top = `${chartPageT + yPos + tooltipPad}px`;
      }

      toolTip
        .style('left', positions.left)
        .style('top', positions.top)
        .style('right', positions.right)
        .style('bottom', positions.bottom)
        .html(tooltipHtml(ctx, line, !positions.top));
    });
}

function renderFilters(ctx, group) {
  const xc = ctx.scatterplotAxes.x;
  const yc = ctx.scatterplotAxes.y;

  const xCol = ctx.columns[xc];
  const yCol = ctx.columns[yc];

  const xFilter = ctx.view.colFilters[xCol.id];
  const yFilter = ctx.view.colFilters[yCol.id];

  const filters = (flt, type) => flt && flt.type === type ? [flt] : [];

  const xRangeFilters = filters(xFilter, 'range');
  const yRangeFilters = filters(yFilter, 'range');
  const xValueFilters = filters(xFilter, 'value');
  const yValueFilters = filters(yFilter, 'value');


  cc.materialize(group, 'g', 'prl-bsp-xfilter-range-group', xRangeFilters).all
    .each(ctx.renderXRangeFilter);

  cc.materialize(group, 'g', 'prl-bsp-yfilter-range-group', yRangeFilters).all
    .each(ctx.renderYRangeFilter);

  cc.materialize(group, 'g', 'prl-bsp-xfilter-value-group', xValueFilters).all
    .each(ctx.renderXValueFilter);

  cc.materialize(group, 'g', 'prl-bsp-yfilter-value-group', yValueFilters).all
    .each(ctx.renderYValueFilter);
}

function renderXRangeFilter(ctx, filter) {
  const group = d3.select(this);

  const c = ctx.scatterplotAxes.x;
  const column = ctx.columns[c];
  const colRange = ctx.colRanges[c];

  const x0 = valueToX(ctx, filter.v0);
  const x1 = valueToX(ctx, filter.v1);
  const y0 = ctx.yAxis + filterRangeSizeHalf;
  const y1 = y0 - filterRangeSize;
  const width = x1 - x0;
  const handlesSize = Math.min(width, filterRangeHandlesSize);
  const x0Text = x0 - filterTextPad;
  const x1Text = x1 + filterTextPad;
  const yText = ctx.yAxis + filterTextCrossPad;


  let sel = cc.materialize(group, 'rect', 'prl-bsp-filter-range', [0]);
  sel.enter
    .attr('height', filterRangeSize);
  sel.all
    .attr('x', x0).attr('width', width)
    .attr('y', y1)
    .call(d3.drag().container(ctx.svg)
      .on('start.prl-bsp', ctx.xRangeFilterDragStart) // eslint-disable-line memoryleaks
      .on('drag.prl-bsp', ctx.xRangeFilterDragMove) // eslint-disable-line memoryleaks
      .on('end.prl-bsp', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks

  sel = cc.materialize(group, 'rect', 'prl-bsp-filter-range-min', [0]);
  sel.enter
    .attr('height', filterRangeSize);
  sel.all
    .attr('x', x0).attr('width', handlesSize)
    .attr('y', y1)
    .call(d3.drag().container(ctx.svg)
      .on('start.prl-bsp', ctx.xRangeFilterDragStart) // eslint-disable-line memoryleaks
      .on('drag.prl-bsp', ctx.xRangeFilterMinDragMove) // eslint-disable-line memoryleaks
      .on('end.prl-bsp', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks

  sel = cc.materialize(group, 'rect', 'prl-bsp-filter-range-max', [0]);
  sel.enter
    .attr('height', filterRangeSize);
  sel.all
    .attr('x', Math.max(x1 - filterRangeHandlesSize, x0))
    .attr('width', handlesSize)
    .attr('y', y1)
    .call(d3.drag().container(ctx.svg)
      .on('start.prl-bsp', ctx.xRangeFilterDragStart) // eslint-disable-line memoryleaks
      .on('drag.prl-bsp', ctx.xRangeFilterMaxDragMove) // eslint-disable-line memoryleaks
      .on('end.prl-bsp', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks


  cc.materialize(group, 'line', 'prl-bsp-filter-line', [x0, x1])
    .all
    .attr('x1', _.identity).attr('x2', _.identity)
    .attr('y1', y1).attr('y2', ctx.chartRect[1]);

  cc.materialize(group, 'text', 'prl-bsp-filter-text-min', [0])
    .all
    .attr('x', x0Text).attr('y', yText)
    .attr('transform', `rotate(-90 ${x0Text},${yText})`)
    .text(cc.sampleToShortText(column, colRange, filter.s0));

  sel = cc.materialize(group, 'text', 'prl-bsp-filter-text-max', [0]);
  sel.enter
    .attr('dy', '1em');
  sel.all
    .attr('x', x1Text).attr('y', yText)
    .attr('transform', `rotate(-90 ${x1Text},${yText})`)
    .text(cc.sampleToShortText(column, colRange, filter.s1));
}

function renderYRangeFilter(ctx, filter) {
  const group = d3.select(this);

  const c = ctx.scatterplotAxes.y;
  const column = ctx.columns[c];
  const colRange = ctx.colRanges[c];

  const x0 = ctx.xAxis - filterRangeSizeHalf;
  const x1 = x0 + filterRangeSize;
  const y0 = valueToY(ctx, filter.v0);
  const y1 = valueToY(ctx, filter.v1);
  const height = y0 - y1;
  const handlesSize = Math.min(height, filterRangeHandlesSize);
  const y0Text = y0 + filterTextPad;
  const y1Text = y1 - filterTextPad;
  const xText = ctx.xAxis - filterTextCrossPad;


  let sel = cc.materialize(group, 'rect', 'prl-bsp-filter-range', [0]);
  sel.enter
    .attr('width', filterRangeSize);
  sel.all
    .attr('x', x0).attr('y', y1)
    .attr('height', height)
    .call(d3.drag().container(ctx.svg)
      .on('start.prl-bsp', ctx.yRangeFilterDragStart) // eslint-disable-line memoryleaks
      .on('drag.prl-bsp', ctx.yRangeFilterDragMove) // eslint-disable-line memoryleaks
      .on('end.prl-bsp', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks

  sel = cc.materialize(group, 'rect', 'prl-bsp-filter-range-min', [0]);
  sel.enter
    .attr('width', filterRangeSize);
  sel.all
    .attr('x', x0)
    .attr('y', Math.max(y0 - filterRangeHandlesSize, y1))
    .attr('height', handlesSize)
    .call(d3.drag().container(ctx.svg)
      .on('start.prl-bsp', ctx.yRangeFilterDragStart) // eslint-disable-line memoryleaks
      .on('drag.prl-bsp', ctx.yRangeFilterMinDragMove) // eslint-disable-line memoryleaks
      .on('end.prl-bsp', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks

  sel = cc.materialize(group, 'rect', 'prl-bsp-filter-range-max', [0]);
  sel.enter
    .attr('width', filterRangeSize);
  sel.all
    .attr('x', x0).attr('y', y1)
    .attr('height', handlesSize)
    .call(d3.drag().container(ctx.svg)
      .on('start.prl-bsp', ctx.yRangeFilterDragStart) // eslint-disable-line memoryleaks
      .on('drag.prl-bsp', ctx.yRangeFilterMaxDragMove) // eslint-disable-line memoryleaks
      .on('end.prl-bsp', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks


  cc.materialize(group, 'line', 'prl-bsp-filter-line', [y0, y1])
    .all
    .attr('x1', x1).attr('x2', ctx.chartRect[2])
    .attr('y1', _.identity).attr('y2', _.identity);

  sel = cc.materialize(group, 'text', 'prl-bsp-filter-text-min', [0]);
  sel.enter
    .attr('dy', '1em');
  sel.all
    .attr('x', xText).attr('y', y0Text)
    .text(cc.sampleToShortText(column, colRange, filter.s0));

  cc.materialize(group, 'text', 'prl-bsp-filter-text-max', [0])
    .all
    .attr('x', xText).attr('y', y1Text)
    .text(cc.sampleToShortText(column, colRange, filter.s1));
}

function renderXValueFilter(ctx, filter) {
  const group = d3.select(this);

  const c = ctx.scatterplotAxes.x;
  const colType = ctx.columns[c].type;
  const colRange = ctx.colRanges[c];

  const x = valueToX(ctx, filter.v0);
  const y0 = ctx.yAxis + filterRangeSizeHalf;
  const yText = ctx.yAxis + filterTextCrossPad;


  let sel = cc.materialize(group, 'circle', 'prl-bsp-filter-value', [0]);
  sel.enter
    .attr('r', filterValueRadius);
  sel.all
    .attr('cx', x).attr('cy', ctx.yAxis);


  cc.materialize(group, 'line', 'prl-bsp-filter-line', [0])
    .all
    .attr('x1', x).attr('x2', x)
    .attr('y1', y0).attr('y2', ctx.chartRect[1]);

  sel = cc.materialize(group, 'text', 'prl-bsp-filter-text', [0]);
  sel.enter
    .attr('dy', '.5em');
  sel.all
    .attr('x', x).attr('y', yText)
    .attr('transform', `rotate(-90 ${x},${yText})`)
    .text(cc.sampleToText(colType, colRange, filter.s1));
}

function renderYValueFilter(ctx, filter) {
  const group = d3.select(this);

  const c = ctx.scatterplotAxes.y;
  const colType = ctx.columns[c].type;
  const colRange = ctx.colRanges[c];

  const x0 = ctx.xAxis + filterRangeSizeHalf;
  const y = valueToY(ctx, filter.v0);
  const xText = ctx.xAxis - filterTextCrossPad;


  let sel = cc.materialize(group, 'circle', 'prl-bsp-filter-value', [0]);
  sel.enter
    .attr('r', filterValueRadius);
  sel.all
    .attr('cx', ctx.xAxis)
    .attr('cy', y);


  cc.materialize(group, 'line', 'prl-bsp-filter-line', [0])
    .all
    .attr('x1', x0).attr('x2', ctx.chartRect[2])
    .attr('y1', y).attr('y2', y);

  sel = cc.materialize(group, 'text', 'prl-bsp-filter-text', [0]);
  sel.enter
    .attr('dy', '.5em');
  sel.all
    .attr('x', xText).attr('y', y)
    .text(cc.sampleToText(colType, colRange, filter.s1));
}

function renderCloseButton(ctx, group) {
  let sel = cc.materialize(group, 'text', 'prl-bsp-btn-close', [0]);
  sel.enter
    .attr('y', -padTop + closePad)
    .attr('dy', '1em')
    .text('\uf066');
  sel.all
    .attr('x', ctx.refSizeX + padRight - closePad)
    .on('click.prl-bsp', ctx.onScatterplotClose); // eslint-disable-line memoryleaks
}

function renderOverlayGroup(ctx, svg) {
  const overlayGroup = cc.materialize(svg, 'g', 'prl-bsp-overlay-group', [0]).all;

  renderFilterBounds(ctx, overlayGroup);
  renderAxis(ctx, overlayGroup);
  renderLabels(ctx, overlayGroup);
  renderHighlights(ctx, overlayGroup);
  renderTooltip(ctx, overlayGroup);
  renderFilters(ctx, overlayGroup);
  renderCloseButton(ctx, overlayGroup);
}

function renderScatterPlot(ctx, renderContent = true, renderOverlay = true) {
  const svg = d3.select(ctx.svg);

  if (renderContent) { renderContentGroup(ctx, svg); }
  if (renderOverlay) { renderOverlayGroup(ctx, svg); }
}

function renderSvgAttributes(ctx) {
  d3.select(ctx.svg)
    .attr('width', ctx.sizeX)
    .attr('height', ctx.sizeY)
    .attr('viewBox', ctx.viewBox.join(' '));
}


// State

function buildTickLabelSizes(ctx, state) {
  state.xTickLabelsHeight =
    d3.select(ctx.svg).select('g.prl-bsp-xaxis-group').node()
      .getBoundingClientRect().height;

  state.yTickLabelsWidth =
    d3.select(ctx.svg).select('g.prl-bsp-yaxis-group').node()
      .getBoundingClientRect().width;

  buildSizes(ctx, state);
  buildScales(ctx);

  renderSvgAttributes(ctx);
  renderScatterPlot(ctx);
}

function buildTicks(ctx) {
  const { columns, scales, scatterplotAxes } = ctx;

  const xc = scatterplotAxes.x;
  const yc = scatterplotAxes.y;

  ctx.xTicks = cc.columnTicks(
    columns[xc], scales.columns[xc], ctx.chartSize[0], ctx.ticksDistance);
  ctx.yTicks = cc.columnTicks(
    columns[yc], scales.columns[yc], ctx.chartSize[1], ctx.ticksDistance);
}

function buildSizes(ctx, state) {
  const containerRect = ctx.container.getBoundingClientRect();

  let { pointsRadius } = ctx;
  pointsRadius = pointsRadius || 3;

  const axisPad = filterRangeSizeHalf + pointsRadius + 2;

  ctx.pointsRadius = pointsRadius;
  ctx.pointsHoverRadius = pointsRadius + 3;

  ctx.sizeX = containerRect.width;
  ctx.sizeY = containerRect.height;

  const spaceBeforeXAxis = padLeft + state.yTickLabelsWidth + axisLabelSize;
  const spaceBeforeYAxis = padBottom + state.xTickLabelsHeight + axisLabelSize;

  ctx.refSizeX = Math.max(ctx.sizeX - spaceBeforeXAxis - padRight, axisPad);
  ctx.refSizeY = Math.max(ctx.sizeY - spaceBeforeYAxis - padTop, axisPad);

  ctx.xAxis = .5;
  ctx.yAxis = Math.round(ctx.refSizeY) + .5;

  ctx.chartRect = [
    ctx.xAxis + axisPad, .5,
    Math.round(ctx.refSizeX) + .5, ctx.yAxis - axisPad
  ];
  ctx.chartSize = cc.rectToSize(ctx.chartRect);

  ctx.xMidChart = (ctx.chartRect[0] + ctx.chartRect[2]) / 2;
  ctx.yMidChart = (ctx.chartRect[1] + ctx.chartRect[3]) / 2;

  ctx.viewBox = [ -spaceBeforeXAxis, -padTop, ctx.sizeX, ctx.sizeY ];
  ctx.viewRect = cc.boxToRect(ctx.viewBox);

  ctx.contentRect = [
    ctx.viewRect[0] + padLeft, ctx.viewRect[1] + padTop,
    ctx.viewRect[2] - padRight, ctx.viewRect[3] - padBottom
  ];

  ctx.hoverRect = [
    ctx.chartRect[0] - pointsRadius, ctx.chartRect[1] - pointsRadius,
    ctx.chartRect[2] + pointsRadius, ctx.chartRect[3] + pointsRadius
  ];

  ctx.filterDefineRect = [
    ctx.xAxis - filterRangeSizeHalf, ctx.chartRect[1],
    ctx.chartRect[2], ctx.yAxis + filterRangeSizeHalf
  ];

  ctx.filterDefineBoxTL = [
    ctx.filterDefineRect[0], ctx.filterDefineRect[1],
    ctx.filterDefineRect[2] - ctx.filterDefineRect[0], ctx.chartSize[1]
  ];
  ctx.filterDefineBoxBR = [
    ctx.chartRect[0], ctx.filterDefineRect[1],
    ctx.chartSize[0], ctx.filterDefineRect[3] - ctx.filterDefineRect[1]
  ];
}

function buildScales(ctx) {
  const axes = ctx.scatterplotAxes;
  assert(axes);

  const vxScale = ctx.scales.columns[axes.x];
  const vyScale = ctx.scales.columns[axes.y];

  const x0 = ctx.chartRect[0];
  const y0 = ctx.chartRect[3];
  const [ xLen, yLen ] = ctx.chartSize;

  [ ctx.xScale, ctx.yScale ] = ctx.pixScales = [
    val => x0 + xLen * vxScale(val),
    val => y0 - yLen * vyScale(val)
  ];
}

function initRenderContext(base, ctx, state) {
  ctx = _.assign({
    viewChanges: {},
    filterDrag: null
  }, base, ctx);

  _.assign(ctx, {
    renderXRangeFilter: _.partial(renderXRangeFilter, ctx),
    renderYRangeFilter: _.partial(renderYRangeFilter, ctx),
    renderXValueFilter: _.partial(renderXValueFilter, ctx),
    renderYValueFilter: _.partial(renderYValueFilter, ctx),
  });

  buildSizes(ctx, state);
  buildScales(ctx);
  buildTicks(ctx);

  ctx.highlights = cc.buildHighlights(ctx);

  return ctx;
}


// Exports

export default class ScatterPlot {
  constructor(opts) {
    this.baseCtx = opts;
    this.ctx = null;

    this.xTickLabelsHeight = 0;
    this.yTickLabelsWidth = 0;
  }

  render(ctx) {
    this.ctx = ctx = initRenderContext(this.baseCtx, ctx, this);

    initMouse(ctx);

    renderSvgAttributes(ctx);
    renderScatterPlot(ctx, false, true);        // No content rendering here

    // Defer to let text nodes be rendered *before* querying their sizes
    _.defer(buildTickLabelSizes, ctx, this);    // First content rendering here
  }

  refresh(viewChanges) {
    if (!this.ctx) { return; }

    let refreshContent = false;
    let refreshOverlay = false;

    _.chain(viewChanges).keys().forEach(viewData => {
      switch (viewData) {
        case 'colFilters': refreshContent = true;
        case 'hover': refreshOverlay = true;
      }
    }).commit();

    renderScatterPlot(this.ctx, refreshContent, refreshOverlay);
  }

  reset() {
    if (!this.ctx) { return; }

    d3.select(this.ctx.svg)
      .on('.prl-bsp .drag', null) // eslint-disable-line memoryleaks
      .selectAll('*')
      .remove();

    d3.select('.prl-bsp-tooltip').remove();

    this.ctx = null;
  }
};

