
/*eslint import/namespace: 0*/
import * as cc from './chartcommons';
import { rangeFilter } from './renderdata';
import * as d3 from './d3';
import _ from 'lodash';

import '../styles/scatterstrip.less';


const padChartX0 = 12;
const padChartY0 = 10;
const padChartX1 = padChartX0;
const padChartY1 = 6;
const axisLabelPadX = 5;
const axisLabelPadY = 3;
const axisPad = 5;
const fontSize = 14;
const descender = .25 * fontSize;
const expandIconPad = axisLabelPadY + 1;
const labelsClipMarginL = 30;
const labelsClipMarginR = padChartX0 - axisLabelPadY - fontSize - 5;

const pointsRadius = 1.5;
const pointsHoverRadius = 3;
const pointsHaloRadius = 5;

const clickDistance = 5;
const clickDistance2 = clickDistance * clickDistance;

const PI2 = Math.PI * 2;


// Local Functions

// Geometry

function xToColumnIndex(ctx, x) {
  return Math.min(Math.max(
    Math.floor((ctx.columns.length - 1) * (x / ctx.refSizeX)),
    0), ctx.columns.length - 2);
}

function mouseToChartPos(ctx, c, mPos) {
  let x0 = c * ctx.xStep;

  return cc.movedInsideRect(ctx.chartRect, [ mPos[0] - x0, mPos[1] ]);
}

function chartPosToValues(ctx, pos) {
  return [
    (pos[0] - ctx.x0Chart) / (ctx.x1Chart - ctx.x0Chart),
    (pos[1] - ctx.y0Chart) / (ctx.y1Chart - ctx.y0Chart)
  ];
}

function hasValues(ctx, c) {
  const xc = ctx.xColIdx[c];
  const yc = ctx.yColIdx[c];

  return line => (
    (line[ctx.visibleToAll[xc]] !== null) &&
    (line[ctx.visibleToAll[yc]] !== null));
}

function cx(ctx, c) {
  const xc = ctx.xColIdx[c];
  const l = ctx.visibleToAll[xc];

  return line => ctx.xScales[xc](line[l]);
}

function cy(ctx, c) {
  const yc = ctx.yColIdx[c];
  const l = ctx.visibleToAll[yc];

  return line => ctx.yScales[yc](line[l]);
}

function closestLine(ctx, posX, posY) {
  const c = xToColumnIndex(ctx, posX);

  const x0 = c * ctx.xStep;
  const px = cx(ctx, c);
  const py = cy(ctx, c);
  const filter = hasValues(ctx, c);

  return cc.closestPoint(
    _(ctx.filtered.records[0]).filter(filter), px, py, posX - x0, posY);
}


// Interaction

function storeMouseDown(ctx) {
  const e = d3.event;

  ctx.mouseDownX = e.clientX;
  ctx.mouseDownY = e.clientY;
}

function updateHover(ctx) {
  const mPos = cc.rawMousePos(ctx, d3.event);
  if (!cc.rectContains(ctx.contentRect, mPos)) { return cc.resetHover(ctx); }

  const { hover } = ctx.view;

  const line = closestLine(ctx, mPos[0], mPos[1]);
  if (line === hover.line) { return; }

  hover.line = line;
  hover.event = d3.event;

  ctx.viewChanges.hover = true;

  cc.refresh(ctx);
}


function filterDefineFilter(ctx) {
  return !d3.event.button && cc.rectContains(ctx.contentRect, d3.mouse(ctx.svg));
}

function filterDefineReady(ctx) {
  if (ctx.filterDefineStarted) { return true; }

  const e = d3.event.sourceEvent;

  const dx = e.clientX - ctx.mouseDownX;
  const dy = e.clientY - ctx.mouseDownY;

  const ready = (dx * dx + dy * dy > clickDistance2);
  if (!ready) { return false; }

  cc.resetHover(ctx);

  return ctx.filterDefineStarted = true;
}

function filterDefineStart(ctx) {
  const mPos = [d3.event.x, d3.event.y];

  const c = xToColumnIndex(ctx, mPos[0]);

  const chartPos = mouseToChartPos(ctx, c, mPos);
  const values = chartPosToValues(ctx, chartPos);

  ctx.filterDrag = {
    c,
    vx: values[0], vy: values[1],
    x: chartPos[0], y: chartPos[1]
  };
}

function filterDefineMove(ctx) {
  if (!filterDefineReady(ctx)) { return; }

  const mPos = [d3.event.x, d3.event.y];

  const c = ctx.filterDrag.c;
  const cy = ctx.yColIdx[c];
  const cx = ctx.xColIdx[c];

  const chartPos = mouseToChartPos(ctx, c, mPos);
  const values = chartPosToValues(ctx, chartPos);

  const cxName = ctx.columns[cx].id;
  const cyName = ctx.columns[cy].id;

  const v0x = Math.min(values[0], ctx.filterDrag.vx);
  const v1x = Math.max(values[0], ctx.filterDrag.vx);

  const v0y = Math.min(values[1], ctx.filterDrag.vy);
  const v1y = Math.max(values[1], ctx.filterDrag.vy);

  ctx.view.colFilters[cxName] = rangeFilter(ctx, cxName, v0x, v1x);
  ctx.view.colFilters[cyName] = rangeFilter(ctx, cyName, v0y, v1y);

  ctx.filterRect = {
    x0: _.min([ctx.filterDrag.x, chartPos[0]]),
    x1: _.max([ctx.filterDrag.x, chartPos[0]]),
    y0: _.min([ctx.filterDrag.y, chartPos[1]]),
    y1: _.max([ctx.filterDrag.y, chartPos[1]])
  };

  ctx.viewChanges.colFilters = true;
  cc.refresh(ctx);
}

function filterDefineEnd(ctx) {
  ctx.filterDrag = ctx.filterRect = null;

  if (!ctx.filterDefineStarted) { return; }
  ctx.filterDefineStarted = false;

  cc.filterDefineEnd(ctx);
}

function scatterplotExpand(ctx, c) {
  d3.event.stopPropagation();
  ctx.onScatterplotExpand({ x: ctx.xColIdx[c], y: ctx.yColIdx[c] });
}


function initMouse(ctx) {
  ctx.filterDefineEnd = () => filterDefineEnd(ctx);
  ctx.scatterplotExpand = c => scatterplotExpand(ctx, c);

  ctx.mouseDownX = null;
  ctx.mouseDownY = null;
  ctx.filterDefineStarted = false;

  d3.select(ctx.svg)
    .on('mousedown.ssg', () => storeMouseDown(ctx)) // eslint-disable-line memoryleaks
    .on('mousemove.ssg', () => updateHover(ctx)) // eslint-disable-line memoryleaks
    .on('mouseleave.ssg', () => cc.resetHover(ctx)) // eslint-disable-line memoryleaks
    .call(d3.drag()
      .container(ctx.svg)
      .filter(() => filterDefineFilter(ctx))
      .on('start.ssg', () => filterDefineStart(ctx)) // eslint-disable-line memoryleaks
      .on('drag.ssg', () => filterDefineMove(ctx)) // eslint-disable-line memoryleaks
      .on('end.ssg', ctx.filterDefineEnd)); // eslint-disable-line memoryleaks
}


// Rendering

function cleanUnusedSurface(ctx, state, svg) {
  const { useCanvas } = ctx;
  const { didUseCanvas } = state;

  if (didUseCanvas !== null && useCanvas !== didUseCanvas) {
    if (didUseCanvas) {
      ctx.canvas.getContext('2d').clearRect(0, 0, ctx.sizeX, ctx.sizeY);
    } else {
      svg.selectAll('ssg-content > *').remove();
    }
  }

  state.didUseCanvas = useCanvas;
}

function renderChartsStrip(ctx, state, renderContent = true, renderOverlay = true) {
  const svg = d3.select(ctx.svg);
  const canvas = ctx.canvas.getContext('2d');

  const { columns } = ctx;
  const colIndexes = _.range(0, columns.length - 1, 1);

  cc.materialize(svg, 'g', 'ssg-chart', colIndexes)
    .all
    .attr('transform', cc.translator(ctx.refSizeX, columns.length))
    .each(function (c) {
      let chart = d3.select(this);

      if (renderContent) { renderContentGroupSvg(ctx, chart, c); }
      if (renderOverlay) { renderOverlayGroup(ctx, chart, c); }
    });

  if (ctx.useCanvas && renderContent) {
    canvas.setTransform(1, 0, 0, 1, 0, 0);
    canvas.clearRect(0, 0, ctx.sizeX, ctx.sizeY);

    _.forEach(colIndexes, function (c) {
      renderContentGroupCanvas(ctx, canvas, c);
    });
  }

  cleanUnusedSurface(ctx, state, svg);
}

function renderContentGroupSvg(ctx, group, c) {
  const contentGroup = cc.materialize(group, 'g', 'ssg-content', [0]).all;

  if (ctx.useCanvas) { return contentGroup.selectAll('*').remove(); }

  renderAxis(ctx, contentGroup, c);
  renderPointsSvg(ctx, contentGroup, c);
}

function renderAxis(ctx, group, c) {
  const axisGroup = cc.materialize(group, 'g', 'ssg-chart-axis', [0]).all;

  let sel = cc.materialize(axisGroup, 'line', 'ssg-axis-x', [0]);
  sel.enter
    .attr('x1', ctx.xAxis);
  sel.all
    .attr('x2', ctx.x1Chart)
    .attr('y1', ctx.yAxis).attr('y2', ctx.yAxis);

  sel = cc.materialize(axisGroup, 'line', 'ssg-axis-y', [0]);
  sel.enter
    .attr('x1', ctx.xAxis).attr('x2', ctx.xAxis)
    .attr('y2', ctx.y1Chart);
  sel.all
    .attr('y1', ctx.yAxis);
}

function renderPointsSvg(ctx, group, c) {
  const filteredLines = ctx.filtered.records;
  const filteredOutLines = ctx.showFilteredOutRecords ? filteredLines[1] : [];

  const x = cx(ctx, c);
  const y = cy(ctx, c);
  const filter = hasValues(ctx, c);


  const outGroup = cc.materialize(group, 'g', 'ssg-filtered-out', [0]).all;

  let sel = cc.materialize(outGroup, 'circle', '', filteredOutLines.filter(filter));
  sel.enter
    .attr('r', pointsRadius);
  sel.all
    .attr('cx', x)
    .attr('cy', y);


  const inGroup = cc.materialize(group, 'g', 'ssg-filtered-in', [0]).all;

  sel = cc.materialize(inGroup, 'circle', '', filteredLines[0].filter(filter));
  sel.enter
    .attr('r', pointsRadius);
  sel.all
    .attr('cx', x)
    .attr('cy', y)
    .attr('fill', ctx.colors.record);
}

function renderContentGroupCanvas(ctx, canvas, c) {
  const step = ctx.xStep;
  canvas.setTransform(1, 0, 0, 1,
    Math.round(c * step - ctx.viewBox[0]),
    Math.round(-ctx.viewBox[1]));

  renderAxisCanvas(ctx, canvas, c);
  renderPointsCanvas(ctx, canvas, c);
}

function renderAxisCanvas(ctx, canvas, c) {
  canvas.strokeStyle = '#bbb';
  canvas.beginPath();

  canvas.moveTo(ctx.xAxis, ctx.yAxis);
  canvas.lineTo(ctx.x1Chart, ctx.yAxis);

  canvas.moveTo(ctx.xAxis, ctx.yAxis);
  canvas.lineTo(ctx.xAxis, ctx.y1Chart);

  canvas.stroke();
}

function renderPointsCanvas(ctx, canvas, c) {
  const x = cx(ctx, c);
  const y = cy(ctx, c);
  const filter = hasValues(ctx, c);

  const filteredLines = ctx.filtered.records;

  if (ctx.showFilteredOutRecords) {
    canvas.fillStyle = 'rgb(230,230,230)';

    _.chain(filteredLines[1])
      .filter(filter)
      .forEach(function renderPoint(line) {
        canvas.beginPath();
        canvas.arc(x(line), y(line), pointsRadius, 0, PI2);
        canvas.fill();
      })
      .commit();
  }

  _.chain(filteredLines[0])
    .filter(filter)
    .forEach(function renderPoint(line) {
      canvas.fillStyle = ctx.colors.record(line);
      canvas.beginPath();
      canvas.arc(x(line), y(line), pointsRadius, 0, PI2);
      canvas.fill();
    })
    .commit();
}


function renderOverlayGroup(ctx, group, c) {
  const overlayGroup = cc.materialize(group, 'g', 'ssg-overlay', [0]).all;

  renderExpand(ctx, overlayGroup, c);
  renderLabels(ctx, overlayGroup, c);
  renderFilter(ctx, overlayGroup, c);
  renderHighlights(ctx, overlayGroup, c);
}

function renderExpand(ctx, group, c) {
  let sel = cc.materialize(group, 'rect', 'ssg-overlay-bounds', [0]);
  sel.enter
    .attr('x', 0).attr('y', 0);
  sel.all
    .attr('width', ctx.xStep).attr('height', ctx.refSizeY);

  sel = cc.materialize(group, 'text', 'ssg-expand-icon', [c]);
  sel.enter
    .attr('dy', '1em')
    .text('\uf065');
  sel.all
    .attr('x', ctx.xAxis - expandIconPad)
    .attr('y', ctx.yAxis + expandIconPad)
    .on('mousedown.ssg', ctx.scatterplotExpand); // eslint-disable-line memoryleaks
}

function renderFilter(ctx, group, c) {
  const filterRect = ctx.filterRect;
  const filterDrag = ctx.filterDrag;

  const rect = cc.materialize(group, 'rect', 'ssg-filter', [0]).all;
  if (!filterRect || !filterDrag || c !== filterDrag.c) {
    return rect.remove();
  }

  rect.attr('x', filterRect.x0).attr('width', filterRect.x1 - filterRect.x0)
    .attr('y', filterRect.y0).attr('height', filterRect.y1 - filterRect.y0);
}

function renderLabels(ctx, group, c) {
  const clipId = `ssg-axis-bounds-${ctx.columnUUIDs[c]}`;
  const clipPath = cc.materialize(group, 'clipPath', null, [0])
    .all
    .attr('id', clipId);

  let sel = cc.materialize(clipPath, 'rect', null, [0]);
  sel.enter
    .attr('x', -labelsClipMarginL).attr('y', 0);
  sel.all
    .attr('width', labelsClipMarginL + ctx.xStep + labelsClipMarginR)
    .attr('height', ctx.refSizeY);

  const labelsGroup = cc.materialize(group, 'g', 'ssg-labels-group', [0])
    .all
    .attr('clip-path', `url(#${clipId})`);

  const xlabel = ctx.columns[ctx.xColIdx[c]].label;

  sel = cc.materialize(labelsGroup, 'text', 'ssg-axis-x-label', [0]);
  sel.enter
    .attr('x', ctx.xAxis + axisLabelPadX)
    .attr('font-size', fontSize);
  sel.all
    .attr('y', ctx.yAxis + axisLabelPadY + fontSize)
    .text(xlabel);

  const ylabel = ctx.columns[ctx.yColIdx[c]].label;

  sel = cc.materialize(labelsGroup, 'text', 'ssg-axis-y-label', [0]);
  sel.enter
    .attr('y', ctx.xAxis - axisLabelPadY - descender)
    .attr('font-size', fontSize)
    .attr('transform', 'rotate(-90 0,0)');
  sel.all
    .attr('x', -ctx.yAxis + axisLabelPadX)
    .text(ylabel);
}

function renderHighlight(ctx, group, highlight, c) {
  const { lines, colors } = highlight;

  const color = colors.highlight || 'black';

  const x = cx(ctx, c);
  const y = cy(ctx, c);
  const filter = hasValues(ctx, c);

  let sel = cc.materialize(group, 'circle', 'ssg-hover-point', lines);
  sel.enter
    .attr('r', pointsHoverRadius)
    .attr('fill', color);
  sel.all
    .attr('cx', x)
    .attr('cy', y);

  sel = cc.materialize(group, 'circle', 'ssg-hover-halo', lines);
  sel.enter
    .attr('r', pointsHaloRadius)
    .attr('stroke', color);
  sel.all
    .attr('cx', x)
    .attr('cy', y);
}

function renderHighlights(ctx, group, c) {
  const { highlights } = ctx;

  cc.materialize(group, 'g', 'ssg-hover-group', highlights, hl => hl.id)
    .all
    .each(function (hl) {
      const highlight = d3.select(this);
      renderHighlight(ctx, highlight, hl, c);
    })
    .order();
}


// State

function buildSizes(ctx) {
  const padding = {
    left: cc.padLeft, right: cc.padRight,
    top: 0, bottom: 0
  };

  const result = cc.buildStripSizes(ctx, padding);

  result.xAxis = padChartX0 + .5;
  result.yAxis = Math.round(result.refSizeY - padChartY0 - fontSize) + .5;

  result.x0Chart = result.xAxis + axisPad;
  result.x1Chart = Math.round(result.xStep - padChartX1) + .5;

  result.y0Chart = result.yAxis - axisPad;
  result.y1Chart = padChartY1 + .5;

  result.chartRect = [
    result.x0Chart, result.y1Chart, result.x1Chart, result.y0Chart
  ];

  _.assign(ctx, result);
}

function buildScales(ctx) {
  const x0 = ctx.x0Chart;
  const y0 = ctx.y0Chart;
  const xLen = ctx.x1Chart - ctx.x0Chart;
  const yLen = ctx.y1Chart - ctx.y0Chart;

  ctx.xScales = _.map(ctx.scales.columns, scale => val => x0 + xLen * scale(val));
  ctx.yScales = _.map(ctx.scales.columns, scale => val => y0 + yLen * scale(val));
}

function buildColumnIdentifiers(ctx) {
  const { columns, forcedOrdinatesColumnId } = ctx;

  const columnIds = _.map(columns, 'id');
  _.assign(ctx, forcedOrdinatesColumnId ? {
    xColumnIds: _.without(columnIds, forcedOrdinatesColumnId),
    yColumnIds: _.times(columnIds.length - 1, _.constant(forcedOrdinatesColumnId))
  } : {
    xColumnIds: columnIds.slice(1, columnIds.length),
    yColumnIds: columnIds.slice(0, columnIds.length - 1)
  });

  const columnIdxsById = _(0).range(columns.length)
    .indexBy(idx => columns[idx].id)
    .value();

  function idToIdx(id) { return columnIdxsById[id]; }

  ctx.xColIdx = _.map(ctx.xColumnIds, idToIdx);
  ctx.yColIdx = _.map(ctx.yColumnIds, idToIdx);
}

function initRenderContext(base, ctx) {
  ctx = _.assign({
    viewChanges: {},
    filterDrag: null,
    filterRect: null,
    columnUUIDs: _.times(ctx.columns.length, () => _.random(1e12)),
  }, base, ctx);

  buildSizes(ctx);
  buildScales(ctx);
  buildColumnIdentifiers(ctx);

  ctx.highlights = cc.buildHighlights(ctx);

  return ctx;
}


// Exports

export default class ScatterStrip {
  constructor(opts) {
    this.baseCtx = opts;
    this.ctx = null;

    this.didUseCanvas = null;
  }

  render(ctx) {
    ctx = initRenderContext(this.baseCtx, ctx);
    if (ctx.columns.length < 2) { return this.reset(); }

    this.ctx = ctx;

    d3.select(ctx.svg)
      .attr('width', ctx.sizeX)
      .attr('height', ctx.sizeY)
      .attr('viewBox', ctx.viewBox.join(' '));

    ctx.canvas.width = ctx.sizeX;
    ctx.canvas.height = ctx.sizeY;
    ctx.canvas.style.width = ctx.sizeX + 'px';
    ctx.canvas.style.height = ctx.sizeY + 'px';

    renderChartsStrip(ctx, this);

    initMouse(ctx);
  }

  refresh(viewChanges) {
    const { ctx } = this;
    if (!ctx) { return; }

    let refreshContent = false;
    let refreshOverlay = false;

    _.chain(viewChanges).keys().forEach(viewData => {
      switch (viewData) {
        case 'colFilters': refreshContent = true;
        case 'hover': refreshOverlay = true;
      }
    }).commit();

    renderChartsStrip(ctx, this, refreshContent, refreshOverlay);
  }

  reset() {
    const { ctx } = this;
    if (!ctx) { return; }

    if (this.didUseCanvas) {
      ctx.canvas.getContext('2d').clearRect(0, 0, ctx.sizeX, ctx.sizeY);
    }

    d3.select(ctx.svg)
      .on('.ssg .drag', null) // eslint-disable-line memoryleaks
      .selectAll('*')
      .remove();

    this.ctx = null;
  }
};

