
import ParallelChart from './js/parallel_chart';
import ScatterStrip from './js/scatterstrip';
import ScatterPlot from './js/scatterplot';

import * as esAdapter from './js/es_adapter';
import RenderData from './js/renderdata';
import * as cc from './js/chartcommons';

import './parallel_vis.less';
import './styles/layout.less';
import './styles/utils.less';

import { uiModules } from 'ui/modules';

import { format } from 'd3';
import _ from 'lodash';


const module = uiModules.get('kibana/parallel_lines_chart_vis', ['kibana']);


function uiConsistentAggs(vis) {
  // A Vis's aggregations list can be out of order with respect to the way it's
  // shown in the UI - metrics first, then buckets

  const { aggs } = vis;
  return (aggs.bySchemaGroup.metrics || []).concat(aggs.bySchemaGroup.buckets || []);
}

function aggIds(aggs) {
  return _(aggs).map('id').join(' ');
}

function refreshViewMode($scope) {
  if (!$scope.renderData.records.length) {
    $scope.viewMode = 'error';
    $scope.error = 'No results found';
  } else if ($scope.renderData.columns.length < 2) {
    $scope.viewMode = 'error';
    $scope.error = 'Required two metrics/aggregations as columns';
  } else if ($scope.scatterplotAxes) {
    $scope.viewMode = 'bigsp';
  } else {
    $scope.viewMode = 'main';
  }
}

function render($scope) {
  let mainChartsWidth;
  let chartSizes;
  let toReset;

  const { renderData } = $scope;

  switch ($scope.viewMode) {
    case 'main':
      mainChartsWidth = $scope.refs.chartsArea.getBoundingClientRect().width;

      chartSizes = {
        sizeX: mainChartsWidth,
        minColDistance: Math.max(mainChartsWidth / 7, 150),
        minRefSizeX: Math.max(4 * mainChartsWidth / 7, 400)
      };

      $scope.parallelChart.render(_.assign({
        sizeY: $scope.refs.parallelChart.getBoundingClientRect().height
      }, chartSizes, renderData));

      $scope.scatterStrip.render(_.assign({
        sizeY: $scope.refs.scatterStrip.getBoundingClientRect().height
      }, chartSizes, renderData));

      toReset = ['scatterPlot'];

      break;

    case 'bigsp':
      $scope.scatterPlot.render(_.assign({
        scatterplotAxes: $scope.scatterplotAxes,
        pointsRadius: renderData.bigSpPointsRadius
      }, renderData));

      // Keeping main charts
      toReset = [];

      break;

    default:
      toReset = ['parallelChart', 'scatterStrip', 'scatterPlot'];
  }

  _($scope).pick(toReset).forEach(chart => chart.reset());

  return $scope;
}

function resetCharts($scope) {
  _($scope).pick(['parallelChart', 'scatterStrip', 'scatterPlot'])
    .forEach(chart => chart.reset());
}

function refreshEditableColumns($scope, chartColumns) {
  const { editableVis, editableParams } = $scope;
  if (!editableVis) { return; }

  const currentAggs = uiConsistentAggs($scope.vis);
  const editableAggs = uiConsistentAggs($scope.editableVis);

  const keepColumnsOrder = (aggIds(currentAggs) === aggIds(editableAggs));

  editableParams.columns = esAdapter.columnsFromAggs(
    editableAggs, editableParams.columns, chartColumns,
    { sortByChart: keepColumnsOrder });
}

function restoreColumnsVisibility() {
  this.renderData.all.columns.forEach(col => col.show = true);
  this.renderData.applyVisibility();

  refreshEditableColumns(this, this.renderData.all.columns);

  render(this);
}

function onViewChanges(viewChanges) {
  if (viewChanges.colFilters) {
    this.renderData.applyColumnFilters();
  }

  switch (this.viewMode) {
    case 'main':
      this.parallelChart.refresh(viewChanges);
      this.scatterStrip.refresh(viewChanges);
      break;

    case 'bigsp':
      this.scatterPlot.refresh(viewChanges);
      break;
  }
}

function onMoveColumn(srcColIdx, dstColIdx) {
  this.renderData.moveColumn(srcColIdx, dstColIdx);

  refreshEditableColumns(this, this.renderData.all.columns);
  render(this);
}

function onSortColumn(colIdx) {
  this.renderData.sortColumn(colIdx);

  refreshEditableColumns(this, this.renderData.all.columns);
  render(this);
}

function onHideColumn(colIdx) {
  if (this.renderData.columns.length <= 2) { return; }

  this.renderData.toggleColumnVisible(colIdx);

  refreshEditableColumns(this, this.renderData.all.columns);
  render(this);
}

function makeCount(records) {
  return format(',')(_.sum(records, line => line[line.length - 1]));
}

function makeMean(column, colRange, colIdx, records) {
  if (!records.length) { return; }

  const colType = column.type;
  if (colType !== 'number' && colType !== 'integer' && colType !== 'date') {
    return;
  }

  const weightFn = column.aggType === 'count'
    ? _.constant(1) : line => line[line.length - 1];

  // Rolling, numerically stable weighted mean
  const numMean = _.reduce(records, function (memo, line) {
    const weight = weightFn(line);

    memo.totWeight += weight;
    memo.mean += weight * (line[colIdx] - memo.mean) / memo.totWeight;

    return memo;
  }, { totWeight: 0, mean: 0 }).mean;

  return colType === 'date'
    ? cc.sampleToText(colType, colRange, new Date(numMean))
    : cc.sampleToText('number', colRange, numMean);
}

function makeSum(column, colIdx, records) {
  if (column.aggType === 'sum') {
    return format(',')(_.round(_.sum(records, colIdx), 2));
  }
}

function tipTableHtml(tableName, rows) {
  const rowData = _(rows)
  .filter(row => row[1] !== undefined)
  .map(row => `<tr> <td><b>${row[0]}&ensp;</b></td> <td>${row[1]}</td> </tr>`)
  .join('');

  return `
<div class="parallel-column-tip-section">
  <b>${tableName}</b>
  <table><tbody>${rowData}</tbody></table>
</div>`;
}

function onColumnTipHtmlRequest(colIdx) {
  const { columns, records, filtered, colRanges, view } = this.renderData;

  const column = columns[colIdx];
  const colRange = colRanges[colIdx];
  const filteredRecords = filtered.records[0];
  const haveFilters = !_.isEmpty(view.colFilters);

  const allTable = tipTableHtml('All Records', [
    [ 'Count', makeCount(records) ],
    [ 'Mean', makeMean(column, colRange, colIdx, records) ],
    [ 'Sum', makeSum(column, colIdx, records) ]
  ]);
  const filteredTable = !haveFilters ? '' :
    tipTableHtml('Filtered Records', [
      [ 'Count', makeCount(filteredRecords) ],
      [ 'Mean', makeMean(column, colRange, colIdx, filteredRecords) ],
      [ 'Sum', makeSum(column, colIdx, filteredRecords) ]
    ]);

  return allTable + filteredTable;
}

function onScatterplotExpand(axes) {
  this.scatterplotAxes = axes;
  refreshViewMode(this);                                // size watching invokes render
}

function onScatterplotClose() {
  this.scatterplotAxes = null;
  refreshViewMode(this);
}


module.controller('ParallelLinesChartVisController', function (
  Private, $scope, $element
) {
  const { vis } = $scope;

  const element = $element[0];
  const editableVis = vis.getEditableVis();

  Object.defineProperty($scope, 'editableParams', {
    get() { return (editableVis || vis).params; }
  });


  // Initialize state

  _.assign($scope, {
    editableVis,
    renderData: new RenderData(),

    refs: {
      vis: element,
      scrollable: element.getElementsByClassName('parallel-scrollable')[0],
      chartsArea: element.getElementsByClassName('parallel-charts-area')[0],
      parallelChart: element.getElementsByClassName('parallel-chart')[0],
      scatterStrip: element.getElementsByClassName('parallel-scatter-strip')[0],
      scatterplot: element.getElementsByClassName('parallel-big-scatterplot')[0],

      parallelChartSvg:
        element.getElementsByClassName('parallel-chart-svg')[0],
      parallelChartCanvas:
        element.getElementsByClassName('parallel-chart-canvas')[0],

      scatterStripSvg:
        element.getElementsByClassName('parallel-scatter-strip-svg')[0],
      scatterStripCanvas:
        element.getElementsByClassName('parallel-scatter-strip-canvas')[0],

      scatterplotSvg:
        element.getElementsByClassName('parallel-big-scatterplot-svg')[0],
    },

    viewMode: 'nodata',
    error: '',
    scatterplotAxes: null,

    restoreColumnsVisibility
  });

  const { refs } = $scope;

  const chartCallbacks = {
    onViewChanges: onViewChanges.bind($scope),
    onMoveColumn: onMoveColumn.bind($scope),
    onSortColumn: onSortColumn.bind($scope),
    onHideColumn: onHideColumn.bind($scope),
    onColumnTipHtmlRequest: onColumnTipHtmlRequest.bind($scope),
    onScatterplotExpand: onScatterplotExpand.bind($scope),
    onScatterplotClose: onScatterplotClose.bind($scope)
  };

  _.assign($scope, {
    parallelChart: new ParallelChart(_.assign({
      svg: refs.parallelChartSvg,
      canvas: refs.parallelChartCanvas
    }, chartCallbacks)),

    scatterStrip: new ScatterStrip(_.assign({
      svg: refs.scatterStripSvg,
      canvas: refs.scatterStripCanvas
    }, chartCallbacks)),

    scatterPlot: new ScatterPlot(_.assign({
      container: refs.scatterplot,
      svg: refs.scatterplotSvg
    }, chartCallbacks))
  });


  // Watches

  $scope.$watchCollection(function chartModeAndSizes(scope) {
    return [
      scope.viewMode,
      scope.refs.vis.clientWidth,
      scope.refs.vis.clientHeight,
      scope.refs.chartsArea.clientWidth,
      scope.refs.chartsArea.clientHeight
    ];
  }, _.debounce(function onResize() {
    render($scope);
  }, 50));

  if (editableVis) {
    $scope.$watch(
      scope => editableVis.getEnabledState().aggs,
      _.debounce(function () {
        refreshEditableColumns($scope, editableVis.params.columns);
      }, 50), true);
  }

  $scope.$on('$destroy', function onDestroy() {
    resetCharts($scope);                    // Removes eventual tooltips from body
  });

  $scope.$watch('esResponse', function buildRecordsAndRedraw(resp) {
    const { vis, editableParams } = $scope;
    const { aggs } = vis;

    const columns = esAdapter.columnsFromAggs(
      aggs, editableParams.columns, editableParams.columns);
    const records = esAdapter.recordsFromResponse(columns, aggs, resp);

    $scope.renderData = new RenderData(
      _.assign({}, editableParams, { columns, records }));

    refreshViewMode($scope);
    render($scope);
  });
});

module.controller('ParallelLinesChartVisColumnsController', function (
  Private, $scope, $element
) {
  $scope.hasColumnParameters = _([
    'number', 'integer', 'number_range', 'integer_range'
  ])
  .indexBy()
  .mapValues(_.constant(true))
  .value();
});
